
#include <iostream.h>

#include "mod.h"
#include "modm.h"
#include "jjassert.h"


umod_t
order(mod x)
{
    return  order_mod(x.x ,mod::modulus, mod::phifact);
}
//================= end ================


mod
div(const mod& m1, const mod& m2)
{
    mod t = inv(m2);
    mul(m1,t,t);
    return t;
}
// ============= end ===========


// tuning parameter:
#define  INV_USE_EGCD  1  // 0 (pow) or 1 (egcd)
// " egcd only works for modulus <=63bit ! "

#if  ( INV_USE_EGCD==1 )
#warning "FYI: inversion uses gcd (vs. powering)"
#else
#warning "FYI: inversion uses powering (vs. gcd)"
#endif

void
inv_fail(const mod &z, smod_t d)
{
    cerr << "tried to invert " << z << " mod " << mod::modulus << endl;
    cerr << "but gcd(" << z << ", " << mod::modulus << ") = " << d << " != 1 " << endl;
    jjassert2(0, "inv() failed" );    
}
// ============= end ===========


mod
inv(const mod& z)
{
    smod_t m = (mod::modulus);
    smod_t x = z.x;

#if  ( INV_USE_EGCD==1 )

#if  ( USE_64BIT_MODULUS )
#error " computation of mod inverse by egcd() doesn't work for moduli >63 bits "
#endif  // USE_64BIT_MODULUS

    smod_t d;
    smod_t u,v;
    d = egcd(m,x,u,v);      // d==m*u+x*v
    if ( d!=1 )  inv_fail(z,d);
    
    if ( v<0 )  v += m;

    mod i = mod(v);

#else //  INV_USE_EGCD

    umod_t d = gcd(x,m);
    if ( d!=1 )  inv_fail(z,d);
    
    mod i = pow(z,mod::phi-1);  // z^-1 = z^(phi(m)-1), cf. cohen p.19

#endif // NV_USE_EGCD

    if ( z*i != (mod::one) )
    {
        cerr << "tried to invert " << z << " mod " << m << endl;
        cerr << "got " << i << " as inverse" << endl;
	jjassert2( 0, "inv() failed" );
    }

    return i;
}
// ============= end ===========


mod
invpow(const mod& a, umod_t ex)
{
    if ( ex==0 )  
    {
        return mod::one;
    }
    else
    {
        mod z = inv(a);
        z = pow(z,ex);
        return z;
    }
}
// ============= end ===========



mod
pow(const mod& a, umod_t ex)
{
//    cout << "\n " << __FUNCTION__ 
//         << "(" << a << ", " << ex << ")\n";

    mod y = mod::one;
    if ( ex )
    {
        mod z(a);
        while ( 1 )
        {
            if ( ex&1 )  y *= z;
            
            ex /= 2;

            if ( ex==0 )  break;

            z *= z;
        }
    }

    return y;
}
// ============= end ===========



mod
root(umod_t r)
//
// find root of unity of order r
//
{
    umod_t mx = mod::maxorder;

    if ( r==mx )
    {
        return mod::maxordelem;
    }
    else
    {
        umod_t mxr = mx/r;

        if ( (mxr*r)!=mx )  // r must be a divider of maxorder
        {
            cerr << " request for root of impossible order " << r << endl;
            jjassert( 0 );
        }

        mod  h = pow(mod::maxordelem,mxr);
//        jjassert( r==order(h) );
        return h;
    }
}
// ============= end ===========


mod
invroot(umod_t r)
//
// find inverse root of unity of order r
//
{
    umod_t mx = mod::maxorder;

    if ( r==mx )
    {
        return mod::invmaxordelem;
    }
    else
    {
        umod_t mxr = mx/r;

        if ( (mxr*r)!=mx )  // r must be a divider of maxorder
        {
            cerr << " request for invroot of impossible order " << r << endl;
            jjassert( 0 );
        }

        mod  h = pow(mod::invmaxordelem,mxr);
//        jjassert( r==order(h) );
        return h;
    }
}
// ============= end ===========
