/**********************************************************
*
* Computation of pi by atan formulas
*
************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

// include picheck.h if you want checking of the result
// #include "picheck.h"


/***********************************************************
             Program Parameters:
************************************************************/

/* Calling sequence:
*  piatan  [no of decimals to compute [which atan-formula]]
*
* Examples:
*
*  piatan             : 1001 decimals by formula 0 (Machin)
*  piatan 100000      : 100001 decimals by formula 0
*  piatan 10 5        : 11 decimals by formula 5
*/

/* default no of decimals */
int m = 1001;

/*************************************************************
*
* atan-formulas for pi/4:
* pi/4 = k1*atan(1/a1) + k2*atan(1/a2) + ... +k4*atan(1/a4)
* and:
* k*arctan(1/a) = Sum( (-1)^i * k/((2i+1)a^(2i+1)) )
*
* For such formulas see
* Joerg Arndt, Remarks on arithmetical algorithms,
*
*************************************************************/

struct {
   short k, a;
} AtanTab[][4] =
   {{{ 16,  5}, {-4,239}}               /* 0: Machin    */
   ,{{  4,  2}, { 4,  3}}               /* 1: Euler     */
   ,{{ 24,  8}, { 8, 57}, {  4, 239}}   /* 2: Stoermer   */
   ,{{ 48, 18}, {32, 57}, {-20, 239}}   /* 3: Gauss      */
   ,{{ 48, 18}, {32, 99}, { 12, 239}, { 32, 307}} /* 4: NN */
   ,{{ 80, 57}, {96, 68}, { 48, 117}, {-20, 239}} /* 5: NN */
   };

/*********************************************************/

/*
* A WORD must be signed and and able to hold 2f-1 ...-f+1
* A DWORD must be signed and able to hold f * amax * amax
*/
typedef short WORD;
typedef long  DWORD;
int    k = 4;        /* no of decimals per WORD          */
WORD   f = 10000;    /* f must be a power of 10          */

typedef WORD *Big;   /* typedef of a big fraction:       */
                     /* A Big is an array of n WORD'S    */
Big Result;           /* Big pi                           */
Big S;               /* Big S = k/a^(2i+1)               */

int  n;              /* Size of the Big's in words       */
int  tol = 1;        /* no of accuracy words             */
int  imeat;          /* index of 1st non-zero word in S  */


void PrintTheDecimals(void)
{ /* prints the decimals of the resulting pi   */
   int  j;

   printf("\npi=");
   for (j=0; j <= m/k; ++j) {
      /* The very first word contains only 1 decimal     */
      /* All words after the first contain k decimals    */
      printf(j ? "%0*d " : "%*d", j ? k : 1, Result[j]);
   }
   return;
}

void DivideAndStore(DWORD denom)
{ /* Divide the big fraction S by denom
  *  and store the result back into S */
   DWORD acc, rem = 0;
   WORD  quot;
   int   i;
   /* Divide from left to right
   *  starting at the first non-zero word  */
   for(i=imeat; i < n; ++i) {
      acc   = f * rem + S[i];
      quot  = (WORD)(acc / denom);
      rem   = acc - quot * denom;
      S[i]  = quot;
   }
   /* Adjust imeat, the index to the first non-zero word */
   if (S[imeat] == 0)
      ++imeat;
}

void DivideAndAdd(DWORD denom)
{ /* Divide big fraction S by denom and
  *  and add the result to the big fraction Result */
   DWORD acc, rem = 0;
   WORD sum, quot, carry = 0;
   int  i;
   for(i = imeat; i < n; ++i) {
      acc        = f * rem + S[i];
      quot       = (WORD)(acc / denom);
      rem        = acc - quot * denom;
      Result[i] += quot;
   }
   /* Normalize the words to 0 .. f-1 */
   for(i=n; --i >= imeat || carry != 0; ) {
      sum = (WORD)(Result[i] + carry);
      if (sum < 0)
         carry = -1, Result[i] = (WORD)(sum + f);
      else
         if (sum >= f)
            carry = +1, Result[i] = (WORD)(sum - f);
         else
            carry = 0, Result[i] = sum;
}  }

int main(int argc, char *argv[])
{ /* main: */
   DWORD a, aexp2;
   DWORD denom, TwoiP1;
   int   i, j;
   clock_t t0, t1;

   /* Command line arguments:
   *  no_of_decimals_to_compute    which_arctan-formula
   */
   if (argc > 1)              /* Get 1st param = no of decimals */
      m = atoi(argv[1]);
   m = ((m+k-2)/k) * k + 1;   /* no of decimals to compute */
                              /* +1 for the very first digit 3 */
   n = m/k + 1 + tol;         /* no of WORD's of each array */
   i = 0;                     /* index to arctan formula table */
   if (argc > 2)              /* Get 2nd param = formula index */
      i = atoi(argv[2]);
   if (i >= sizeof(AtanTab)/sizeof(*AtanTab))
      i = 0;                  /* use Machin if user index too large */
   Result = calloc(n, sizeof(*Result));
   S  = calloc(n, sizeof(*S));
   if (!Result || !S)
      fprintf(stdout, "\nCannot allocate."), exit(1);
   fprintf(stdout, "\nComputing pi to %d decimals by ", m);
   for (j=0; j < 4 && AtanTab[i][j].a != 0; ++j)
      fprintf(stdout, "%+d*atan(1/%d)", AtanTab[i][j].k, AtanTab[i][j].a);
   t0 = clock();
   /* Work along the terms of the atan formula */
   for (j=0; j < 4 && AtanTab[i][j].a != 0; ++j) {
      memset(S, 0, n*sizeof(WORD));
      /* Use next term of atan formula */
      a = AtanTab[i][j].a;
      S[0] = (WORD)(AtanTab[i][j].k);
      imeat = 0;        /* Index to first non-zero word of S */
      aexp2 = a * a;
      denom = a;
      TwoiP1 = 1;         /* (2i+1)*(-1)^i   */
      fprintf(stdout, "\nComputing %d*arctan(1/%d)\n", S[0], a);
      while (imeat < n) { /* loop until all words of S are zero */
         /* fprintf(stdout, "\r%3ld%%", 101L*imeat/n);*/ /* show progress */
         if (denom >= 65536UL) { /* a >= 256 -- denom too large */
            DivideAndStore(a);   /* thus, divide twice by a */
            DivideAndStore(a);
         } else
            DivideAndStore(denom); /* Compute and store Si */
         denom = aexp2;
         DivideAndAdd(TwoiP1);   /* Compute (but don't store)   */
                                 /* Si/(2i+1) and add it */
                                 /* to Result            */
         if (TwoiP1 < 0)         /* Alternating sign */
            TwoiP1 = -TwoiP1 + 2;/* Either -(2n+1)   */
         else
            TwoiP1 = -TwoiP1 - 2;/* Or +(2n+1)       */
      }
   }
   t1 = clock();                 /* t1-t0 is computing time     */
   PrintTheDecimals();
   fprintf(stdout, "\nTime to compute %d decimals was %.1f sec",
                  m, 1.0*(t1-t0)/ CLOCKS_PER_SEC);
   fprintf(stdout, "\nProgram finished.\n");
   return 0;
}
