type AsmVar a = a
type Dynamic a = a

infixl 0 :=
infixl 1 =>>

-- Declaration of primitive functions ---------------------------------------

primitive primAsmVariable "primAsmNewFun0"  
    :: String -> Int -> (a -> a -> Bool) -> a -> Dynamic a
primitive primAsmFunction "primAsmNewFun1"  
    :: String -> Int -> val -> (key -> key -> Int)
         -> (val -> val -> Bool) -> [(key,val)] -> Dynamic (key -> val)
primitive primAsmRead   "primAsmReadFun"  :: Dynamic a -> a
primitive primAsmWrite  "primAsmWriteFun" :: Dynamic a -> a -> IO ()
primitive primAsmUpdate "primAsmUpdateFun":: IO Bool
primitive primAsmAssocs  "primAsmAssocsFun":: Dynamic (key -> val) -> [(key,val)]
primitive primAsmEmptyDom "primAsmEmptyDom":: Dynamic (key -> val) -> Bool
primitive primAsmRandom  "primAsmRandom"      :: Int -> Int
primitive primAsmDefault "primAsmDefault"     :: a
primitive primAsmInDom   "primAsmInDom"
    :: key -> Dynamic (key -> val) -> Bool
primitive primAsmStrictEval  "primAsmStrictEval" :: a -> IO a
primitive primAsmCopyExpr    "primAsmCopyExpr"   :: a -> a
primitive dynamicExpr        "primAsmCopyExpr"   :: a -> a
primitive primAsmNewInteger  "primAsmNewInteger" :: Int
primitive primAsmSetInteger  "primAsmSetInteger" :: Int -> IO ()

primitive genericEq  "primGenericEq"  :: a -> a -> Bool
primitive genericCmp "primGenericCmp" :: a -> a -> Int

primitive primTrace  "primTrace" :: String -> a -> a

-- Rule Monad ---------------------------------------------------------------

data Rule a = Rule (IO a)
            | Skip

skip :: Rule ()
skip = done

instance Functor Rule where
  map f (Rule m) = Rule (map f m)
  map f Skip     = Skip

instance Monad Rule where
  result = Rule  . result
  Rule m `bind` f = Rule (m `bind` (\x -> case f x of
                                           Rule r -> r
                                           Skip   -> result (error "skip occured")))
  Skip `bind` f   = f (error "skip does not return a value")


instance Monad0 Rule where
  zero = Skip

instance MonadPlus Rule where
  a ++ b = do a; b

(=>>) :: Monad m => Bool -> m a -> m ()
True  =>> act = void act
False =>> _   = done

-- AsmTerm Types --------------------------------------------------------------

class AsmOrd a where
   asmCompare :: a -> a -> Int

asmLt :: AsmOrd a => a -> a -> Int
asmLt = asmCompare

class AsmTerm a where
   (:=) :: Dynamic a -> a -> Rule ()
   l := r = Rule (primAsmWrite l r)

   asmDefault :: a
   asmDefault = primAsmDefault ()

instance AsmTerm Bool
  where asmDefault = False

instance AsmTerm Int
instance AsmTerm Float
instance AsmTerm Char
instance AsmTerm String 

instance AsmTerm a => AsmTerm [a]
instance AsmTerm a => AsmTerm {a}
instance (AsmTerm a, AsmTerm b) => AsmTerm (a,b)
instance (AsmTerm a, AsmTerm b, AsmTerm c) => AsmTerm (a,b, c)
instance (AsmTerm a, AsmTerm b, AsmTerm c, AsmTerm d) => AsmTerm (a,b, c, d)

{-  
instance AsmTerm a => AsmTerm [a] where
  f := g  = seqs  (zipWith(:=) f g)

instance (AsmTerm a, AsmTerm b) => AsmTerm (a,b) where
   (l,r) := (l',r') = do l := l'
                         r := r'
-}


instance AsmTerm a => Eq a where
  (==) = genericEq

instance AsmTerm a => AsmOrd a where
  asmCompare = genericCmp

instance AsmTerm a => Text a where
  showsPrec i e = showString (primPrint i e [])

-- ASM dynamic variables and functions  ------------------------------------

initVal :: (Eq a, AsmTerm a) => String -> a -> Dynamic a
initVal name = primAsmVariable name 0 (==)

initValStdin :: String -> Dynamic String
initValStdin name = primAsmVariable name 1 (==) ""

initValStdout :: Bool -> String -> Dynamic String
initValStdout True name  = primAsmVariable name 3 (==) ""
initValStdout False name = primAsmVariable name 2 (==) ""

initAssocs :: (AsmTerm key, AsmTerm val, AsmOrd key, Eq val)
    => String -> [(key,val)] -> Dynamic (key -> val)
initAssocs name = primAsmFunction name 0 asmDefault asmCompare (==)

initAssocsStdin :: (AsmTerm key, AsmOrd key)
    => String -> Dynamic (key -> String)
initAssocsStdin name = primAsmFunction name 1 asmDefault asmCompare (==) []

initAssocsStdout :: (AsmTerm key, AsmOrd key)
    => Bool -> String -> Dynamic (key -> String)
initAssocsStdout True name = primAsmFunction name 3
                                  asmDefault asmCompare (==) []
initAssocsStdout False name = primAsmFunction name 2
                                  asmDefault asmCompare (==) []


initAssocs' :: (AsmTerm key, AsmTerm val, AsmOrd key, Eq val)
    => String -> val -> (key -> key -> Int) -> (val -> val -> Bool) 
        -> [(key,val)] -> Dynamic (key -> val)
initAssocs' name dflt compare eq = primAsmFunction name 0 dflt compare eq

-- access functions --------------------------------------------------------

assocs :: Dynamic (key -> val) -> [(key,val)]
assocs = primAsmAssocs

dom :: Ord key => Dynamic (key -> val) -> {key}
dom = mkSet . map fst . assocs

ran :: Ord val => Dynamic (key -> val) -> {val}
ran = mkSet . map snd . assocs

inDom :: key -> Dynamic (key -> val) -> Bool
inDom = primAsmInDom

emptyDom :: Dynamic (key -> val) -> Bool
emptyDom = primAsmEmptyDom

notInDom :: key -> Dynamic (key -> val) -> Bool
notInDom key fun = not(key `inDom` fun)

-- combining rules ---------------------------------------------------------

forall :: [Rule ()] -> Rule ()
forall = seqs

rforall :: [Rule a] -> Rule [a]
rforall = binds

choose :: [Rule a] -> Rule ()
choose = \xs -> case xs of
                 [] -> Skip
                 _  -> void(xs !! (primAsmRandom (length xs)))

chooseIfNone :: [Rule a] -> Rule () -> Rule ()
chooseIfNone = \xs r -> case xs of
                 [] -> r
                 xs -> do i <- Rule $ strictEval (primAsmRandom (length xs))
                          void (xs !! i)

-- general functions ------------------------------------------------------

rule2io :: Rule a -> IO ()
rule2io (Rule r) = void r
rule2io Skip     = done

rule2io' :: Rule a -> IO a
rule2io' (Rule r) = r
rule2io' Skip     = error "rule2io': skip not expected"

strictEval = primAsmStrictEval

stdout :: Dynamic String
stdout = initValStdout False "stdout"

stdin :: Dynamic String
stdin = initValStdin "stdin"



str2int :: String -> Int
str2int ('-':xs) = - (str2int xs)             
str2int xs = foldl (\r c -> 10*r + ord c - ord '0') 0 xs


-- fire functions on IO Monad ---------------------------------------------

fire1_ :: IO () -> IO Bool
fire1_ rule = do primAsmCopyExpr rule; primAsmUpdate

fireWhile_ :: Bool -> IO () -> IO Bool
fireWhile_ f rule | primAsmCopyExpr f  
                      = do b1 <- fire1_ rule
                           if b1 then do fireWhile_ f rule
                                         result True
                                 else result False
fireWhile_ f rule | otherwise = result False

fireUntil_ :: Bool -> IO () -> IO Bool
fireUntil_ f rule = do b1 <- fire1_ rule
                       if b1 then if primAsmCopyExpr f 
                                   then result True
                                   else fireUntil_ f rule
                             else result False

fixpoint_ :: IO () -> IO ()
fixpoint_ r = do primAsmCopyExpr r
                 b <- primAsmUpdate
                 if b then fixpoint_ r
                      else done

-- fire functions on Rule Monad --------------------------------------------

fire1 :: Rule a -> IO ()
fire1 = void . fire1_ . rule2io

fire :: Int -> Rule () -> IO ()
fire i = void . map or . binds . copy i . fire1_ . rule2io


fireWhile :: Bool -> Rule () -> IO ()
fireWhile f = void . fireWhile_ f . rule2io

fireUntil :: Bool -> Rule () -> IO ()
fireUntil f = void . fireUntil_ f .rule2io


fixpoint :: Rule () -> IO ()
fixpoint = fixpoint_ . rule2io


trace :: IO () -> Rule () -> Rule ()
trace f Skip     = Skip
trace f (Rule r) = Rule (do primAsmCopyExpr f; r)


newInteger :: Int
newInteger = primAsmNewInteger

newIntegers :: Int -> [Int]
newIntegers n = 
              take n
              . map (\i -> i + 0)
              . repeat
              . dynamicExpr $ newInteger + 0

asmSetInteger :: Int -> IO ()
asmSetInteger = primAsmSetInteger

-- create ----------------------------------------------------------------

class Create c where
  createElem    :: Int -> c


instance Create Int where
  createElem    = id

creates :: Create c => (Int, [c] -> Rule b) -> Rule b
creates (i,rule) = do
    let rs = map createElem (newIntegers i)
    rule (rs)

create :: Create c => Int -> ([c] -> Rule b) -> Rule b
create = \i rule -> creates(i,rule)
    
