/***********************************************************************
 *                                                                     *
 *             Information Theoretic Inequality Prover (ITIP)          *
 *                                                                     *
 *                   Raymond W. Yeung and Ying-On Yan                  *
 *                                                                     *
 *                              Version 3.0                            *
 *                    Last updated on August 10, 2001                  *
 *                          All rights reserved                        *
 *                                                                     *
 *         Please send comments and suggestions to the authors at:     *
 *               whyeung@ie.cuhk.edu.hk and yy26@cornell.edu           *
 *                                                                     *
 ***********************************************************************/

#include "mex.h"

double *row,*col,*coef ;
long int nvar,nrow,ncol ;

long int index, count;


void make_NS_1row(long int i1, long int i2, long int i3, long int i4)
{

/*************************************************************************

  Non-Shannon-type Unconstrained Inequality (ZY98):

  I(X1;X2) - 2 I(X3;X4) + I(X1;X3,X4) + I(X3;X4|X2) + 3 I(X3;X4|X1) >= 0

  I(X1;X2)      = H1 + H2 - H12
  -2 I(X3;X4)   = -2 H3 - 2 H4 + 2 H34
  I(X1;X3,X4)   = H1 + H34 - H134
  I(X3;X4|X2)   = H23 + H24 - H2 - H234
  3 I(X3;X4|X1) = 3 H13 + 3 H14 - 3 H1 - 3 H134

  Or,

  I(X1;X2)      =    H1   + H2                                 - H12
  -2 I(X3;X4)   =               2 H34              -2 H3 - 2 H4
  I(X1;X3,X4)   =    H1         + H34  -   H134
  I(X3;X4|X2)   =         - H2                     + H23 + H24 - H234
  3 I(X3;X4|X1) = -3 H1                - 3 H134      + 3 H13 + 3 H14

  Hence,

  -H1 + 3 H34 - 4 H134 - 2 H3 - 2 H4 - H12
                      + H23 + H24 + 3 H13 + 3 H14 - H234 >= 0

*************************************************************************/

	long int ind1, ind2, ind3, ind4 ;

	ind1 = (1L << i1) ;
	ind2 = (1L << i2) ;
	ind3 = (1L << i3) ;
	ind4 = (1L << i4) ;

	/* -H1 + 3 H34 - 4 H134 - 2 H3 - 2 H4 - H12 */

	row[count] = index ;
	col[count] = ind1 ;
	coef[count++] = -1 ;     /*    - H1    */

	row[count] = index ;
	col[count] = ind3 | ind4 ;
	coef[count++] = 3 ;      /*    3 H34   */

	row[count] = index ;
	col[count] = ind1 | ind3 | ind4 ;
	coef[count++] = -4 ;     /*  -4 H134   */

	row[count] = index ;
	col[count] = ind3 ;
	coef[count++] = -2 ;     /*   -2 H3    */

	row[count] = index ;
	col[count] = ind4 ;
	coef[count++] = -2 ;     /*   -2 H4    */

	row[count] = index ;
	col[count] = ind1 | ind2 ;
	coef[count++] = -1 ;     /*    - H12   */

	/* + H23 + H24 + 3 H13 + 3 H14 - H234  */

	row[count] = index ;
	col[count] = ind2 | ind3 ;
	coef[count++] = 1 ;      /*     H23    */

	row[count] = index ;
	col[count] = ind2 | ind4 ;
	coef[count++] = 1 ;      /*     H24    */

	row[count] = index ;
	col[count] = ind1 | ind3 ;
	coef[count++] = 3 ;      /*    3 H13   */
 
	row[count] = index ;
	col[count] = ind1 | ind4 ;
	coef[count++] = 3 ;      /*    3 H14   */

	row[count] = index ;
	col[count] = ind2 | ind3 | ind4 ;
	coef[count++] = -1 ;     /*     H234   */

	index++;
}

void make_NS()
{
	long int n1, n2, n3, n4;
	count = 0 ;
	index = 1 ;

	for(n1=0;n1<(nvar-3);n1++)
		for(n2=n1+1;n2<(nvar-2);n2++)
			for(n3=n2+1;n3<(nvar-1);n3++)
				for(n4=n3+1;n4<nvar;n4++)
				{
					make_NS_1row(n1, n2, n3, n4) ;
					make_NS_1row(n1, n3, n2, n4) ;
					make_NS_1row(n1, n4, n2, n3) ;

					make_NS_1row(n2, n1, n3, n4) ;
					make_NS_1row(n2, n3, n1, n4) ;
					make_NS_1row(n2, n4, n1, n3) ;

					make_NS_1row(n3, n1, n2, n4) ;
					make_NS_1row(n3, n2, n1, n4) ;
					make_NS_1row(n3, n4, n1, n2) ;

					make_NS_1row(n4, n1, n2, n3) ;
					make_NS_1row(n4, n2, n1, n3) ;
					make_NS_1row(n4, n3, n1, n2) ;
				}
}


#ifdef __STDC__
void mexFunction(nlhs,plhs,nrhs,prhs)
#else
mexFunction(nlhs,plhs,nrhs,prhs)
#endif
int nlhs,nrhs;
mxArray *plhs[];
const mxArray *prhs[];
{
	double *ptr ;
	long s ;

	if(nrhs < 1)
		mexErrMsgTxt("One input argument is needed.") ;
	if(nlhs != 5)
		mexErrMsgTxt("Five output arguments are needed.") ;

	nvar = *mxGetPr(prhs[0]) ;

	if(nvar < 4)
		mexErrMsgTxt("At least 4 random variables are needed.") ;

	ncol = (1L << nvar) - 1 ;
	nrow = nvar*(nvar - 1)*(nvar - 2)*(nvar - 3)/2 ;
					/* equal to nvar_chooses_4 * 12 */

	s = nrow * 11 ;		/* Each inequality has 11 nonzero terms */

	plhs[0] = mxCreateDoubleMatrix(s,1,mxREAL) ;
	plhs[1] = mxCreateDoubleMatrix(s,1,mxREAL) ;
	plhs[2] = mxCreateDoubleMatrix(s,1,mxREAL) ;
	plhs[3] = mxCreateDoubleMatrix(1,1,mxREAL) ;
	plhs[4] = mxCreateDoubleMatrix(1,1,mxREAL) ;

	row = mxGetPr(plhs[0]) ;
	col = mxGetPr(plhs[1]) ;
	coef = mxGetPr(plhs[2]) ;
	ptr = mxGetPr(plhs[3]) ;
	*ptr = nrow ;
	ptr = mxGetPr(plhs[4]) ;
	*ptr = ncol ;

	make_NS() ;
}
