package vgp.tutor.vectorField;

import java.util.Enumeration;
import java.util.Vector;

import jv.geom.PgElementSet;
import jv.geom.PgVectorField;
import jv.number.PuDouble;
import jv.object.PsDebug;
import jv.object.PsObject;
import jv.project.PvDisplayIf;
import jv.project.PjProject;
import jv.project.PvGeometryIf;
import jv.vecmath.PdVector;
import jvx.surface.PgDomain;
import jvx.surface.PgDomainDescr;

/**
 * Tutorial project on vector fields.
 * Sample vector field is gradient or rotation of a potential on a square grid. User
 * may interactively modify potential by marking or unmarking vertices.
 * 
 * @see			jv.geom.PgVectorField
 * @author		Konrad Polthier
 * @version		18.09.99, 1.00 created (kp).
 */
public class PjVectorField extends PjProject {
	/** Vectorfield is gradient of potential. */
	public		static final int		GRADIENT = 0;
	/** Vectorfield is J*gradient of potential. */
	public		static final int		ROTATION = 1;
	/** Base geometry is a rectangular grid which carries the vector field. */
	protected	PgDomain					m_geom;
	/** Vector field, associated with base geometry. */
	protected	PgVectorField			m_vf;
	/** Type of vector field. Possible values are GRADIENT, ROTATION etc. */
	protected	int						m_fieldType;

	/** Centers of potential. */
	protected	Vector					m_center;
	/** Radius of influence of each potential. */
	protected	PuDouble					m_radius;
	/** Default value of radius of influence of each potential. */
	protected	double					m_defRadius = 0.5;
	
	public PjVectorField() {
		super("Potential");

		m_vf = new PgVectorField(2);
		m_vf.setName("Vector Field");
		m_fieldType = ROTATION;

		// Create empty surface in 3-dimensional space, and add vector field
		m_geom = new PgDomain(2);
		m_geom.setName("Domain with Vectorfield");
		m_geom.setParent(this);

		m_center = new Vector();
		m_radius = new PuDouble("Radius of Decay", this);
		
		if (getClass() == PjVectorField.class)
			init();
	}

	public void init() {
		super.init();
		m_radius.setDefBounds(0.001, 5., 0.01, 0.1);
		m_radius.setDefValue(m_defRadius);
		m_radius.init();

		PgDomainDescr m_descr = m_geom.getDescr();
		m_descr.setMaxSize(-10., -10., 10., 10.);
		m_descr.setSize( -5., -5., 5., 5.);
		m_descr.setDiscrBounds(2, 2, 50, 50);
		m_descr.setDiscr(20, 20);

		m_geom.computeSurface();
		m_geom.setTagVertex(0, PsObject.IS_SELECTED);
		m_geom.setTagVertex(87, PsObject.IS_SELECTED);
		m_geom.setTagVertex(305, PsObject.IS_SELECTED);
		m_geom.setTagVertex(375, PsObject.IS_SELECTED);
	}
	
	public void start() {
		if (PsDebug.NOTIFY)
			PsDebug.notify("PjVectorField.start:");
		m_geom.addVectorField(m_vf);
		m_vf.setGeometry(m_geom);
		m_geom.showEdges(false);
		m_geom.showElements(false);
		m_geom.update(null);
		addGeometry(m_geom);
		selectGeometry(m_geom);
		PvDisplayIf disp = getDisplay();
		disp.selectCamera(PvDisplayIf.CAMERA_ORTHO_XY);
		super.start();
	}
	/**
	 * Update the class whenever a child has changed.
	 * Method is usually invoked from the children.
	 */
	public boolean update(Object event) {
		if (event == null) {
			super.update(null);
		} else if (event == m_geom) {
			m_center.removeAllElements();
			int numVertices = m_geom.getNumVertices();
			for (int i=0; i<numVertices; i++) {
				if (m_geom.hasTagVertex(i, PsObject.IS_SELECTED))
					m_center.addElement(m_geom.getVertex(i));
			}
			computeVectorfield();
			return true;
		} else if (event == m_radius) {
			computeVectorfield();
			m_geom.update(null);
			return true;
		} else if (event == m_infoPanel) {
			computeVectorfield();
			m_geom.update(null);
			return true;
		}
		return super.update(event);
	}
	public void computeVectorfield() {
		int numVectors			= m_geom.getNumVertices();
		m_vf.setNumVectors(numVectors);
		PdVector [] vector	= m_vf.getVectors();
		PdVector [] vertex	= m_vf.getVertices();
		double f, dfx, dfy;
		int i;
		PdVector p;
		for (i=0; i<numVectors; i++) {
			f = 0.;
			dfx = dfy = 0.;
			for (Enumeration e = m_center.elements() ; e.hasMoreElements() ;) {
				p = (PdVector)e.nextElement();
				f = Math.exp(-2.*PdVector.dist(p, vertex[i])*m_radius.getValue());
				dfx += -2.*(vertex[i].m_data[0]-p.m_data[0])*f;
				dfy += -2.*(vertex[i].m_data[1]-p.m_data[1])*f;
			}
			if (m_fieldType == GRADIENT) {
				vector[i].set(dfx, dfy);
			} else {
				vector[i].set(-dfy, dfx);
			}
		}
		if (m_vf.hasInspector(PsObject.INSPECTOR_INFO) &&
			 m_vf.getInspector(PsObject.INSPECTOR_INFO).isShowing())
			m_vf.getInspector(PsObject.INSPECTOR_INFO).update(m_vf);
	}
	public int getFieldType() { return m_fieldType; }
	public void setFieldType(int type) { m_fieldType = type; }
}

