package vgp.tutor.lsystem;

import java.awt.*;
import java.applet.*;
import java.util.Stack;

import jv.geom.PgPolygonSet;
import jv.number.PuDouble;
import jv.object.PsUpdateIf;
import jv.project.PvDisplayIf;
import jv.vecmath.PdVector;
import jv.vecmath.PiVector;
import jv.viewer.PvViewer;

/**
 * Generate a tree with an L-system and display using turtle graphics.
 * Additionally, this subclass implements the interface PsUpdateIf
 * to catch events from a parameter slider.
 * 
 * @author		Konrad Polthier
 * @version		17.02.00, 1.10 revised (kp) Slider added.<br>
 *					16.02.00, 1.00 created (kp)
 */
public class PaLSystem extends Applet implements PsUpdateIf {
	/** Frame to allow applet to run as application too. */
	protected		Frame						m_frame;
	/** Angle parameter used in L-system. */
	protected		PuDouble					m_delta;
	/** LSystem as collection of polygons. */
	protected		PgPolygonSet			m_polySet;
	/** Production of the L-system encoded with characters from alphabet {F,+,-,[,]}. */
	protected		String					m_descr;
	/**
	 * The entry point for the applet automatically invoked by the browser.
	 * Method creates a string using an L-system and translates the string
	 * into a polygonSet using turtle graphics commands.
	 * Finally, the polygonSet is registered as geometry in a JavaView display.
	 */
	public void init() {
		LSystem tree = new LSystem();
		tree.iterate(200);
		m_descr = tree.getTree();
		Label treeLabel = new Label();
		treeLabel.setText(m_descr);
		
		m_polySet = new PgPolygonSet(2);
		m_polySet.setName("My LSystem");

		// Create a double value with slider, and register applet as parent.
		m_delta = new PuDouble("Delta", this);
		m_delta.setValue(25.7/180. * Math.PI);
		makePolygonSet(m_delta.getValue());
		
		// Create viewer for viewing 3d geometries, and register applet.
		PvViewer viewer = new PvViewer(this, m_frame);
		
		// Get default display from viewer
		PvDisplayIf disp = viewer.getDisplay();
		// Register geometry in display, and make it active.
		// For more advanced applications it is advisable to create a separate project
		// and register geometries in the project via project.addGeometry(geom) calls.
		disp.addGeometry(m_polySet);
		disp.selectGeometry(m_polySet);

		// Add display to applet
		// setLayout(new BorderLayout());
		this.setLayout(new BorderLayout());
		this.add("North", treeLabel);
		this.add("Center", (Component)disp);
		this.add("South", m_delta.getInfoPanel());
		this.validate();
		
		// Start viewer, e.g. to read parameters from Html page (only purpose here)
		viewer.start();
	}
	/**
	 * Recompute the polygon set by translating the string given in m_descr.
	 * This method resets the polygonSet and fills it again.
	 * 
	 * @param		delta		angle in the turtle graphics used when rotating '+' or '-'
	 */
	private void makePolygonSet(double delta) {
		m_polySet.init();
		
		Stack branchPos = new Stack();
		Stack branchPoly = new Stack();
		int vertInd, len = m_descr.length();
		double x, y, a;
		double size = 0.2;
		PdVector pos = new PdVector(3);
		PiVector pg	= new PiVector(3);

		x = 0.;
		y = 0.;
		a = Math.PI/2.;
		pos.set(x, y, a);
		vertInd = m_polySet.addVertex(pos);
		pg.setEntry(0, vertInd);
		for (int i=0; i<len; i++) {
			switch (m_descr.charAt(i)) {
			case 'F':
				a = pos.getEntry(2);
				x = pos.getEntry(0)+size*Math.cos(a);
				y = pos.getEntry(1)+size*Math.sin(a);
				pos.set(x, y, a);
				vertInd = m_polySet.addVertex(pos);
				pg.setEntry(pg.getSize(), vertInd);
				break;
			case '[':
				branchPos.push(pos);
				pos = PdVector.copyNew(pos);
				branchPoly.push(pg);
				pg = new PiVector();
				pg.setEntry(0, vertInd);
				break;
			case ']':
				m_polySet.addPolygon(pg);
				pos = (PdVector)branchPos.pop();
				pg = (PiVector)branchPoly.pop();
				vertInd = pg.getEntry(pg.getSize()-1);
				break;
			case '+':
				pos.set(x, y, pos.getEntry(2)+delta);
				break;
			case '-':
				pos.set(x, y, pos.getEntry(2)-delta);
				break;
			default:
			}
		}
		m_polySet.addPolygon(pg);
	}
	/**
	 * The variable m_delta sends update events to its parent to notify the parent
	 * whenever its value has changed by user interaction.
	 * Method must be implemented to fulfill interface PsUpdateIf.
	 */
	public boolean update(Object event) {
		if (event!=null && event==m_delta) {
			makePolygonSet(m_delta.getValue());
			m_polySet.update(m_polySet);
		}
		return true;
	}
	/** Method must be implemented to fulfill interface PsUpdateIf */
	public PsUpdateIf	getFather() { return null; }
	/** Method must be implemented to fulfill interface PsUpdateIf */
	public void setParent(PsUpdateIf aParent) { return; }
	/**
	 * Standalone application support. The main() method acts as the applet's
	 * entry point when it is run as a standalone application. It is ignored
	 * if the applet is run from within an HTML page.
	 */
	public static void main(String args[]) {
		PaLSystem va	= new PaLSystem();
		// Create toplevel window of application containing the applet
		Frame frame	= new jv.object.PsMainFrame(va, args);
		frame.pack();
		va.m_frame = frame;
		va.init();
		frame.setBounds(new Rectangle(420, 5, 640, 550));
		frame.setVisible(true);
	}
}
