// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

package nom.rb.MtgClt;
import nom.rb.common.*;

import java.awt.*;


///////////////////////////////////////////////////////////////////

//
//   A d v a n c e d
//

public class Advanced extends ParamSpace {


//
//   v a r i a b l e s
//

    // the following variables are for the GUI        

private boolean m_oneClick;

private CheckboxGroup m_daysPerYearGroup;
private ExtCheckbox m_daysPerYear255;
private ExtCheckbox m_daysPerYear365;
private ExtCheckbox m_daysPerYear365_25;
private ExtCheckbox m_daysPerYearOther;
private TextField m_daysPerYear;

private CheckboxGroup m_trimmingGroup;
private ExtCheckbox m_trimmingOff;
private ExtCheckbox m_trimmingOn;
private ExtTextField m_deviations;
private Label m_devLabel;

private ExtCheckbox m_steps10;
private ExtCheckbox m_steps5;
private ExtCheckbox m_steps2;
private ExtCheckbox m_steps1;
private ExtCheckbox m_steps_2;
private ExtCheckbox m_steps_5;
private ExtCheckbox m_steps_10;
private ExtCheckbox m_steps_20;
private ExtCheckbox[] m_stepsOther = new ExtCheckbox[4];
private ExtTextField[] m_stepsEntry = new ExtTextField[4];

private CheckboxGroup m_customGroup;
private ExtCheckbox m_customLarge;
private ExtCheckbox m_customSmall;

private CheckboxGroup m_grAccuracyGroup;
private ExtCheckbox m_cbExactAccuracy;
private ExtCheckbox m_cbLowAccuracy;

private CheckboxGroup m_grFDGroup;
private ExtCheckbox m_cbExplicit;
private ExtCheckbox m_cbImplicit;

private final static String[] m_myExamples = {
    "Evaluation in a leap year with 2, 3, 4 and 5 timesteps per day",
    "Counting business days only, an approximation with large timesteps",
    "If no time step is selected, \"One per day\" is the default",
    "To select several timesteps helps to get an idea of convergence",
    "Lattice trimming off increases the running time considerably"
};

private final static int m_maxLoadIndex = 1000;

    // these are the parameters for the calculation

private double m_paramDaysPerYear;
private double m_paramDeviations;
private int m_paramTimestep[];
private int m_paramAccuracy;
private int m_paramFD;


//
//   A d v a n c e d
//

public Advanced()

{
    this( false );
}


//
//   A d v a n c e d
//

public Advanced( boolean oneClick )

{
    super( Id.ADVANCED );

    m_oneClick = oneClick;

    GridBagLayout layout = new GridBagLayout();
    setLayout( layout );

    GridBagConstraints c = new GridBagConstraints();

    c.anchor = GridBagConstraints.NORTHWEST;
    c.gridx = 0;

    if( m_oneClick ) {
        c.insets = new Insets( 0, 30, 0, 0 );
        add( createTable2(), c );
    }
    else {
        add( createTable1(), c );
        c.insets = new Insets( 30, 0, 0, 0 );
        add( createTable2(), c );
        c.insets = new Insets( 20, 0, 0, 0 );
        add( createTable3(), c );
        c.gridx = 1;
        c.gridheight = GridBagConstraints.REMAINDER;
        c.insets = new Insets( 0, 40, 0, 0 );
        add( createTable4(), c );
    }

    if( Pool.m_bgColor != null )
        setBackground( Pool.m_bgColor );

    m_examples = m_myExamples;

    clear();                    
}


//
//   c l e a r
//

public void clear()

{
    clearWrong();

    m_steps1.setState( true );
    m_steps_5.setState( false );
    m_steps_10.setState( false );
    m_steps_20.setState( false );

    if( ! m_oneClick ) {
        m_daysPerYearGroup.setCurrent( m_daysPerYear365 );
        m_trimmingGroup.setCurrent( m_trimmingOn );
        m_customGroup.setCurrent( m_customSmall );

        m_daysPerYear.setText( "" );
        m_deviations.setText( "3.5" );

        m_daysPerYear.disable();
        m_deviations.enable();
        m_devLabel.enable();

        m_steps10.setState( false );
        m_steps5.setState( false );
        m_steps2.setState( false );
        m_steps_2.setState( false );

        for( int k = 0; k < m_stepsOther.length; ++k ) {
            m_stepsOther[k].setState( false );
            m_stepsEntry[k].setText( "" );
            m_stepsEntry[k].disable();
        }

        m_grAccuracyGroup.setCurrent( m_cbExactAccuracy );
    }
}


//
//   e x a m p l e
//

public void example( int num )

{
    if( m_oneClick ) {
    }
    else {
        clear();

        switch( num ) {
            case 1 :
                m_daysPerYearGroup.setCurrent( m_daysPerYearOther );
                m_daysPerYear.setText( "366" );
                m_daysPerYear.enable();
                m_steps1.setState( false );
                m_steps_2.setState( true );
                m_steps_5.setState( true );
                m_stepsOther[0].setState( true );
                m_stepsEntry[0].enable();
                m_stepsEntry[0].setText( "3" );
                m_stepsOther[1].setState( true );
                m_stepsEntry[1].enable();
                m_stepsEntry[1].setText( "4" );
                break;

            case 2 :
                m_daysPerYearGroup.setCurrent( m_daysPerYear255 );
                m_steps10.setState( true );
                m_steps5.setState( true );
                m_steps1.setState( false );
                break;

            case 3 :
                m_steps1.setState( false );
                break;

            case 4 :
                m_steps_2.setState( true );
                m_steps_5.setState( true );
                m_steps_10.setState( true );
                m_steps_20.setState( true );
                m_stepsOther[0].setState( true );
                m_stepsEntry[0].enable();
                m_stepsEntry[0].setText( "50" );
                break;

            case 5 :
                m_trimmingGroup.setCurrent( m_trimmingOff );
                m_deviations.disable();
                break;
        }
    }
}


//
//   a c t i o n
//

public boolean action( Event evt, Object what )

{
    if( evt.target instanceof ExtCheckbox ) {
        int id = ( (ExtCheckbox) evt.target ).id();

        switch( id ) {
            case Id.TRIMMINGOFF :
                m_deviations.disable();
                m_devLabel.disable();
                return true;

            case Id.TRIMMINGON :
                m_deviations.enable();
                m_devLabel.enable();
                return true;

            case Id.DAYSPERYEAR255 :
            case Id.DAYSPERYEAR365 :
            case Id.DAYSPERYEAR365_25 :
                m_daysPerYear.disable();
                return true;

            case Id.DAYSPERYEAROTHER :
                m_daysPerYear.enable();
                return true;

            default :
                if( id >= Id.STEPSOTHER &&
                    id < Id.STEPSOTHER + m_stepsOther.length ) {
                    int k = id - Id.STEPSOTHER;
                    m_stepsEntry[k].enable( m_stepsOther[k].getState() );
                }
                break;
        }
    }

    return false;
}


//
//   p r e p a r e
//

public void prepare( Portfolio portfolio, Scenario scenario )
    throws DataException

{
    clearWrong();

    if( m_oneClick ) {
        m_paramDaysPerYear = 365;
        m_paramDeviations = 3.5;
        m_paramAccuracy = Id.ACCURACYEXACT;
        m_paramFD = Id.EXPLICIT;
    }
    else {
        ExtCheckbox p = (ExtCheckbox) m_daysPerYearGroup.getCurrent();
        if( p == null )
            panic( "No days per year selection made" );

        switch( p.id() ) {
            case Id.DAYSPERYEAR255 :
                m_paramDaysPerYear = 255;
                break;

            case Id.DAYSPERYEAR365 :
                m_paramDaysPerYear = 365;
                break;

            case Id.DAYSPERYEAR365_25 :
                m_paramDaysPerYear = 365.25;
                break;

            case Id.DAYSPERYEAROTHER :
                m_paramDaysPerYear = getPosDouble( m_daysPerYear );
                break;

            default :
                throw new InternalError();
        }

        ExtCheckbox q = (ExtCheckbox) m_trimmingGroup.getCurrent();
        if( q == null )
            panic( "No trimming selection made" );

        switch( q.id() ) {
            case Id.TRIMMINGOFF :
                m_paramDeviations = 0;
                break;

            case Id.TRIMMINGON :
                m_paramDeviations = getPosDouble( m_deviations );
                break;

            default :
                throw new InternalError();
        }

        ExtCheckbox r = (ExtCheckbox) m_grAccuracyGroup.getCurrent();
        if( r == null )
            panic( "No accuracy selection made" );

        m_paramAccuracy = r.id();

        ExtCheckbox s = (ExtCheckbox) m_grFDGroup.getCurrent();
        if( s == null )
            panic( "No finite difference scheme selected" );

        m_paramFD = s.id();

        if( m_paramFD == Id.EXPLICIT && scenario.paramVolShock() ) {
            int c = Pool.yesNoCancel( "Warning",
                    "The volatility shock scenario can be slow under the\n" +
                    "explicit scheme. Shall I activate Crank-Nicholson\n" +
                    "for this computation only?" );

            if( c == MessageBox.ID_CANCEL )
                panic();
            if( c == MessageBox.ID_YES )
                m_paramFD = Id.IMPLICIT;
        }
    }

    prepareSteps();
    prepareSize( portfolio );
}


//
//   p a r a m T i m e s t e p
//

public int[] paramTimestep()

{
    return m_paramTimestep;
}


//
//   p r o t M o d e l
//

public String protModel()

{
    return "days_per_year " + m_paramDaysPerYear;
}


//
//   p r o t A d v a n c e d
//

public String protAdvanced( Scenario scenario )

{
    String s = "";
    String p = "";
    String q = "";

    switch( m_paramAccuracy ) {
        case Id.ACCURACYEXACT:
            q = ", accuracy exact";
            break;

        case Id.ACCURACYLOW:
            q = ", accuracy low";
            break;

        default:
            throw new InternalError();
    }

    switch( m_paramFD ) {
        case Id.EXPLICIT:
            p = ", method explicit, box";
            break;

        case Id.IMPLICIT:
            p = ", method implicit, box";
            break;

        default:
            throw new InternalError();
    }

    for( int k = 0; k < m_paramTimestep.length; ++k ) {
        s += "lattice L { model M, portfolio P" + p;

        if( m_paramDeviations > 0 )
            s += " " + m_paramDeviations;
        if( m_paramTimestep[k] > 0 ) {
            s += ", time_step " + m_paramTimestep[k];
        }
        else {
            double t = 1 / (double) -m_paramTimestep[k];
            s += ", time_step " + t;
        }
        s += " }\n";

        if( scenario.paramSeller() ) {
            s += "evaluate { model M, lattice L, " + 
                    "portfolio P, scenario SELLER" + q + " }\n";
        }
        if( scenario.paramBuyer() ) {
            s += "evaluate { model M, lattice L, " + 
                    "portfolio P, scenario BUYER" + q + " }\n";
        }
    }

    return s;
}


//
//   c o p y F r o m
//

public void copyFrom( Advanced advanced )

{
    m_steps1.setState( advanced.m_steps1.getState() );
    m_steps_5.setState( advanced.m_steps_5.getState() );
    m_steps_10.setState( advanced.m_steps_10.getState() );
    m_steps_20.setState( advanced.m_steps_20.getState() );
}


//
//   c r e a t e T a b l e 1
//

private Panel createTable1()

{
    if( m_oneClick )
        throw new InternalError();

    Panel table = new Panel();

    GridBagLayout layout = new GridBagLayout();
    table.setLayout( layout );

    GridBagConstraints c = new GridBagConstraints();
    c.anchor = GridBagConstraints.NORTHWEST;

    m_daysPerYearGroup = new CheckboxGroup();
    m_trimmingGroup = new CheckboxGroup();

    m_daysPerYear255 = new ExtCheckbox( Id.DAYSPERYEAR255,
        "255", m_daysPerYearGroup, false );
    m_daysPerYear365 = new ExtCheckbox( Id.DAYSPERYEAR365,
        "365", m_daysPerYearGroup, true );
    m_daysPerYear365_25 = new ExtCheckbox( Id.DAYSPERYEAR365_25,
        "365.25", m_daysPerYearGroup, false );
    m_daysPerYearOther = new ExtCheckbox( Id.DAYSPERYEAROTHER,
        "Other", m_daysPerYearGroup, false );
    m_daysPerYear = new ExtTextField( 0, 6 );

    m_trimmingOff = new ExtCheckbox( Id.TRIMMINGOFF,
        "Off", m_trimmingGroup, false );
    m_trimmingOn = new ExtCheckbox( Id.TRIMMINGON,
        "On", m_trimmingGroup, true );
    m_deviations = new ExtTextField( 0, 6 );

    c.gridx = 0;
    c.gridwidth = 3;
    add( table, new Label( "Days per year" ), c );
    c.gridwidth = 1;
    add( table, m_daysPerYear255, c );
    add( table, m_daysPerYear365, c );
    c.gridx = 1;
    c.gridy = 1;
    c.gridwidth = 2;
    add( table, m_daysPerYear365_25, c );
    c.gridwidth = 1;
    c.gridy = 2;
    add( table, m_daysPerYearOther, c );
    c.gridx = 2;
    c.gridy = 2;
    add( table, m_daysPerYear, c );

    c.insets = new Insets( 0, 30, 0, 0 );
    c.gridx = 3;
    c.gridy = -1;
    c.gridwidth = 2;
    add( table, new Label( "Lattice trimming" ), c );
    c.gridwidth = 1;
    add( table, m_trimmingOff, c );
    add( table, m_trimmingOn, c );
    c.gridx = 4;
    c.gridy = 1;
    c.insets = new Insets( 0, 0, 0, 0 );
    m_devLabel = new Label( "Deviations" );
    add( table, m_devLabel, c );
    c.gridy = 2;
    add( table, m_deviations, c );

    if( Pool.m_bgColor != null ) {
        m_daysPerYear255.setBackground( Pool.m_bgColor );
        m_daysPerYear365.setBackground( Pool.m_bgColor );
        m_daysPerYear365_25.setBackground( Pool.m_bgColor );
        m_daysPerYearOther.setBackground( Pool.m_bgColor );
        m_trimmingOff.setBackground( Pool.m_bgColor );
        m_trimmingOn.setBackground( Pool.m_bgColor );
        table.setBackground( Pool.m_bgColor );
    }
    table.layout();

    return table;
}


//
//   c r e a t e T a b l e 2
//

private Panel createTable2()

{
    Panel table = new Panel();

    GridBagLayout layout = new GridBagLayout();
    table.setLayout( layout );

    GridBagConstraints c = new GridBagConstraints();

    c.anchor = GridBagConstraints.NORTHWEST;

    if( m_oneClick ) {
        m_steps1 = new ExtCheckbox( Id.STEPS1, "1 per day" );
        m_steps_5 = new ExtCheckbox( Id.STEPS_5, "5 per day" );
        m_steps_10 = new ExtCheckbox( Id.STEPS_10, "10 per day" );
        m_steps_20 = new ExtCheckbox( Id.STEPS_20, "20 per day" );
    }
    else {
        m_steps1 = new ExtCheckbox( Id.STEPS1, "One per day" );
        m_steps_5 = new ExtCheckbox( Id.STEPS_5, "5 per day" );
        m_steps_10 = new ExtCheckbox( Id.STEPS_10, "10 per day" );
        m_steps_20 = new ExtCheckbox( Id.STEPS_20, "20 per day" );
    }

    if( Pool.m_bgColor != null ) {
        m_steps1.setBackground( Pool.m_bgColor );
        m_steps_5.setBackground( Pool.m_bgColor );
        m_steps_10.setBackground( Pool.m_bgColor );
        m_steps_20.setBackground( Pool.m_bgColor );
        table.setBackground( Pool.m_bgColor );
    }

    if( m_oneClick ) {
        c.gridx = 0;
        c.gridwidth = 2;
        add( table, new Label( "Timesteps" ), c );
        c.gridwidth = 1;
        c.gridy = 1;
        add( table, m_steps1, c );
        c.gridy = 2;
        add( table, m_steps_5, c );
        c.gridx = 1;
        c.gridy = 1;
        add( table, m_steps_10, c );
        c.gridy = 2;
        add( table, m_steps_20, c );
    }
    else {
        m_steps10= new ExtCheckbox( Id.STEPS5, "Every 10 days" );
        m_steps5 = new ExtCheckbox( Id.STEPS5, "Every 5 days" );
        m_steps2 = new ExtCheckbox( Id.STEPS2, "Every 2 days" );
        m_steps_2 = new ExtCheckbox( Id.STEPS_2, "2 per day" );

        c.insets = new Insets( 0, 0, 0, 10 );
        c.gridx = 0;
        add( table, new Label( "Large timesteps" ), c );
        add( table, m_steps10, c );
        add( table, m_steps5, c );
        add( table, m_steps2, c );
        add( table, m_steps1, c );
        c.gridx = 1;
        add( table, new Label( "Small timesteps" ), c );
        add( table, m_steps_2, c );
        add( table, m_steps_5, c );
        add( table, m_steps_10, c );
        add( table, m_steps_20, c );
        c.gridx = 2;
        c.gridwidth = 2;
        add( table, new Label( "Custom timesteps" ), c );
        c.gridwidth = 1;
        c.insets = new Insets( 0, 0, 0, 0 );

        for( int k = 0; k < m_stepsOther.length; ++k ) {
            c.gridy = k + 1;
            c.gridx = 2;
            m_stepsOther[k] = new ExtCheckbox( Id.STEPSOTHER + k, "Other" );
            add( table, m_stepsOther[k], c );
            c.gridx = 3;
            m_stepsEntry[k] = new ExtTextField( 0, 6 );
            add( table, m_stepsEntry[k], c );
        }

        if( Pool.m_bgColor != null ) {
            m_steps10.setBackground( Pool.m_bgColor );
            m_steps5.setBackground( Pool.m_bgColor );
            m_steps2.setBackground( Pool.m_bgColor );
            m_steps_2.setBackground( Pool.m_bgColor );
            for( int k = 0; k < m_stepsOther.length; ++k ) 
                m_stepsOther[k].setBackground( Pool.m_bgColor );
            table.setBackground( Pool.m_bgColor );
        }
    }

    return table;
}


//
//   c r e a t e T a b l e 3
//

private Panel createTable3()

{
    if( m_oneClick )
        throw new InternalError();

    Panel table = new Panel();

    GridBagLayout layout = new GridBagLayout();
    table.setLayout( layout );

    GridBagConstraints c = new GridBagConstraints();
    c.anchor = GridBagConstraints.NORTHWEST;

    m_customGroup = new CheckboxGroup();

    m_customLarge = new ExtCheckbox( Id.CUSTOMLARGE,
        "days", m_customGroup, false );
    m_customSmall = new ExtCheckbox( Id.CUSTOMSMALL,
        "periods per day", m_customGroup, true );

    c.gridy = 0;
    add( table, new Label( "Custom timesteps count" ), c );
    add( table, m_customLarge, c );
    add( table, m_customSmall, c );

    if( Pool.m_bgColor != null ) {
        m_customLarge.setBackground( Pool.m_bgColor );
        m_customSmall.setBackground( Pool.m_bgColor );
        table.setBackground( Pool.m_bgColor );
    }

    return table;
}


//
//   c r e a t e T a b l e 4
//

private Panel createTable4()

{
    if( m_oneClick )
        throw new InternalError();

    Panel table = new Panel();

    GridBagLayout layout = new GridBagLayout();
    table.setLayout( layout );

    GridBagConstraints c = new GridBagConstraints();
    c.anchor = GridBagConstraints.NORTHWEST;

    m_grAccuracyGroup = new CheckboxGroup();

    m_cbExactAccuracy = new ExtCheckbox( Id.ACCURACYEXACT,
        "exact (always exact, but slow)", m_grAccuracyGroup, true );;
    m_cbLowAccuracy = new ExtCheckbox( Id.ACCURACYLOW,
        "low (approximate, but fast)", m_grAccuracyGroup, false );;

    m_grFDGroup = new CheckboxGroup();

    m_cbExplicit = new ExtCheckbox( Id.EXPLICIT,
        "explicit", m_grFDGroup, true );;
    m_cbImplicit = new ExtCheckbox( Id.IMPLICIT,
        "Crank-Nicholson", m_grFDGroup, false );;

    c.gridx = 0;
    add( table, new Label( "Accuracy for American options" ), c );
    add( table, m_cbExactAccuracy, c );
    add( table, m_cbLowAccuracy, c );

    c.insets = new Insets( 20, 0, 0, 0 );
    add( table, new Label( "Finite difference scheme" ), c );
    c.insets = new Insets( 0, 0, 0, 0 );
    add( table, m_cbExplicit, c );
    add( table, m_cbImplicit, c );

    if( Pool.m_bgColor != null ) {
        m_cbExactAccuracy.setBackground( Pool.m_bgColor );
        m_cbLowAccuracy.setBackground( Pool.m_bgColor );
        m_cbExplicit.setBackground( Pool.m_bgColor );
        m_cbImplicit.setBackground( Pool.m_bgColor );
        table.setBackground( Pool.m_bgColor );
    }

    return table;
}


//
//   p r e p a r e S t e p s
//

private void prepareSteps()
    throws DataException

{
    int numOfSteps = 0;

    if( m_steps1.getState() )   ++numOfSteps;
    if( m_steps_5.getState() )  ++numOfSteps;
    if( m_steps_10.getState() ) ++numOfSteps;
    if( m_steps_20.getState() ) ++numOfSteps;

    if( ! m_oneClick ) {
        if( m_steps10.getState() )  ++numOfSteps;
        if( m_steps5.getState() )   ++numOfSteps;
        if( m_steps2.getState() )   ++numOfSteps;
        if( m_steps_2.getState() )  ++numOfSteps;

        for( int k = 0; k < m_stepsOther.length; ++k ) {
            if( m_stepsOther[k].getState() )
                ++numOfSteps;
        }
    }

    if( numOfSteps == 0 ) {
        m_steps1.setState( true );
        numOfSteps = 1;
    }

    m_paramTimestep = new int[numOfSteps];

    numOfSteps = 0;

    if( ! m_oneClick ) {
        if( m_steps10.getState() )  m_paramTimestep[numOfSteps++] = 10;
        if( m_steps5.getState() )   m_paramTimestep[numOfSteps++] = 5;
        if( m_steps2.getState() )   m_paramTimestep[numOfSteps++] = 2;
    }

    if( m_steps1.getState() )
        m_paramTimestep[numOfSteps++] = 1;

    if( ! m_oneClick ) {
        if( m_steps_2.getState() )
            m_paramTimestep[numOfSteps++] = -2;
    }

    if( m_steps_5.getState() )  m_paramTimestep[numOfSteps++] = -5;
    if( m_steps_10.getState() ) m_paramTimestep[numOfSteps++] = -10;
    if( m_steps_20.getState() ) m_paramTimestep[numOfSteps++] = -20;

    if( ! m_oneClick ) {
        for( int k = 0; k < m_stepsOther.length; ++k ) {
            if( m_stepsOther[k].getState() ) {
                int step = 0;

                try {
                    if( m_customLarge.getState() )
                        step = m_stepsEntry[k].getPosInt();
                    else
                        step = -m_stepsEntry[k].getPosInt();
                }
                catch( DataException e ) {
                    setWrong( m_stepsEntry[k], e );
                }

                int j = numOfSteps;
                while( j > 0 && m_paramTimestep[j - 1] < step )
                    --j;

                if( j == 0 || m_paramTimestep[j - 1] > step ) {
                    for( int i = numOfSteps++; i > j; --i )
                        m_paramTimestep[i] = m_paramTimestep[i - 1];
                    m_paramTimestep[j] = step;
                }
            }
        }
    }

    if( numOfSteps < m_paramTimestep.length ) {
        int x[] = new int[numOfSteps];

        for( int j = 0; j < numOfSteps; ++j )
            x[j] = m_paramTimestep[j];

        m_paramTimestep = x;
    }
}


//
//   p r e p a r e S i z e
//

private void prepareSize( Portfolio portfolio )
    throws DataException

{       // limit the size of the problem        
    int big = portfolio.paramMaturity();

    if( m_paramTimestep[m_paramTimestep.length - 1] > 0 )
        big /= m_paramTimestep[m_paramTimestep.length - 1];
    else
        big *= -m_paramTimestep[m_paramTimestep.length - 1];

    if( m_paramDeviations > 0 )
        big = (int) ( m_paramDeviations * Math.sqrt( big ) );

    big /= 100;
    big *= 100;

    if( big > m_maxLoadIndex ) {
        panic( "You're overloading the server.\n" +
               "The load index for your problem is " + big + ".\n" +
               "In this version of the program, the load index must\n" +
               "be " + m_maxLoadIndex +
               " or less (this is a rather arbitrary coice).\n\n" +
               "The load index is roughly the size of the lattice.\n\n" +
               "Please make your problem smaller." );
    }
}

} // end of class
