// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

package nom.rb.MtgClt;
import nom.rb.common.*;

import java.awt.*;


///////////////////////////////////////////////////////////////////

//
//   R e s u l t G r a p h
//

public class ResultGraph extends Canvas {

private double[][] m_lastSellSamples;
private double[][] m_lastBuySamples;

private double[][] m_sellSamples;
private double[][] m_buySamples;

private double m_assetPrice;

private double m_minX;
private double m_maxX;
private double m_minY;
private double m_maxY;

private double m_unitX;
private double m_unitY;
private double m_originX;
private double m_originY;

private double m_ticksX[];
private String m_labelX[];
private double m_ticksY[];
private String m_labelY[];

private boolean m_axisX;
private boolean m_axisY;

private boolean m_initialized = false;


//
//   R e s u l t G r a p h
//

public ResultGraph()

{
}


//
//   s e t S a m p l e s
//

public void setSamples( double assetPrice, double[][] samples )

{
    setSamples( assetPrice, samples, samples );
}


//
//   s e t S a m p l e s
//

public void setSamples( double assetPrice, 
    double[][] sellSamples, double[][] buySamples )

{
    m_lastSellSamples = m_sellSamples;
    m_lastBuySamples = m_buySamples;

    m_assetPrice = assetPrice;
    m_sellSamples = sellSamples;
    m_buySamples = buySamples;

    initialize();
}


//
//   u p d a t e
//

public void update( Graphics g )

{
    paint( g );
}


//
//   p a i n t
//
    
public void paint( Graphics g )

{
    Rectangle b = g.getClipRect();

    g.setColor( Color.white );
    g.fillRect( b.x, b.y, b.width, b.height );
    g.setColor( Color.black );

    if( ! m_initialized )
        return;

    if( ! setupCoordSpace( g.getFontMetrics() ) )
        return;

    paintAxis( g );

    g.setColor( Color.lightGray );

    if( m_lastSellSamples != null )
        paintData( g, m_lastSellSamples );

    if( m_lastBuySamples != null && m_lastBuySamples != m_lastSellSamples )
        paintData( g, m_lastBuySamples );

    if( m_sellSamples != null ) {
        if( m_buySamples != null && m_buySamples != m_sellSamples )
            g.setColor( Color.red );
        else
            g.setColor( Color.black );

        paintData( g, m_sellSamples );
    }

    if( m_buySamples != null && m_buySamples != m_sellSamples ) {
        if( m_sellSamples != null )    
            g.setColor( Color.blue );
        else
            g.setColor( Color.black );
        paintData( g, m_buySamples );
    }
}


//
//   m i n i m u m S i z e
//
    
public Dimension minimumSize()

{
    return new Dimension( 0, 0 );
}


//
//   p r e f e r r e d S i z e
//

public Dimension preferredSize()

{                           
    FontMetrics f = getFontMetrics( getFont() );
    return new Dimension( 20 * f.charWidth( 'M' ), 8 * f.getHeight() );
}


//
//   m o u s e U p
//

public boolean mouseUp( Event evt, int x, int y )

{
    if( ! m_initialized )
        return true;

    if( m_lastBuySamples != null || m_lastSellSamples != null ) {
        m_lastBuySamples = null;
        m_lastSellSamples = null;
        initialize();
    }

    return true;
}


private void initialize()

{
    if( m_sellSamples != null ) {
        adjustMinMax( m_sellSamples, true );
        if( m_buySamples != null )
            adjustMinMax( m_buySamples, false );
    }
    else
    if( m_buySamples != null ) {
        adjustMinMax( m_buySamples, true );
    }
    else {
        m_initialized = false;
        return;
    }

    if( m_lastSellSamples != null || m_lastBuySamples != null ) {
        double x;

        if( m_lastSellSamples != null ) {
            x = ( m_lastSellSamples[0][0] + 
                m_lastSellSamples[m_lastSellSamples.length - 1][0] ) / 2;
        }
        else {
            x = ( m_lastBuySamples[0][0] + 
                m_lastBuySamples[m_lastBuySamples.length - 1][0] ) / 2;
        }

        if( x >= m_minX && x <= m_maxX ) {
            if( m_lastSellSamples != null )
                adjustMinMax( m_lastSellSamples, false );
            if( m_lastBuySamples != null &&
                m_lastBuySamples != m_lastSellSamples ) {
                adjustMinMax( m_lastBuySamples, false );
            }
        }
        else {
            m_lastSellSamples = null;
            m_lastBuySamples = null;
        }
    }

    if( ! findTicks() ) {
        m_initialized = false;
        return;
    }

    m_initialized = true;
    repaint();
}


//
//   a d j u s t M i n M a x
//

private void adjustMinMax( double[][] samples, boolean first )

{
    int l = samples.length;

    if( first || samples[0][0] < m_minX )
        m_minX = samples[0][0];

    if( first || samples[l - 1][0] > m_maxX )
        m_maxX = samples[l - 1][0];

    double minY = samples[0][1];
    double maxY = minY;

    for( int i = 1; i < l; ++i ) { 
        if( samples[i][1] < minY )
            minY = samples[i][1];
        else
            if( samples[i][1] > maxY )
                maxY = samples[i][1];
    }

    if( first || minY < m_minY )
        m_minY = minY;
    if( first || maxY > m_maxY )
        m_maxY = maxY;
}


//
//   f i n d T i c k s
//

private boolean findTicks()

{
    if( m_minX > 0 && m_maxX - m_minX >= 5 * m_minX )
        m_minX = 0;

    if( ( m_ticksX = findTicks( m_minX, m_maxX ) ) == null )
        return false;

    m_labelX = ticks2Label( m_ticksX );

    if( m_minY > 0 )
        m_minY = 0;
    else
        if( m_maxY < 0 )
            m_maxY = 0;

    if( ( m_ticksY = findTicks( m_minY, m_maxY ) ) == null )
        return false;

    m_labelY = ticks2Label( m_ticksY );

    return true;
}


//
//   f i n d T i c k s
//

private double[] findTicks( double low, double high )

{
    if( low == high ) {
        double[] t = new double[2];

        t[0] = Math.ceil( low - 1 );
        t[1] = Math.floor( low + 1 );

        return t;
    }

    double d = ( high - low ) / 3;
    long scale = 1;

    while( d < 1 ) {
        low *= 10;
        high *= 10;
        d *= 10;
        scale *= 10;
        if( scale < 0 )
            return null;    // overflow
    }

    long base = 1;

    while( base < d ) {
        base *= 10;
        if( base < 0 )
            return null;    // overflow
    }

    if( base > d ) {
        if( base / 2 <= d )
            base /= 2;
        else
        if( base / 5 <= d )
            base /= 5;
        else
            base /= 10;
    }

    double l = ( low == 0 ) ? 0 : base * Math.floor( low / base );
    double h = ( high == 0 ) ? 0 : base * Math.ceil( high / base );

    double[] t = new double[2];

    t[0] = l / scale;
    t[1] = h / scale;

    return t;
}


//
//   t i c k s 2 L a b e l
//

private String[] ticks2Label( double[] ticks )

{
    String t[] = new String[ticks.length];

    for( int i = 0; i < ticks.length; ++i ) {
        t[i] = Double.toString( ticks[i] );
        if( t[i].endsWith( ".0" ) )
            t[i] = t[i].substring( 0, t[i].length() - 2 );
    }

    return t;
}


//
//   s e t u p C o o r d S p a c e
//

private boolean setupCoordSpace( FontMetrics fm )

{
    Rectangle b = bounds();

    int w = b.width - 1;
    int h = b.height - 1;

    if( w <= 1 || h <= 1 )
        return false;

    int mx1 = 0;
    int mx2 = 0;
    int my1 = 0;
    int my2 = 0;

    m_axisX = false;
    m_axisY = false;

    if( 4 * fm.getHeight() <= h ) {
        int l = 0;                
        for( int i = 0; i < m_ticksY.length; ++i ) {
            int l1 = fm.stringWidth( m_labelY[i] );
            if( l1 > l )
                l = l1;
        }

        if( l + 8 <= w / 3 ) {
            mx1 = l + 8;
            my1 = ( fm.getHeight() + 4 ) / 2;
            my2 = my1;
            m_axisY = true;
        }

        int tx1 = ( fm.stringWidth( m_labelX[0] ) + 4 ) / 2;
        int tx2 = ( fm.stringWidth( m_labelX[m_ticksX.length - 1] ) + 4 ) / 2;

        if( ( tx1 <= mx1 || tx1 <= w / 4 ) && tx2 <= w / 4 ) {
            if( tx1 > mx1 )
                mx1 = tx1;
            mx2 = tx2;
            int ty2 = fm.getHeight() + 6;
            if( ty2 > my2 )
                my2 = ty2;
            m_axisX = true;
        }
    }

    double x1 = m_ticksX[0];
    double x2 = m_ticksX[m_ticksX.length - 1];
    double y1 = m_ticksY[0];
    double y2 = m_ticksY[m_ticksY.length - 1];

    if( mx1 == 0 ) mx1 = 1;
    if( mx2 == 0 ) mx2 = 1;
    if( my1 == 0 ) my1 = 1;
    if( my2 == 0 ) my2 = 1;

    m_unitX = ( x2 - x1 ) / ( w - ( mx1 + mx2 ) );
    m_unitY = ( y2 - y1 ) / ( h - ( my1 + my2 ) );

    m_originX = -x1 / m_unitX + mx1;
    m_originY = -y1 / m_unitY + my2;

    m_originY = h - m_originY;
    m_unitY = -m_unitY;

    return true;
}


//
//   p a i n t A x i s
//

private void paintAxis( Graphics g )

{
    FontMetrics fm = g.getFontMetrics();

    int x1 = xmark( m_ticksX[0] );
    int x2 = xmark( m_ticksX[m_ticksX.length - 1] );

    int y1 = ymark( m_ticksY[0] );
    int y2 = ymark( m_ticksY[m_ticksY.length - 1] );

    if( m_axisX ) {
        int b = y1 + fm.getAscent() + 4;

        g.drawLine( x1, y1, x2, y1 );

        for( int i = 0; i < m_ticksX.length; ++i ) {
            int x = xmark( m_ticksX[i] );

            g.drawLine( x, y1 - 2, x, y1 + 2 );
            g.drawString( m_labelX[i],
                x - fm.stringWidth( m_labelX[i] ) / 2, b );
        }
    }

    if( m_axisY ) {
        int b = fm.getAscent() / 2;

        g.drawLine( x1, ymark( m_ticksY[0] ),
            x1, ymark( m_ticksY[m_ticksY.length - 1] ) );

        for( int i = 0; i < m_ticksY.length; ++i ) {
            int y = ymark( m_ticksY[i] );

            g.drawLine( x1 - 2, y, x1 + 2, y );
            g.drawString( m_labelY[i],
                x1 - fm.stringWidth( m_labelY[i] ) - 4, y + b );
        }
    }

    if( m_ticksY[0] < 0 && m_ticksY[m_ticksY.length - 1] > 0 ) {
        int y = ymark( 0 );
        g.drawLine( x1, y, x2, y );

        if( m_axisY ) {
            int h = fm.getHeight();
            int b = fm.getAscent() / 2;

            if( y1 - y > h && y - y2 > h )
                g.drawString( "0", x1 - fm.stringWidth( "0" ) - 4, y + b );
        }
    }

    if( m_assetPrice >= m_ticksX[0] && 
        m_assetPrice <= m_ticksX[m_ticksX.length - 1] ) {
        int x = xmark( m_assetPrice );
        for( int y = y2; y <= y1; y += 2 )
            g.drawLine( x, y, x, y );
    }
}


//
//   p a i n t D a t a
//

private void paintData( Graphics g, double samples[][] )

{
    int x1 = xmark( samples[0][0] );
    int y1 = ymark( samples[0][1] );

    if( samples.length == 1 ) {
        g.drawLine( x1, y1, x1, y1 );
    }
    else {
        for( int i = 1; i < samples.length; ++i ) {
            int x2 = xmark( samples[i][0] );
            int y2 = ymark( samples[i][1] );

            g.drawLine( x1, y1, x2, y2 );

            x1 = x2;
            y1 = y2;
        }
    }
}


//
//   x m a r k
//

private int xmark( double x )

{
    return (int) Math.round( m_originX + x / m_unitX );
}


//
//   y m a r k
//

private int ymark( double y )

{
    return (int) Math.round( m_originY + y / m_unitY );
}

} // end of class
