// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

package nom.rb.MtgClt;
import nom.rb.common.*;

import java.awt.*;


///////////////////////////////////////////////////////////////////

//
//   S c e n a r i o
//

public class Scenario extends ParamSpace {


//
//   v a r i a b l e s
//

    // the following variables are for the GUI        

private boolean m_oneClick;

private int m_numOfSamples;

private ExtTextField m_tfPeriod[];
private ExtTextField m_tfVol[];
private ExtTextField m_tfDriftR[];
private ExtTextField m_tfDriftQ[];

private ExtTextField m_tfSpotPrice;
private ExtTextField m_tfVolBand;

private CheckboxGroup m_grType;
private ExtCheckbox m_cbWorstCase;
private ExtCheckbox m_cbVolShock;

private Label m_lbShockLabel[] = new Label[4];
private ExtTextField m_tfNumOfShocks;
private ExtTextField m_tfShockDuration;
private ExtTextField m_tfShockPeriodicity;

private CheckboxGroup m_grPosition;
private ExtCheckbox m_cbSeller;
private ExtCheckbox m_cbBuyer;
private ExtCheckbox m_cbBoth;

private String[] m_myExamples = {
    "Constant implied volatility and rates in ann. % for the 180 day period",
    "Uncertain volatility and constant rates for the 180 day period",
    "Deterministic volatility term structure, implied for 90 and 180 days",
    "Uncertain volatility term structure, implied for 90 and 180 days",
    "The same volatility term structure, uncertainty entered as band",
    "Volatility data is essential, interest and dividend rates not",
    "Sample dates for volatility and rates can be different",
    "The star '*' is replaced with the maturity of the portfolio"
};

    // these are the parameters for the calculation

private ForwardVol[] m_paramForwardVol;
private ForwardDrift[] m_paramForwardDriftR;
private ForwardDrift[] m_paramForwardDriftQ;

private int m_paramScenario;
private int m_paramNumOfShocks;
private int m_paramShockDuration;
private int m_paramShockPeriodicity;

private boolean m_paramIsUncertain;
private double m_paramSpotPrice;

private boolean m_paramBuyer;
private boolean m_paramSeller;


//
//   S c e n a r i o
//

public Scenario()

{
    this( false );
}


//
//   S c e n a r i o
//

public Scenario( boolean oneClick )

{
    super( Id.SCENARIO );

    m_oneClick = oneClick;

    if( m_oneClick )
        m_numOfSamples = 1;
    else
        m_numOfSamples = 5;

    if( ! m_oneClick )
        m_tfPeriod = new ExtTextField[m_numOfSamples];

    m_tfVol = new ExtTextField[m_numOfSamples];
    m_tfDriftR = new ExtTextField[m_numOfSamples];
    m_tfDriftQ = new ExtTextField[m_numOfSamples];

    GridBagLayout layout = new GridBagLayout();
    setLayout( layout );

    GridBagConstraints c = new GridBagConstraints();

    c.anchor = GridBagConstraints.NORTHWEST;
    if( m_oneClick ) {
        c.gridy = 0;
        add( createTable1(), c );
        c.insets = new Insets( 0, 30, 0, 0 );
        add( createTable2(), c );
    }
    else {
        c.gridx = 0;
        add( createTable1(), c );
        c.insets = new Insets( 30, 0, 0, 0 );
        add( createTable2(), c );
    }

    if( Pool.m_bgColor != null )
        setBackground( Pool.m_bgColor );

    m_examples = m_myExamples;

    clear();                    
}


//
//   c l e a r
//

public void clear()

{
    clearWrong();

    for( int k = 0; k < m_numOfSamples; ++k ) {
        if( ! m_oneClick )
            m_tfPeriod[k].setText( "" );
        m_tfVol[k].setText( "" );
        m_tfDriftR[k].setText( "" );
        m_tfDriftQ[k].setText( "" );
    }

    m_tfVol[0].setText( "15" );
    m_tfSpotPrice.setText( "100" );
    m_grPosition.setCurrent( m_cbSeller );

    if( ! m_oneClick ) {
        m_tfPeriod[0].setText( "*" );
        m_tfVolBand.setText( "" );
        m_grType.setCurrent( m_cbWorstCase );
        m_tfNumOfShocks.setText( "1" );
        m_tfShockDuration.setText( "4" );
        m_tfShockPeriodicity.setText( "2" );
        selectVolShock( false );
    }
}


//
//   e x a m p l e
//

public void example( int num )

{
    if( m_oneClick ) {
    }
    else {
        clear();

        switch( num ) {
            case 1 :
                m_tfPeriod[0].setText( "180" );
                m_tfVol[0].setText( "17" );
                m_tfDriftR[0].setText( "5.5" );
                m_tfDriftQ[0].setText( "3.2" );
                break;

            case 2 :
                m_tfPeriod[0].setText( "180" );
                m_tfVol[0].setText( "12-22" );
                m_tfDriftR[0].setText( "5.5" );
                m_tfDriftQ[0].setText( "3.2" );
                break;

            case 3 :
                m_tfPeriod[0].setText( "90" );
                m_tfVol[0].setText( "15" );
                m_tfPeriod[1].setText( "180" );
                m_tfVol[1].setText( "17" );
                m_tfDriftR[1].setText( "5.5" );
                m_tfDriftQ[1].setText( "3.2" );
                break;

            case 4 :
                m_tfPeriod[0].setText( "90" );
                m_tfVol[0].setText( "10-20" );
                m_tfPeriod[1].setText( "180" );
                m_tfVol[1].setText( "12-22" );
                m_tfDriftR[1].setText( "5.5" );
                m_tfDriftQ[1].setText( "3.2" );
                break;

            case 5 :
                m_tfPeriod[0].setText( "90" );
                m_tfVol[0].setText( "15" );
                m_tfPeriod[1].setText( "180" );
                m_tfVol[1].setText( "17" );
                m_tfDriftR[1].setText( "5.5" );
                m_tfDriftQ[1].setText( "3.2" );
                m_tfVolBand.setText( "5" );
                break;

            case 6 :
                m_tfPeriod[0].setText( "90" );
                m_tfVol[0].setText( "10-20" );
                m_tfPeriod[1].setText( "180" );
                m_tfVol[1].setText( "12-22" );
                break;

            case 7 :
                m_tfPeriod[0].setText( "90" );
                m_tfVol[0].setText( "15" );
                m_tfPeriod[1].setText( "120" );
                m_tfDriftR[1].setText( "5.1" );
                m_tfDriftQ[1].setText( "2.9" );
                m_tfPeriod[2].setText( "180" );
                m_tfVol[2].setText( "17" );
                m_tfDriftR[2].setText( "5.5" );
                m_tfDriftQ[2].setText( "3.2" );
                m_tfVolBand.setText( "+10/-5" );
                break;

            case 8 :
                m_tfPeriod[0].setText( "*" );
                m_tfVol[0].setText( "17" );
                m_tfDriftR[0].setText( "5.5" );
                m_tfDriftQ[0].setText( "3.2" );
                break;
        }
    }
}


//
//   a c t i o n
//

public boolean action( Event evt, Object what )

{
    if( evt.target instanceof ExtCheckbox ) {
        int id = ( (ExtCheckbox) evt.target ).id();

        switch( id ) {
            case Id.WORSTCASE:
            case Id.VOLSHOCK:
                selectVolShock( 
                    ( (ExtCheckbox) m_grType.getCurrent() ).id() ==
                        Id.VOLSHOCK );
                return true;
        }
    }

    return false;
}


//
//   p r e p a r e
//

public void prepare( Portfolio portfolio )
    throws DataException

{
    clearWrong();

    m_paramSpotPrice = getPosDouble( m_tfSpotPrice );

    m_paramSeller = ( m_cbSeller.getState() || m_cbBoth.getState() );
    m_paramBuyer = ( m_cbBuyer.getState() || m_cbBoth.getState() );

    prepareVol( portfolio );
    prepareDrift( portfolio, true );
    prepareDrift( portfolio, false );
    prepareScenario();

    if( ! m_paramIsUncertain && m_paramSeller )
        m_paramBuyer = false;
}


//
//   p a r a m S e l l e r
//

public boolean paramSeller()

{
    return m_paramSeller;
}


//
//   p a r a m B u y e r
//

public boolean paramBuyer()

{
    return m_paramBuyer;
}


//
//   p a r a m I s U n c e r t a i n
//

public boolean paramIsUncertain()

{
    return m_paramIsUncertain;
}


//
//   p a r a m S p o t P r i c e
//

public double paramSpotPrice()

{
    return m_paramSpotPrice;
}


//
//   p a r a m V o l S h o c k
//

public boolean paramVolShock()

{
    return m_paramScenario == 1;
}


//
//   p r o t S c e n a r i o
//

public String protScenario( Advanced advanced )

{
    String s = "factor s {}\n";
    String m = "model M { type bs, s " + m_paramSpotPrice;

    if( m_paramForwardVol != null ) {
        s += "vol V { ";
        for( int k = 0; k < m_paramForwardVol.length; ++k ) {
            if( k > 0 )
                s += ", ";
            s += m_paramForwardVol[k].toString();
        }
        s += " }\n";
        m += ", vol V";
    }

    if( m_paramForwardDriftR != null ) {
        s += "drift R { ";
        for( int k = 0; k < m_paramForwardDriftR.length; ++k ) {
            if( k > 0 )
                s += ", ";
            s += m_paramForwardDriftR[k].toString();
        }
        s += " }\n";
        m += ", discount R";
    }

    if( m_paramForwardDriftQ != null ) {
        s += "drift Q { ";
        for( int k = 0; k < m_paramForwardDriftQ.length; ++k ) {
            if( k > 0 )
                s += ", ";
            s += m_paramForwardDriftQ[k].toString();
        }
        s += " }\n";
        m += ", carry Q";
    }

    s += m + ", " + advanced.protModel() + " }\n";

    String q = "";

    switch( m_paramScenario ) {
        case 0:     
            q = "type worst_case";
            break;

        case 1:
            q = "type shock" +
                ", repetitions " + m_paramNumOfShocks +
                ", duration " + m_paramShockDuration +
                ", periodicity " + m_paramShockPeriodicity;
            break;

        default:
            throw new InternalError();
    }

    if( m_paramSeller )
        s += "scenario SELLER { " + q + ", seller }\n";
    if( m_paramBuyer )
        s += "scenario BUYER { " + q + ", buyer }\n";

    return s;
}


//
//   c o p y F r o m
//

public void copyFrom( Scenario scenario )

{
    int n = m_numOfSamples;

    if( n > scenario.m_numOfSamples )
        n = scenario.m_numOfSamples;

    for( int i = 0; i < n; ++i ) {
        m_tfVol[i].setText( scenario.m_tfVol[i].getText() );
        m_tfDriftR[i].setText( scenario.m_tfDriftR[i].getText() );
        m_tfDriftQ[i].setText( scenario.m_tfDriftQ[i].getText() );
    }

    m_tfSpotPrice.setText( scenario.m_tfSpotPrice.getText() );
    
    switch( ( (ExtCheckbox) scenario.m_grPosition.getCurrent() ).id() ) {
        case Id.SELLER:      m_grPosition.setCurrent( m_cbSeller ); break;
        case Id.BUYER:       m_grPosition.setCurrent( m_cbBuyer );  break; 
        case Id.SELLERBUYER: m_grPosition.setCurrent( m_cbBoth );   break;
    }    
}


//
//   c r e a t e T a b l e 1
//

private Panel createTable1()

{
    Panel table = new Panel();

    GridBagLayout layout = new GridBagLayout();
    table.setLayout( layout );

    GridBagConstraints c = new GridBagConstraints();

    c.anchor = GridBagConstraints.NORTHWEST;
    c.ipadx = 10;
    c.gridx = 0;
    if( ! m_oneClick )
        add( table, new Label( "Period in days" ), c );
    add( table, new Label( "Volatility" ), c );
    add( table, new Label( "Interest rate" ), c );
    add( table, new Label( "Dividend rate" ), c );
    c.insets = new Insets( 10, 0, 0, 0 );
    add( table, new Label( "Spot price" ), c );
    c.insets = new Insets( 0, 0, 0, 0 );
    c.ipadx = 0;

    for( int k = 0; k < m_numOfSamples; ++k ) {
        c.gridx = k + 1;

        if( ! m_oneClick ) {
            m_tfPeriod[k] = new ExtTextField( k, 6 );
            add( table, m_tfPeriod[k], c );
        }

        m_tfVol[k] = new ExtTextField( k, 6 );
        add( table, m_tfVol[k], c );
        m_tfDriftR[k] = new ExtTextField( k, 6 );
        add( table, m_tfDriftR[k], c );
        m_tfDriftQ[k] = new ExtTextField( k, 6 );
        add( table, m_tfDriftQ[k], c );

        if( k == 0 ) {
            c.insets = new Insets( 10, 0, 0, 0 );
            m_tfSpotPrice = new ExtTextField( 0, 6 );
            add( table, m_tfSpotPrice, c );
            c.insets = new Insets( 0, 0, 0, 0 );
        }
    }

    if( ! m_oneClick ) {
        c.gridx = m_numOfSamples + 1;
        c.insets = new Insets( 0, 10, 0, 0 );
        m_tfVolBand = new ExtTextField( 0, 6 );
        add( table, new Label( "Band (+/-)" ), c );
        add( table, m_tfVolBand, c );
    }

    if( Pool.m_bgColor != null )
        table.setBackground( Pool.m_bgColor );

    return table;
}


//
//   c r e a t e T a b l e 2
//

private Panel createTable2()

{
    Panel table = new Panel();

    GridBagLayout layout = new GridBagLayout();
    table.setLayout( layout );

    GridBagConstraints c = new GridBagConstraints();
    c.anchor = GridBagConstraints.NORTHWEST;
    c.ipadx = 10;

    m_grPosition = new CheckboxGroup();

    m_cbSeller = new ExtCheckbox( Id.SELLER,
        "Seller only", m_grPosition, true );
    m_cbBuyer = new ExtCheckbox( Id.BUYER,
        "Buyer only", m_grPosition, false );
    m_cbBoth = new ExtCheckbox( Id.SELLERBUYER,
        "Both seller and buyer", m_grPosition, false );

    if( Pool.m_bgColor != null ) {
        m_cbSeller.setBackground( Pool.m_bgColor );
        m_cbBuyer.setBackground( Pool.m_bgColor );
        m_cbBoth.setBackground( Pool.m_bgColor );
    }

    c.gridx = 0;
    add( table, new Label( "Evaluate position of" ), c );
    add( table, m_cbSeller, c );
    add( table, m_cbBuyer, c );
    add( table, m_cbBoth, c );

    if( ! m_oneClick ) {
        m_grType = new CheckboxGroup();

        m_cbWorstCase = new ExtCheckbox( Id.WORSTCASE,
            "Worst case", m_grType, true );
        m_cbVolShock = new ExtCheckbox( Id.VOLSHOCK,
            "Volatility shock", m_grType, true );

        m_lbShockLabel[0] = new Label( "Volatility shock" );
        m_lbShockLabel[1] = new Label( "Number of shocks" );
        m_lbShockLabel[2] = new Label( "Shock duration in days" );
        m_lbShockLabel[3] = new Label( "Periodicity in days" );
        m_tfNumOfShocks = new ExtTextField( 0, 6 );
        m_tfShockDuration = new ExtTextField( 0, 6 );
        m_tfShockPeriodicity = new ExtTextField( 0, 6 );

        if( Pool.m_bgColor != null ) {
            m_cbWorstCase.setBackground( Pool.m_bgColor );
            m_cbVolShock.setBackground( Pool.m_bgColor );
        }

        c.gridx = 1;    
        add( table, new Label( "Type of scenario" ), c );
        add( table, m_cbWorstCase, c );
        add( table, m_cbVolShock, c );
        c.gridx = 2;
        add( table, m_lbShockLabel[0], c );
        add( table, m_lbShockLabel[1], c );
        add( table, m_lbShockLabel[2], c );
        add( table, m_lbShockLabel[3], c );
        c.gridx = 3;
        c.gridy = 1;
        add( table, m_tfNumOfShocks, c );
        c.gridy = 2;
        add( table, m_tfShockDuration, c );
        c.gridy = 3;
        add( table, m_tfShockPeriodicity, c );
    }

    if( Pool.m_bgColor != null )
        table.setBackground( Pool.m_bgColor );

    return table;
}


//
//   s e l e c t V o l S h o c k
//

private void selectVolShock( boolean select )

{
    if( m_oneClick )
        return;

    if( select ) {
        //for( int i = 0; i < 4; ++i )
            //m_lbShockLabel[i].show();
        m_tfNumOfShocks.enable();
        m_tfShockDuration.enable();
        m_tfShockPeriodicity.enable();
    }
    else {
        //for( int i = 0; i < 4; ++i )
            //m_lbShockLabel[i].hide();
        m_tfNumOfShocks.disable();
        m_tfShockDuration.disable();
        m_tfShockPeriodicity.disable();
    }
    //invalidate();
    //validate();
}


//
//   h a s V a l u e
//

private boolean hasValue( ExtTextField[] text, int col )

{
    if( m_oneClick )
        return text[col].hasValue();
    return ( text[col].hasValue() && m_tfPeriod[col].hasValue() );
}


//
//   p r e p a r e V o l
//

private void prepareVol( Portfolio portfolio )
    throws DataException

{
    double plusBand = 0;
    double minusBand = 0;

    if( ! m_oneClick ) {
        if( m_tfVolBand.hasValue() ) {
            double[] band = getBand( m_tfVolBand );
            plusBand = band[1];
            minusBand = band[0];
        }
    }

    int numOfVols = 0;

    for( int k = 0; k < m_tfVol.length; ++k ) {
        if( hasValue( m_tfVol, k ) )
            ++numOfVols;
    }

    if( numOfVols == 0 ) {
        if( ! m_tfVol[0].hasValue() )
            setWrong( m_tfVol[0] );
        else
            setWrong( m_tfPeriod[0] );
        panic( "Need at least one implied volatility sample" );
    }

    ImpliedVol[] imp = new ImpliedVol[numOfVols];
    m_paramForwardVol = new ForwardVol[numOfVols];

    numOfVols = 0;
    for( int k = 0; k < m_tfVol.length; ++k ) {
        if( hasValue( m_tfVol, k ) ) {
            int maturity;

            if( m_oneClick ) {
                maturity = portfolio.paramMaturity();
            }
            else {
                if( m_tfPeriod[k].getText().trim().startsWith( "*" ) ) {
                    maturity = portfolio.paramMaturity();
                    m_tfPeriod[k].setText( "* (" + maturity + ")" );
                }
                else {
                    maturity = getPosInt( m_tfPeriod[k] );
                }
            }

            double values[] = getPosTriple( m_tfVol[k] );

            int j = numOfVols++;
            while( j > 0 && imp[j - 1].maturity() >= maturity ) {
                if( imp[j - 1].maturity() == maturity ) {
                    setWrong( m_tfPeriod[k] );
                    panic( "Redefinition of period" );
                }
                imp[j] = imp[j - 1];
                --j;
            }
            imp[j] = new ImpliedVol( maturity, values, m_tfVol[k] );
        }
    }

    int k = 0;
    m_paramIsUncertain = false;

    try {
        m_paramForwardVol[0] = new ForwardVol( imp[0], plusBand, minusBand );
        if( m_paramForwardVol[0].isUncertain() )
            m_paramIsUncertain = true;

        for( k = 1; k < numOfVols; ++k ) {
            m_paramForwardVol[k] =
                new ForwardVol( imp[k - 1], imp[k], plusBand, minusBand );
            if( m_paramForwardVol[k].isUncertain() )
                m_paramIsUncertain = true;
        }
    }
    catch( DataException e ) {
        String s = e.toString();
        setWrong( imp[k].text() );
        panic( s.substring( s.indexOf( ":" ) + 1 ).trim() );
    }
}


//
//   p r e p a r e D r i f t
//

private void prepareDrift( Portfolio portfolio, boolean driftR )
    throws DataException

{
    int numOfDrifts = 0;
    ExtTextField[] drift = driftR ? m_tfDriftR : m_tfDriftQ;

    for( int k = 0; k < drift.length; ++k ) {
        if( hasValue( drift, k ) )
            ++numOfDrifts;
    }

    if( numOfDrifts == 0 ) {
        if( driftR )
            m_paramForwardDriftR = null;
        else
            m_paramForwardDriftQ = null;
        return;
    }

    ImpliedDrift[] imp = new ImpliedDrift[numOfDrifts];
    ForwardDrift[] fwd = new ForwardDrift[numOfDrifts];

    numOfDrifts = 0;
    for( int k = 0; k < drift.length; ++k ) {
        if( hasValue( drift, k ) ) {
            int maturity;

            if( m_oneClick ) {
                maturity = portfolio.paramMaturity();
            }
            else {
                if( m_tfPeriod[k].getText().trim().startsWith( "*" ) ) {
                    maturity = portfolio.paramMaturity();
                    m_tfPeriod[k].setText( "* (" + maturity + ")" );
                }
                else {
                    maturity = getPosInt( m_tfPeriod[k] );
                }
            }

            double value = getDouble( drift[k] );

            int j = numOfDrifts++;
            while( j > 0 && imp[j - 1].maturity() >= maturity ) {
                if( imp[j - 1].maturity() == maturity ) {
                    setWrong( m_tfPeriod[k] );
                    panic( "Redefinition of period" );
                }
                imp[j] = imp[j - 1];
                --j;
            }
            imp[j] = new ImpliedDrift( maturity, value, drift[k] );
        }
    }

    int k = 0;

    try {
        fwd[0] = new ForwardDrift( imp[0] );
        for( k = 1; k < numOfDrifts; ++k )
            fwd[k] = new ForwardDrift( imp[k - 1], imp[k] );
    }
    catch( DataException e ) {
        setWrong( imp[k].text() );
        panic( e );
    }

    if( driftR )
        m_paramForwardDriftR = fwd;
    else
        m_paramForwardDriftQ = fwd;
}


//
//   p r e p a r e S c e n a r i o
//

private void prepareScenario()
    throws DataException

{
    if( m_oneClick ) {
        m_paramScenario = 0;
    }
    else {
        if( m_cbWorstCase.getState() )
            m_paramScenario = 0;
        else
        if( m_cbVolShock.getState() )
            m_paramScenario = 1;
        else
            panic( "No scenario selected" );
    }

    if( m_paramScenario == 1 ) {
        m_paramNumOfShocks = getPosInt( m_tfNumOfShocks );
        if( m_paramNumOfShocks > 3 ) {
            setWrong( m_tfNumOfShocks );
            panic( "Too many shocks: the maximum is 3" );
        }

        m_paramShockDuration = getPosInt( m_tfShockDuration );
        m_paramShockPeriodicity = getPosInt( m_tfShockPeriodicity );

        if( ! m_paramIsUncertain ) {
            if( Pool.okCancel( "Warning",
                    "The volatility shock scenario settings\n" +
                    "will have no effect, since no volatility\n" +
                    "band has been entered." ) == MessageBox.ID_CANCEL ) {
                panic();
            }
            m_paramScenario = 0;
        }
    }
}

} // end of class
