// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

package nom.rb.MtgClt;
import nom.rb.common.*;

import java.awt.*;
import java.net.*;
import java.io.*;


///////////////////////////////////////////////////////////////////

//
//   S e r v e r S t u b
//

public class ServerStub extends Object {

//
//   v a r i a b l e s
//

private String m_host = "127.0.0.1";
private int m_port = 1234;

private boolean m_isConnected = false;

private Socket m_socket;

private InputStream m_rawInput;
private DataInputStream m_dataInput;

private OutputStream m_rawOutput;
private DataOutputStream m_dataOutput;

private ServerStubReceiver m_receiver;

private ReportArea m_report;


//
//   S e r v e r S t u b
//

public ServerStub( String host )

{
    this( null, host );
}


//
//   S e r v e r S t u b
//
    
public ServerStub( String host, int port )

{
    this( null, host, port );
}


//
//   S e r v e r S t u b
//

public ServerStub( ReportArea report, String host )

{
    m_report = report;
    m_host = host;
}


//
//   S e r v e r S t u b
//
    
public ServerStub( ReportArea report, String host, int port )

{
    m_report = report;
    m_host = host;
    m_port = port;
}


//
//   c o n n e c t
//

public void connect()
    throws UnknownHostException, IOException

{
    if( ! m_isConnected ) {
        if( m_report != null ) {
            m_report.print(
                "Connecting to " + m_host + " .." );
        }

        try {
            m_socket = new Socket( m_host, m_port );
            m_rawInput = m_socket.getInputStream();
            m_rawOutput = m_socket.getOutputStream();
        }
        catch( IOException e ) {
            if( m_report != null ) 
                m_report.println( " aborted (" + e.getMessage() + ")" );
            throw e;    // pass it on
        }
        catch( SecurityException e ) {
            if( m_report != null ) 
                m_report.println( " not allowed" );
            throw new IOException( "Security violation" );
        }

        m_dataInput = new DataInputStream( m_rawInput );
        m_dataOutput = new DataOutputStream( m_rawOutput );

        if( m_report != null ) 
            m_report.println( " done" );

            // receive in the background

        m_receiver = new ServerStubReceiver( this, m_dataInput );
        m_receiver.start();

        m_isConnected = true;
    }
}


//
//   c o n n e c t U I
//

public boolean connectUI( Frame frame )

{
    try {
        connect();
    }
    catch( UnknownHostException e ) {
        MessageBox.ok( frame, "Error", "Unknown host" );
        return false;
    }
    catch( IOException e ) {
        MessageBox.ok( frame, "Error", "No connection" );
        return false;
    }

    return true;
}


//
//   c l o s e
//

public void close()

{
    if( m_isConnected ) {
        m_isConnected = false;

        synchronized( m_receiver ) {
            m_receiver.stop();
        }

        try {
            m_socket.close();
        }
        catch( IOException e ) {
            // ignore
        }

        m_socket = null;
        m_rawInput = null;
        m_dataInput = null;
        m_rawOutput = null;
        m_dataOutput = null;
        m_receiver = null;

        if( m_report != null ) {
            // m_report.println( "Disconnected from " + m_host );                
        }
    }
}


//
//   r e a d S t r i n g
//

public String readString()
    throws IOException

{
    try {
        return m_receiver.readString();
    }
    catch( IOException e ) {
        if( m_report != null )
            m_report.println( "*** Lost connection" );
        throw e;
    }
}


//
//   w r i t e
//

public void write( Object obj )
    throws IOException

{
    try {
        m_dataOutput.writeBytes( obj.toString() );
    }
    catch( IOException e ) {
        if( m_report != null )
            m_report.println( "*** Lost connection" );
        throw e;
    }
}


//
//   h a n d l e M a c r o
//

public void handleMacro( String macro )

{
    synchronized( m_receiver ) {
            // don't want to kill receiver in the
            // middle of a print statement
        if( m_report != null )
            m_report.print( macro );
    }
}


} // end of class


///////////////////////////////////////////////////////////////////

//
//   S e r v e r S t u b R e c e i v e r
//

class ServerStubReceiver extends Thread {


//
//   v a r i a b l e s
//

private ServerStub m_server;
private DataInputStream m_input;

private String m_ring[] = new String[256];

private int m_nextWrite = 0;
private int m_nextRead = 0;

private Integer m_emptyLock = new Integer( 0 );
private boolean m_empty = true;

private Integer m_fullLock = new Integer( 0 );
private boolean m_full = false;

private Integer m_eofLock = new Integer( 0 );
private boolean m_eof = false;

private long m_timeout = 500;

private Integer m_exceptionLock = new Integer( 0 );
private IOException m_exception;

private static final char m_beginMacro = '#';


//
//   S e r v e r S t u b R e c e i v e r
//

public ServerStubReceiver( ServerStub server, DataInputStream input )

{
    m_server = server;
    m_input = input;
}

    
//
//   r u n
//
    
public void run()

{
        // run() competes with getString()

    while( true ) {
        String s;

        try {
            s = m_input.readLine();
            if( s == null ) {
                synchronized( m_eofLock ) {
                    m_eof = true;
                }
                return;
            }
        }
        catch( IOException e ) {
            synchronized( m_exceptionLock ) {
                m_exception = e;
            }
            return;
        }

        if( ! s.endsWith( "\n" ) )
            s = s + "\n";

        if( ! checkMacro( s ) ) {
            synchronized( m_fullLock ) {
                if( m_full ) {
                    try { 
                        m_fullLock.wait();
                    }
                    catch( InterruptedException e ) {
                        return;
                    }
                }
            }

            synchronized( m_ring ) {
                m_ring[m_nextWrite++] = s;
                if( m_nextWrite == m_ring.length )
                    m_nextWrite = 0;

                if( ( m_nextWrite + 1 ) % m_ring.length == m_nextRead )
                    m_full = true;

                if( m_empty )
                    setEmpty( false );
            }
        }
    }
}


//
//   r e a d S t r i n g
//

public String readString()
    throws IOException

{
        // getString() competes with run()

    String s = null;

    synchronized( m_emptyLock ) {
        if( m_empty ) {
            checkException();

            synchronized( m_eofLock ) {
                if( m_eof )
                    throw new IOException();
            }
            try {
                m_emptyLock.wait( m_timeout );
            }
            catch( InterruptedException e ) {
                throw new InternalError();
            }
            if( m_empty )
                return null;
        }
    }
                
    synchronized( m_ring ) {
        s = m_ring[m_nextRead++];

        if( m_nextRead == m_ring.length )
            m_nextRead = 0;

        if( m_nextRead == m_nextWrite )
            m_empty = true;

        if( m_full )
            setFull( false );
    }

    return s;
}


//
//   s e t E m p t y
//

private void setEmpty( boolean empty )

{
    synchronized( m_emptyLock ) {
        m_empty = empty;
        if( ! m_empty )
            m_emptyLock.notify();
    }
}


//
//   s e t E m p t y
//

private void setFull( boolean full )

{
    synchronized( m_fullLock ) {
        m_full = full;
        if( ! m_full )
            m_fullLock.notify();
    }
}


//
//   c h e c k M a c r o
//

private boolean checkMacro( String s )

{
    if( s != null && s.length() > 0 && s.charAt( 0 ) == m_beginMacro ) {
        m_server.handleMacro( s.substring( 1 ) );
        return true;
    }
    return false;
}


//
//   c h e c k E x c e p t i o n
//

private void checkException()
    throws IOException

{
    synchronized( m_exceptionLock ) {
        if( m_exception != null ) {
            IOException e = m_exception;
            m_exception = null;
            throw e;
        }
    }
}

} // end of class
