// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

package nom.rb.common;

import java.net.*;
import java.io.*;
import java.util.*;


///////////////////////////////////////////////////////////////////

//
//   H t t p D e t o u r
//

public class HttpDetour {


//
//   v a r i a b l e s
//

HttpDetourInputStream m_input;
HttpDetourOutputStream m_output;
HttpDetourMonitor m_monitor;


//
//   H t t p D e t o u r
//
   
public HttpDetour( String serverHost, int serverPort,
    String proxyHost, int proxyPort )

{
    m_monitor = new HttpDetourMonitor( serverHost, serverPort,
        proxyHost, proxyPort );

    m_input = new HttpDetourInputStream( m_monitor );
    m_output = new HttpDetourOutputStream( m_monitor );

    m_monitor.setInput( m_input );
    m_monitor.setOutput( m_output );

    m_monitor.start();
}


//
//   g e t I n p u t S t r e a m
//

public InputStream getInputStream()
    throws IOException

{
    m_monitor.checkException();
    return m_input;
}


//
//   g e t O u t p u t S t r e a m
//

public OutputStream getOutputStream()
    throws IOException

{
    m_monitor.checkException();
    return m_output;
}


//
//   c l o s e
//

public void close()
    throws IOException

{
    m_monitor.stop();
    m_input.close();
    m_output.close();
}

} // end of class


///////////////////////////////////////////////////////////////////

//
//   H t t p D e t o u r M o n i t o r
//

class HttpDetourMonitor extends Thread {

    
//
//   v a r i a b l e s
//

private int m_tag = ( new Random() ).nextInt();
private int m_seq = 0;

private String m_serverHost;
private int m_serverPort;
private String m_proxyHost;
private int m_proxyPort;

private Socket m_socket;

private InputStream m_rawInput;
private DataInputStream m_dataInput;
private HttpDetourInputStream m_detourInput;

private OutputStream m_rawOutput;
private DataOutputStream m_dataOutput;
private HttpDetourOutputStream m_detourOutput;

private boolean m_inClosed = false;
private boolean m_outClosed = false;

    // Any exception is stored temporarily here, because we
    // have nowhere to throw it to. In m_detourInput resp.
    // m_detoutOutput, the exception is retrieved and
    // rethrown (it makes sense there).

private Object m_exceptionLock = new Object();
private IOException m_exception;


//
//   H t t p D e t o u r M o n i t o r
//

public HttpDetourMonitor( String serverHost, int serverPort,
    String proxyHost, int proxyPort )

{
    m_serverHost = serverHost;
    m_serverPort = serverPort;
    m_proxyHost = proxyHost;
    m_proxyPort = proxyPort;
}


//
//   r u n
//
    
public void run()

{
    while( connect() ) {
        byte[] content = null;

        if( ! m_outClosed ) {
                // Get message for server, if any.
                // Wait a bit after connection.
            try {
                content = m_detourOutput.getOutput( 10 );
            }
            catch( EOFException e ) {
                m_outClosed = true;

                if( m_inClosed ) {
                    setException( e );
                    close();
                    return;
                }
                    // Don't pass exception on, output stream
                    // has been closed separately, but input
                    // stream is still open. Ignore.
                content = null;
            }
        }

        int l = ( content == null ) ? 0 : content.length;

        String request =
            "POST http://" + m_serverHost + ":" + m_serverPort +
            "/" + m_tag + "/" + m_seq++ + " HTTP/1.0\n" +
            "Content-Length: " + l + "\n\n";

        try {
            m_dataOutput.writeBytes( request );
            if( l > 0 )
                m_dataOutput.write( content, 0, l );
        }
        catch( IOException e ) {
            setException( e );
            close();
            return;
        }

        if( ! response() ) {
            close();
            return;
        }

        if( ! close() )
            return;
    }
}


//
//   s e t I n p u t
//

protected void setInput( HttpDetourInputStream detourInput )

{
    m_detourInput = detourInput;
}


//
//   s e t O u t p u t
//

protected void setOutput( HttpDetourOutputStream detourOutput )

{
    m_detourOutput = detourOutput;
}


//
//   c l o s e
//

private boolean close()

{
    try {
        m_socket.close();
    }
    catch( IOException e ) {
        setException( e );
        return false;
    }
    return true;
}


//
//   c o n n e c t
//

private boolean connect()

{
    try {
        m_socket = new Socket( m_proxyHost, m_proxyPort );
        m_rawInput = m_socket.getInputStream();
        m_rawOutput = m_socket.getOutputStream();
    }
    catch( IOException e ) {
        setException( e );
        return false;
    }
    catch( SecurityException e ) {
        setException( new IOException( "Security violation" ) );
        return false;
    }

    m_dataInput = new DataInputStream( m_rawInput );
    m_dataOutput = new DataOutputStream( m_rawOutput );

    return true;
}


//
//   r e s p o n s e
//

private boolean response()

{
    if( ! readStatus() )
        return false;

    int contentLength = -1;
    int port = -1;
    String s;

    do{
        s = readLine();
        if( s == null ) {
            setException( new EOFException() );
            return false;
        }

        if( s.length() > 0 ) {
            int k = s.indexOf( ':' );

            if( k > 0 ) {
                String h = s.substring( 0, k ).trim();
                String t = s.substring( k + 1 ).trim();

                if( h.equalsIgnoreCase( "content-length" ) ) {
                    if( contentLength >= 0 ) {
                        setException( new IOException( "Bad format" ) );
                        return false;
                    }

                    try {
                        contentLength = Integer.valueOf( t ).intValue();
                    }
                    catch( NumberFormatException e ) {
                        setException( new IOException( "Bad format" ) );
                        return false;
                    }
                }
                else
                if( h.equalsIgnoreCase( "pragma" ) ) {
                    k = t.indexOf( '=' );

                    if( k > 0 ) {
                        h = s.substring( 0, k ).trim();
                        t = s.substring( k + 1 ).trim();

                        if( h.equalsIgnoreCase( "port" ) ) {
                            if( port >= 0 ) {
                                setException( new IOException( "Bad format" ) );
                                return false;
                            }

                            try {
                                port = Integer.valueOf( t ).intValue();
                            }
                            catch( NumberFormatException e ) {
                                setException( new IOException( "Bad format" ) );
                                return false;
                            }
                        }
                    }
                }
            }
        }
    } while( s.length() > 0 );

    if( port >= 0 )
        m_serverPort = port;

    return readContent( contentLength );
}


//
//   r e a d S t a t u s
//

private boolean readStatus()

{
    String s = readLine();
    if( s == null ) {
        setException( new EOFException() );
        return false;
    } 

    StringTokenizer st = new StringTokenizer( s );
    if( ! st.hasMoreTokens() || ! st.nextToken().equals( "HTTP/1.0" ) ) {
        setException( new IOException( "Bad format" ) );
        return false;
    }

    if( ! st.hasMoreTokens() || ! st.nextToken().equals( "200" ) ) {
        setException( new IOException( "Service unavailable" ) );
        return false;
    }

    return true;
}


//
//   r e a d C o n t e n t
//

private boolean readContent( int contentLength )

{
    if( contentLength > 0 ) {
        byte[] buffer = new byte[contentLength];

        try {
            m_dataInput.readFully( buffer );
        }
        catch( IOException e ) {
            setException( e );
            return false;
        }

        if( m_inClosed ) {
            if( m_outClosed ) {
                setException( new EOFException() );
                return false;
            }
        }
        else {
            if( ! m_detourInput.setInput( buffer ) ) {
                m_inClosed = true;
                if( m_outClosed ) {
                    setException( new EOFException() );
                    return false;
                }
            }
        }
    }

    return true;
}


//
//   r e a d L i n e
//

private String readLine()

{
    try {
        return m_dataInput.readLine();
    }
    catch( IOException e ) {
        setException( e );
    }

    return null;
}


//
//   s e t E x c e p t i o n
//

private void setException( IOException e )

{
    synchronized( m_exceptionLock ) {
        if( m_exception != null )
            m_exception = e;
    }
}


//
//   c h e c k E x c e p t i o n
//

protected void checkException()
    throws IOException

{
    synchronized( m_exceptionLock ) {
        if( m_exception != null ) {
            IOException e = m_exception;
            m_exception = null;
            throw e;
        }
    }
}

} // end of class


///////////////////////////////////////////////////////////////////

//
//   H t t p D e t o u r I n p u t S t r e a m
//

class HttpDetourInputStream extends InputStream {


//
//   v a r i a b l e s
//

private byte[] m_input = new byte[1024];
private int m_nextRead = 0;
private int m_nextWrite = 0;

private boolean m_open = true;

private HttpDetourMonitor m_monitor;


//
//   H t t p D e t o u r I n p u t S t r e a m
//

public HttpDetourInputStream( HttpDetourMonitor monitor )

{
    m_monitor = monitor;
}


//
//   a v a i l a b l e
//
    
public synchronized int available()

{
    if( ! m_open )
        return 0;

    if( m_nextRead == m_nextWrite )
        return 0;
    if( m_nextRead < m_nextWrite )
        return m_nextWrite - m_nextRead;
    return m_input.length + m_nextWrite - m_nextRead;
}


//
//   c l o s e
//
  
public synchronized void close()
    throws IOException

{
    if( m_open ) {
        m_open = false;
        notifyAll();
    }
}


//
//   r e a d
//

public synchronized int read()
    throws IOException

{
    waitForInput();

    boolean full = isFull();
    int c = (int) m_input[m_nextRead];

    if( ++m_nextRead == m_input.length )
        m_nextRead = 0;

    if( full )
        notify();

    return c;
}


//
//   r e a d
//

public synchronized int read( byte[] b, int off, int len )
    throws IOException

{
    waitForInput();

    boolean full = isFull();

    int k = off;
    int l = 0;

    while( m_nextRead != m_nextWrite && l < len ) {
        b[k++] = m_input[m_nextRead];
        if( ++m_nextRead == m_input.length )
            m_nextRead = 0;
        ++l;            
    }

    if( full && l > 0 )
        notify();

    return l;
}


//
//   s e t I n p u t
//

protected synchronized boolean setInput( byte[] buffer )

{
    if( buffer != null && buffer.length > 0 ) {
        for( int k = 0; k < buffer.length; ++k ) {
            if( isFull() ) {
                    // somebody should read some bytes:
                notify();
                    // now wait for notification
                try {
                    wait();
                }
                catch( InterruptedException e ) {
                    return false;
                }

                if( ! m_open || isFull() )
                    return false;   // input has been closed
            }
            m_input[m_nextWrite] = buffer[k];
            if( m_nextWrite++ == m_input.length )
                m_nextWrite = 0;
        }
            // notify again that there is stuff in the buffer
        notify();
    }

    return true;
}


//
//   w a i t F o r I n p u t
//

private void waitForInput()
    throws IOException

{
    m_monitor.checkException();

    if( ! m_open )
        throw new EOFException();

    if( m_nextRead == m_nextWrite ) {
        try {
            wait();
        }
        catch( InterruptedException e ) {
            throw new EOFException();
        }
    }

    if( ! m_open || m_nextRead == m_nextWrite )
        throw new EOFException();
}


//
//   i s F u l l
//

private boolean isFull()

{
    if( m_nextRead == 0 )
        return m_nextWrite == m_input.length - 1;
    return m_nextWrite + 1 == m_nextRead;
}

} // end of class


///////////////////////////////////////////////////////////////////

//
//   H t t p D e t o u r O u t p u t S t r e a m
//

class HttpDetourOutputStream extends OutputStream {

    
//
//   v a r i a b l e s
//

private Vector m_output = new Vector();

private boolean m_open = true;

private HttpDetourMonitor m_monitor;


//
//   H t t p D e t o u r O u t p u t S t r e a m
//

public HttpDetourOutputStream( HttpDetourMonitor monitor )

{
    m_monitor = monitor;
}


//
//   c l o s e
//
    
public synchronized void close()
    throws IOException

{
    if( m_open ) {
        m_open = false;
        notifyAll();
    }
}


//
//   f l u s h
//

public synchronized void flush()
    throws IOException

{
    if( m_open )
        throw new EOFException();
}


//
//   w r i t e
//

public synchronized void write( int b )
    throws IOException

{
    m_monitor.checkException();

    if( ! m_open )
        throw new EOFException();

    boolean empty = isEmpty();

    byte[] B = new byte[1];
    B[0] = (byte) b;
    m_output.addElement( B );

    if( empty )
        notify();
}


//
//   w r i t e
//

public synchronized void write( byte[] b, int off, int len )
    throws IOException

{
    m_monitor.checkException();

    if( ! m_open )
        throw new EOFException();

    boolean empty = isEmpty();

    if( len > 0 ) {
        if( off == 0 && len == b.length ) {
            m_output.addElement( b );
        }
        else {
            byte[] B = new byte[len];
            System.arraycopy( b, off, B, 0, len );
            m_output.addElement( B );
        }

        if( empty )
            notify();
    }
}


//
//   g e t O u t p u t
//

protected synchronized byte[] getOutput( int timeout )
    throws EOFException

{
    if( isEmpty() ) {
        if( ! m_open )
            throw new EOFException();

        try {
            wait( timeout * 1000 );
        }
        catch( InterruptedException e ) {
            throw new EOFException();
        }

        if( ! m_open )
            throw new EOFException();
        if( isEmpty() )
            return null;
    }

    int l = 0;
    for( int k = 0; k < m_output.size(); ++k )
        l += ( (byte[]) m_output.elementAt( k ) ).length;

    byte[] b = new byte[l];

    l = 0;
    for( int k = 0; k < m_output.size(); ++k ) {
        byte[] bb = (byte[]) m_output.elementAt( k );
        System.arraycopy( bb, 0, b, l, bb.length );
        l += bb.length;
    }

    m_output.removeAllElements();
    return b;
}


//
//   i s E m p t y
//

private boolean isEmpty()

{
    return m_output.isEmpty();
}

} // end of class
