// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#if ! defined(_MTG_ARRAY_BOUNDS_)
#define _MTG_ARRAY_BOUNDS_

#include "MtgHeap.h"

MTG_BEGIN_NAMESPACE


//
//   t A r r a y B o u n d s
//

class tArrayBounds {

    struct tDimension {
        int m_nFrom;
        int m_nTo;
        int m_nSize;
    };

    tHeap<tDimension> m_Dim;

    int m_nOfs;

    void init();

    int partialPos( int nSamples, ... ) const;
    void reduceDim( int nShift );

public:

    tArrayBounds();

    tArrayBounds( const tArrayBounds& Bounds );
    tArrayBounds( const tArrayBounds& Bounds, int x1 );
    tArrayBounds( const tArrayBounds& Bounds, int x1, int x2 );
    tArrayBounds( const tArrayBounds& Bounds, int x1, int x2, int x3 );

    tArrayBounds& operator=( const tArrayBounds& Bounds );

    void reset();

    void copyFrom( const tArrayBounds& Bounds );

    tArrayBounds* clone() const;

    void append( int nFrom, int nTo );
    void append( const tArrayBounds& Bounds );

    int dimension() const {
        return m_Dim.numOfElems();
    }

    int from( int nDim ) const {
        return m_Dim[nDim].m_nFrom;
    }

    int to( int nDim ) const {
        return m_Dim[nDim].m_nTo;
    }

    int size( int nDim ) const {
        return m_Dim[nDim].m_nSize;
    }

    int rowSize() const {
        return m_Dim.last().m_nSize;
    }

    void reduce( int x1 );
    void reduce( int x1, int x2 );
    void reduce( int x1, int x2, int x3 );

#if defined(_DEBUG)

    int pos( int x1 ) const;
    int pos( int x1, int x2 ) const;
    int pos( int x1, int x2, int x3 ) const;

    int row( int x1 ) const;
    int row( int x1, int x2 ) const;
    int row( int x1, int x2, int x3 ) const;

#else

    int pos( int x1 ) const {
        return x1 + m_nOfs;
    }

    int pos( int x1, int x2 ) const {
        return x1 * m_Dim[1].m_nSize + x2 + m_nOfs;
    }

    int pos( int x1, int x2, int x3 ) const {
        return ( x1 * m_Dim[1].m_nSize + x2 ) * m_Dim[2].m_nSize + 
                 x3 + m_nOfs;
    }

    int row( int x1 ) const {
        return pos( x1, m_Dim[1].m_nFrom );
    }

    int row( int x1, int x2 ) const {
        return pos( x1, x2, m_Dim[2].m_nFrom );
    }

    int row( int x1, int x2, int x3 ) const {
        return pos( x1, x2, x3, m_Dim[3].m_nFrom );
    }

#endif

    int pos( int x1, int x2, int x3, int x4, va_list Marker ) const;
    int pos( int x1, int x2, int x3, int x4, ... ) const;
    int pos( const tHeap<int>& x ) const;
    int pos( const int x[] ) const;
    int pos( int x1, const tHeap<int>& x ) const;

    int row( int x1, int x2, int x3, int x4, va_list Marker ) const;
    int row( int x1, int x2, int x3, int x4, ... ) const;
    int row( const tHeap<int>& x ) const;
    int row( const int x[] ) const;

#if defined(_MTG_WITH_MATHLINK)
    tRetCode toMathLink( MLINK MathLink ) const;
#endif
};

MTG_END_NAMESPACE

#endif
