// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#include "MtgIncl.h"
#include "MtgBSModel.h"
#include "MtgGeoEngine.h"
#include "MtgGeoSpaceAxis.h"
#include "MtgFactor.h"
#include "MtgParser.h"
#include "MtgShockEngine.h"
#include "MtgShockScenario.h"
#include "MtgStepDrift.h"
#include "MtgVol.h"
#include "MtgWorstCase.h"

MTG_BEGIN_NAMESPACE


//
//   i n i t
//

void tBSModel::init()

{
    m_pDiscount = 0;
    m_pCarry = 0;
    m_pMu = 0;
    m_pVol = 0;
    m_gRoot = 0;
    m_nScale = xYear;
}


//
//   c o p y F r o m
//

void tBSModel::copyFrom( const tBSModel& Model )

{
    if( &Model == this )
        return;

    if( m_pMu != 0 ) {
        delete m_pMu;
        m_pMu = 0;
    }

    if( Model.m_pMu != 0 )
        m_pMu = static_cast<tDrift*>( Model.m_pMu->clone() );

    setObjRefToZeroOrSo( m_pDiscount, Model.m_pDiscount );
    setObjRefToZeroOrSo( m_pCarry, Model.m_pCarry );
    setObjRefToZeroOrSo( m_pVol, Model.m_pVol );

    m_gRoot = Model.m_gRoot;

    super::copyFrom( Model );
}


//
//   p a r s e P a r a m
//

tRetCode tBSModel::parseParam( tParser& Parser, tParseInfoStub& Info )

{
    tRetCode nRet;
    tObject* pObj;
    tDrift* pDrift;
    tVol* pVol;
    tFactor* pFactor;
    double gRoot;

    switch( Parser.curToken() ) {
        case xTokDiscount :
        case xTokDomestic :
            if( m_pDiscount != 0 ) {
                return Parser.setError( "discount/domestic drift "
                                        "already defined" );
            }
            if( ( nRet = Parser.readToken() ) != OK )
                return nRet;
            if( Parser.curToken() != xTokId )
                return Parser.setError( "drift id expected" );
            if( ( pObj = Parser.findObject() ) == 0 )
                return Parser.setError( "drift object does not exist" );
            if( ( pDrift = dynamic_cast<tDrift*>( pObj ) ) == 0 ) 
                return Parser.setError( "drift object expected" );
            if( pDrift->scale() != xDay )
                return Parser.setError( "scale of drift object must be day" );
            setObjRefToZeroOrSo( m_pDiscount, pDrift );
            if( ( nRet = Parser.readToken() ) != OK )
                return nRet;
            break;

        case xTokForeign :
        case xTokCarry :
            if( m_pCarry != 0 ) {
                return Parser.setError( "foreign/carry drift "
                                        "already defined" );
            }
            if( ( nRet = Parser.readToken() ) != OK )
                return nRet;
            if( Parser.curToken() != xTokId )
                return Parser.setError( "drift id expected" );
            if( ( pObj = Parser.findObject() ) == 0 )
                return Parser.setError( "drift object does not exist" );
            if( ( pDrift = dynamic_cast<tDrift*>( pObj ) ) == 0 ) 
                return Parser.setError( "drift object expected" );
            if( pDrift->scale() != xDay )
                return Parser.setError( "scale of drift object must be day" );
            setObjRefToZeroOrSo( m_pCarry, pDrift );
            if( ( nRet = Parser.readToken() ) != OK )
                return nRet;
            break;

        case xTokVol :
            if( m_pVol != 0 )
                return Parser.setError( "volatility already defined" );
            if( ( nRet = Parser.readToken() ) != OK )
                return nRet;
            if( Parser.curToken() != xTokId )
                return Parser.setError( "vol id expected" );
            if( ( pObj = Parser.findObject() ) == 0 )
                return Parser.setError( "vol object does not exist" );
            if( ( pVol = dynamic_cast<tVol*>( pObj ) ) == 0 ) 
                return Parser.setError( "vol object expected" );
            setObjRefToZeroOrSo( m_pVol, pVol );
            if( ( nRet = Parser.readToken() ) != OK )
                return nRet;
            break;

        case xTokFactor :
        case xTokSpot :
            if( ( nRet = Parser.readToken() ) != OK )
                return nRet;
            if( Parser.curToken() != xTokId ) 
                return Parser.setError( INVALID_KEYWORD );

                // fall through

        case xTokId :
            if( numOfFactors() > 0 )
                return Parser.setError( "factor already defined" );
            if( ( pObj = Parser.findObject() ) == 0 )
                return Parser.setError( "factor does not exist" );
            if( ( pFactor = dynamic_cast<tFactor*>( pObj ) ) == 0 ) 
                return Parser.setError( "factor expected" );
            if( ( nRet = Parser.readToken() ) != OK ||
                ( nRet = Parser.scanPosDouble( gRoot ) ) != OK ) {
                return nRet;
            }
            appendFactor( pFactor );
            m_gRoot = gRoot;
            break;

        default :
            return super::parseParam( Parser, Info );
    }

    return OK;
}


//
//   p a r s e P o s t f i x
//

tRetCode tBSModel::parsePostfix( tParser& Parser, tParseInfoStub& Info )

{
    if( m_pVol == 0 )
        return Parser.setError( MISSING_VOLATILITY );
    if( numOfFactors() == 0 )
        return Parser.setError( MISSING_FACTOR );
    return super::parsePostfix( Parser, Info );
}


//
//   t B S M o d e l
//

tBSModel::tBSModel()

{
    init();
}


//
//   t B S M o d e l
//

tBSModel::tBSModel( const tBSModel& Model )

{
    init();
    copyFrom( Model );
}


//
//   ~ t B S M o d e l
//

tBSModel::~tBSModel()

{
    if( m_pMu != 0 ) {
        delete m_pMu;
        m_pMu = 0;
    }

    setObjRefToZeroOrSo( m_pDiscount );
    setObjRefToZeroOrSo( m_pCarry );
    setObjRefToZeroOrSo( m_pVol );
}


//
//   o p e r a t o r =
//

tBSModel& tBSModel::operator=( const tBSModel& Model )

{
    if( &Model != this )
        copyFrom( Model );
    return *this;
}


//
//   c l o n e
//

tObject* tBSModel::clone() const

{
    return new tBSModel( *this );
}


//
//   f i n a l i z e
//

tRetCode tBSModel::finalize()

{
    MTG_ASSERT( m_pVol != 0 && m_gRoot > 0 );

    tRetCode nRet;

    if( isFinalized() )
        return OK;

    if( m_pMu != 0 ) {
        delete m_pMu;
        m_pMu = 0;
    }

    if( m_pDiscount != 0 ) {
        m_pMu = static_cast<tDrift*>( m_pDiscount->clone() );
    }
    else {
        tStepDrift* p = new tStepDrift;

        if( ( nRet = p->finalize() ) != OK ) {
            delete p;
            return nRet;
        }
        m_pMu = p;
    }

    if( m_pCarry != 0 ) {
        if( ( nRet = ( *m_pMu -= *m_pCarry ) ) != OK ) {
            delete m_pMu;
            m_pMu = 0;
            return nRet;
        }
    }

    return super::finalize();
}


//
//   g e t D r i f t
//

void tBSModel::getDrift( int nFromDay, int nToDay, double& gDiscount,
    double& gMu ) const

{
        // The forward() function excludes the last day. The paramter
        // nToday is inclusive, however.

    if( m_pDiscount != 0 )
        gDiscount = m_pDiscount->forward( nFromDay, nToDay + 1 );
    else
        gDiscount = 0;

    if( m_pMu != 0 )
        gMu = m_pMu->forward( nFromDay, nToDay + 1 );
    else
        gMu = 0;
}
 
    
//
//   g e t V o l
//
    
void tBSModel::getVol( int nFromDay, int nToDay, double& gPrior,
    double& gMin, double& gMax ) const

{
    MTG_ASSERT( m_pVol != 0 );
    m_pVol->forward( nFromDay, nToDay + 1, gPrior, gMin, gMax );
}


//
//   i s L i n e a r
//

bool tBSModel::isLinear( int nDay ) const

{
    MTG_ASSERT( m_pVol != 0 );
    
    if( m_pVol->certainAfter() <= nDay )
        return true;
    return false;
}


//
//   c r e a t e S p a c e A x i s
//

tRetCode tBSModel::createSpaceAxis( tFDMethod nMethod, double gMaxDt,
    tHeap<tSpaceAxis*>& Space, const tPortfolio* pPf )

{
    MTG_ASSERT( isFinalized() );
    MTG_ASSERT( Space.numOfElems() == 0 );

    double gMinVol, gMaxVol, gMinMu, gMaxMu, gPassMaxDt,
        gDayUnit, gDayUnit2;
    tRetCode nRet;

    m_pVol->getFwdRange( gMinVol, gMaxVol );
    m_pMu->getFwdRange( gMinMu, gMaxMu );

    gDayUnit = dayUnit();
    gDayUnit2 = sqrt( gDayUnit );

    switch( m_nScale ) {
        case xDay : 
                // Scale parameters down to the base unit "day."
            gMinMu *= gDayUnit;
            gMaxMu *= gDayUnit;
            gMinVol *= gDayUnit2;
            gMaxVol *= gDayUnit2;

            gPassMaxDt = gMaxDt;
            break;

        case xYear :
                // gMaxDt is no longer in terms of days, but of years.
            gPassMaxDt = gMaxDt * gDayUnit; 
            break;

        default :
            MTG_ASSERT( false );
    }
    
    tGeoSpaceAxis* p = new tGeoSpaceAxis;

    if( pPf != 0 ) {
        tHeap<double> AlignAt;

        pPf->getBarriers( factor( 0 ), true, AlignAt );
        nRet = p->prepare( nMethod, m_gRoot, gMinVol, gMaxVol,
                    gMinMu, gMaxMu, gPassMaxDt, AlignAt );
    }
    else {
        nRet = p->prepare( nMethod, m_gRoot, gMinVol, gMaxVol,
                    gMinMu, gMaxMu, gPassMaxDt );
    }

    if( nRet != OK ) {
        delete p;
        return nRet;
    }

    if( m_nScale == xYear )
        p->adjustStableDt( gPassMaxDt, gMaxDt, gDayUnit );

    ++Space;
    Space.last() = p;
    return OK;
}


//
//   c r e a t e E n g i n e
//

tRetCode tBSModel::createEngine( const tScenario* pScenario,
    tFDEngine*& pEngine, tAccuracy nAccuracy )

{
    MTG_ASSERT( isFinalized() );
    MTG_ASSERT( pScenario != 0 );

    if( dynamic_cast<const tShockScenario*>( pScenario ) != 0 ) {
        pEngine = new tShockEngine;
    }
    else
    if( dynamic_cast<const tWorstCase*>( pScenario ) != 0 ) {
        pEngine = new tGeoEngine;
    }
    else {
        return NOT_AVAILABLE;
    }

    pEngine->setAccuracy( nAccuracy );
    return OK;
}

MTG_END_NAMESPACE
