// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#include "MtgIncl.h"
#include "MtgBondMath.h"

MTG_BEGIN_NAMESPACE


//
//   i n i t
//

void tBondMath::init()

{
        // The defaults are essentially those of the US market:

    m_pSettlement = new tFollowingDayShift( 1 );

    m_Discount.set( tInterest::xLinear, tInterest::xDiscount,
        tDayCount( tDayCount::xACT_360 ), tPeriodCompounder( 1 ) );
    m_MMYield.set( tInterest::xLinear, tInterest::xYield,
        tDayCount( tDayCount::xACT_360 ), tPeriodCompounder( 1 ) );
    m_BEYield.set( tInterest::xLinear, tInterest::xYield,
        tDayCount( tDayCount::xACT_365 ), tPeriodCompounder( 2 ) );

    m_BondEquation.set( tBondEquation::xStandard, DayCountACT_ACT ); 
}


//
//   c o p y F r o m
//

void tBondMath::copyFrom( const tBondMath& Math )

{
    if( &Math == this )
        return;

    if( m_pSettlement != 0 ) {
        delete m_pSettlement;
        m_pSettlement = 0;
    }

    if( Math.m_pSettlement != 0 )
        m_pSettlement = Math.m_pSettlement->clone();

    m_Discount = Math.m_Discount;
    m_MMYield = Math.m_MMYield;
    m_BEYield = Math.m_BEYield;
    m_BondEquation = Math.m_BondEquation;
}


//
//   s e t S e t t l e m e n t
//

void tBondMath::setSettlement( const tDayShift& Settlement )

{
    if( m_pSettlement != 0 ) {
        delete m_pSettlement;
        m_pSettlement = 0;
    }

    m_pSettlement = Settlement.clone();
}


//
//   s e t D i s c o u n t
//

void tBondMath::setDiscount( const tInterest& Discount )

{
    m_Discount = Discount;
}


//
//   s e t M M Y i e l d
//

void tBondMath::setMMYield( const tInterest& MMYield )

{
    m_MMYield = MMYield;
}


//
//   s e t B E Y i e l d
//

void tBondMath::setBEYield( const tInterest& BEYield )

{
    m_BEYield = BEYield;
}


//
//   s e t B o n d E q u a t i o n
//

void tBondMath::setBondEquation( const tBondEquation& BondEquation )

{
    m_BondEquation = BondEquation;
}


//
//   b o n d P r i c e
//

double tBondMath::bondPrice( tDate Settlement, tDate Maturity,
    double gCoupon, double gYield, bool bClean )

{
    tCompounder* pCompounder = createCompounder( Settlement, Maturity );
    MTG_ASSERT( pCompounder != 0 );

    double gPrice;
    tDate M;

    if( dynamic_cast<tCouponCompounder*>( pCompounder ) == 0 )
        M = Maturity;
    else
        M = dynamic_cast<tCouponCompounder*>( pCompounder )->maturity();

    if( bClean ) {
        gPrice = m_BondEquation.cleanPrice( Settlement, M, 
            gCoupon, gYield, pCompounder );
    }
    else {
        gPrice = m_BondEquation.dirtyPrice( Settlement, M,
            gCoupon, gYield, pCompounder );
    }

    delete pCompounder;
    return gPrice;
}


//
//   b o n d Y i e l d
//

double tBondMath::bondYield( tDate Settlement, tDate Maturity,
    double gCoupon, double gPrice, bool bClean )

{
    tCompounder* pCompounder = createCompounder( Settlement, Maturity );
    MTG_ASSERT( pCompounder != 0 );

    double gYield;
    tDate M;

    if( dynamic_cast<tCouponCompounder*>( pCompounder ) == 0 )
        M = Maturity;
    else
        M = dynamic_cast<tCouponCompounder*>( pCompounder )->maturity();

    if( bClean ) {
        gYield = m_BondEquation.clean2Yield( Settlement, M, 
            gCoupon, gPrice, pCompounder );
    }
    else {
        gYield = m_BondEquation.dirty2Yield( Settlement, M,
            gCoupon, gPrice, pCompounder );
    }

    delete pCompounder;
    return gYield;
}


//
//   b o n d P r i c e
//

double tBondMath::bondPrice( tDate Settlement, tDate Maturity,
    double gCoupon, tInterestSpline& Spline, bool bClean )

{
    tCompounder* pCompounder = createCompounder( Settlement, Maturity );
    MTG_ASSERT( pCompounder != 0 );

    tDate M;

    if( dynamic_cast<tCouponCompounder*>( pCompounder ) == 0 )
        M = Maturity;
    else
        M = dynamic_cast<tCouponCompounder*>( pCompounder )->maturity();

    double gPrice =
        Spline.presentValue( Settlement, M, 100, gCoupon, *pCompounder );

    if( ! bClean ) {
            // Note that here we need to adjust if we want the dirty
            // price, not vice-versa.
        gPrice -= m_BondEquation.accruedInterest( Settlement, M, 
            gCoupon, pCompounder );
    }

    delete pCompounder;
    return gPrice;
}


//
//   b o n d P r i c e
//

double tBondMath::bondPrice( tDate Settlement, tDate Maturity,
    double gCoupon, const tDrift& Drift, bool bClean )

{
    tCompounder* pCompounder = createCompounder( Settlement, Maturity );
    MTG_ASSERT( pCompounder != 0 );

    tDate M;

    if( dynamic_cast<tCouponCompounder*>( pCompounder ) == 0 )
        M = Maturity;
    else
        M = dynamic_cast<tCouponCompounder*>( pCompounder )->maturity();

    double gPrice =
        Drift.presentValue( Settlement, M, 100, gCoupon, *pCompounder );

    if( ! bClean ) {
            // Note that here we need to adjust if we want the dirty
            // price, not vice-versa.
        gPrice -= m_BondEquation.accruedInterest( Settlement, M, 
            gCoupon, pCompounder );
    }

    delete pCompounder;
    return gPrice;
}


//
//   t B o n d M a t h
//

tBondMath::tBondMath()

{
    init();
}


//
//   t B o n d M a t h
//

tBondMath::tBondMath( const tBondMath& Math )

{
    init();
    copyFrom( Math );
}


//
//   ~ t B o n d M a t h
//

tBondMath::~tBondMath()

{
    if( m_pSettlement != 0 )
        delete m_pSettlement;
}


//
//   t o S e t t l e m e n t
//

tDate tBondMath::toSettlement( tDate Trade ) const

{
    if( m_pSettlement == 0 )
        return Trade;
    return toSettlement( Trade, *m_pSettlement );
}


//
//   t o S e t t l e m e n t
//

tDate tBondMath::toSettlement( tDate Trade,
    const tDayShift& Settlement ) const

{
    return Settlement.shift( Trade );
}


//
//   c r e a t e C o m p o u n d e r
//

tCompounder* tBondMath::createCompounder( tDate Settlement,
    tDate Maturity ) const

{
    return new tCouponCompounder( Maturity, 
        tIdentityDayShift(), 2 );
//        tFollowingDayShift(), 2 );
}


//
//   b i l l P r i c e
//

double tBondMath::billPrice( tDate Maturity, double gDiscount )

{
    return billPrice( tDate::today(), Maturity, gDiscount );
}


//
//   b i l l P r i c e
//

double tBondMath::billPrice( tDate Maturity, double gDiscount,
    const tDayShift& Settlement )

{
    return billPrice( tDate::today(), Maturity, gDiscount, Settlement );
}


//
//   b i l l P r i c e
//

double tBondMath::billPrice( tDate Trade, tDate Maturity,
    double gDiscount )

{
    return 100 * m_Discount.presentValue(
        toSettlement( Trade ), Maturity, gDiscount );
}


//
//   b i l l P r i c e
//

double tBondMath::billPrice( tDate Trade, tDate Maturity,
    double gDiscount, const tDayShift& Settlement )

{
    return 100 * m_Discount.presentValue(
        toSettlement( Trade, Settlement ), Maturity, gDiscount );
}                         


//
//   b i l l D i s c o u n t
//

double tBondMath::billDiscount( tDate Maturity, double gPrice )

{
    return billDiscount( tDate::today(), Maturity, gPrice );
}


//
//   b i l l D i s c o u n t
//

double tBondMath::billDiscount( tDate Maturity, double gPrice,
    const tDayShift& Settlement )

{
    return billDiscount( tDate::today(), Maturity, gPrice, Settlement );
}


//
//   b i l l D i s c o u n t
//

double tBondMath::billDiscount( tDate Trade, tDate Maturity, double gPrice )

{
    return m_Discount.invPresentValue( 
        toSettlement( Trade ), Maturity, gPrice / 100 );
}


//
//   b i l l D i s c o u n t
//

double tBondMath::billDiscount( tDate Trade, tDate Maturity,
     double gPrice, const tDayShift& Settlement )

{
    return m_Discount.invPresentValue( 
        toSettlement( Trade, Settlement ), Maturity, gPrice / 100 );
}


//
//   b i l l M M Y i e l d
//

double tBondMath::billMMYield( tDate Maturity, double gPrice )

{
    return billMMYield( tDate::today(), Maturity, gPrice );
}


//
//   b i l l M M Y i e l d
//

double tBondMath::billMMYield( tDate Maturity, double gPrice, 
    const tDayShift& Settlement )

{
    return billMMYield( tDate::today(), Maturity, gPrice, Settlement );
}


//
//   b i l l M M Y i e l d
//

double tBondMath::billMMYield( tDate Trade, tDate Maturity, double gPrice )

{
    return m_MMYield.invPresentValue( 
        toSettlement( Trade ), Maturity, gPrice / 100 );
}
     

//
//   b i l l M M Y i e l d
//

double tBondMath::billMMYield( tDate Trade, tDate Maturity, 
     double gPrice, const tDayShift& Settlement )

{
    return m_MMYield.invPresentValue( 
        toSettlement( Trade, Settlement ), Maturity, gPrice / 100 );
}


//
//   b i l l B E Y i e l d
//

double tBondMath::billBEYield( tDate Maturity, double gPrice )

{
    return billBEYield( tDate::today(), Maturity, gPrice );
}


//
//   b i l l B E Y i e l d
//

double tBondMath::billBEYield( tDate Maturity, double gPrice,
    const tDayShift& Settlement )

{
    return billBEYield( tDate::today(), Maturity, gPrice, Settlement );
}


//
//   b i l l B E Y i e l d
//

double tBondMath::billBEYield( tDate Trade, tDate Maturity,
    double gPrice )
   
{
    return m_BEYield.invPresentValue( 
        toSettlement( Trade ), Maturity, gPrice / 100 );
}


//
//   b i l l B E Y i e l d
//

double tBondMath::billBEYield( tDate Trade, tDate Maturity,
    double gPrice, const tDayShift& Settlement )

{
    return m_BEYield.invPresentValue( 
        toSettlement( Trade, Settlement ), Maturity, gPrice / 100 );
}


//
//   a c c r u e d I n t e r e s t
//

double tBondMath::accruedInterest( tDate Maturity, double gCoupon ) const

{
    return accruedInterest( tDate::today(), Maturity, gCoupon );
}


//
//   a c c r u e d I n t e r e s t
//

double tBondMath::accruedInterest( tDate Maturity,
    double gCoupon, const tDayShift& Settlement ) const

{
    return accruedInterest( tDate::today(), Maturity, gCoupon, Settlement );
}


//
//   a c c r u e d I n t e r e s t
//

double tBondMath::accruedInterest( tDate Trade, tDate Maturity,
    double gCoupon ) const

{
    tDate SettleDate = toSettlement( Trade );
    tCompounder* pCompounder = createCompounder( SettleDate, Maturity );

    MTG_ASSERT( pCompounder != 0 );
   
    double gAI = m_BondEquation.accruedInterest(
        SettleDate, Maturity, gCoupon, pCompounder );

    delete pCompounder;
    return gAI;
}


//
//   a c c r u e d I n t e r e s t
//

double tBondMath::accruedInterest( tDate Trade, tDate Maturity,
    double gCoupon, const tDayShift& Settlement ) const

{
    tDate SettleDate = toSettlement( Trade, Settlement );
    tCompounder* pCompounder = createCompounder( SettleDate, Maturity );

    MTG_ASSERT( pCompounder != 0 );
   
    double gAI = m_BondEquation.accruedInterest(
        SettleDate, Maturity, gCoupon, pCompounder );

    delete pCompounder;
    return gAI;
}


//
//   b o n d D i r t y P r i c e
//

double tBondMath::bondDirtyPrice( tDate Maturity, double gCoupon,
    double gYield )

{
    return bondDirtyPrice( tDate::today(), Maturity, gCoupon, gYield );
}


//
//   b o n d D i r t y P r i c e
//

double tBondMath::bondDirtyPrice( tDate Maturity, double gCoupon,
    double gYield, const tDayShift& Settlement )

{
    return bondDirtyPrice( tDate::today(), Maturity, gCoupon, gYield,
        Settlement );
}


//
//   b o n d D i r t y P r i c e
//

double tBondMath::bondDirtyPrice( tDate Trade, tDate Maturity,
    double gCoupon, double gYield )

{
    return bondPrice( toSettlement( Trade ),
        Maturity, gCoupon, gYield, false );
}


//
//   b o n d D i r t y P r i c e
//

double tBondMath::bondDirtyPrice( tDate Trade, tDate Maturity,
    double gCoupon, double gYield, const tDayShift& Settlement )

{
    return bondPrice( toSettlement( Trade, Settlement ),
        Maturity, gCoupon, gYield, false );
}


//
//   b o n d C l e a n P r i c e
//

double tBondMath::bondCleanPrice( tDate Maturity, double gCoupon,
    double gYield )

{
    return bondCleanPrice( tDate::today(), Maturity, gCoupon, gYield );
}


//
//   b o n d C l e a n P r i c e
//

double tBondMath::bondCleanPrice( tDate Maturity, double gCoupon,
    double gYield, const tDayShift& Settlement )

{
    return bondCleanPrice( tDate::today(), Maturity, gCoupon, gYield,
        Settlement );
}


//
//   b o n d C l e a n P r i c e
//

double tBondMath::bondCleanPrice( tDate Trade, tDate Maturity,
    double gCoupon, double gYield )

{
    return bondPrice( toSettlement( Trade ),
        Maturity, gCoupon, gYield, true );
}


//
//   b o n d C l e a n P r i c e
//

double tBondMath::bondCleanPrice( tDate Trade, tDate Maturity,
    double gCoupon, double gYield, const tDayShift& Settlement )

{
    return bondPrice( toSettlement( Trade, Settlement ),
        Maturity, gCoupon, gYield, true );
}


//
//   b o n d D i r t y P r i c e
//

double tBondMath::bondDirtyPrice( tDate Maturity, double gCoupon,
    tInterestSpline& Spline )

{
    return bondDirtyPrice( tDate::today(), Maturity, gCoupon, Spline );
}


//
//   b o n d D i r t y P r i c e
//

double tBondMath::bondDirtyPrice( tDate Maturity, double gCoupon,
    tInterestSpline& Spline, const tDayShift& Settlement )

{
    return bondDirtyPrice( tDate::today(), Maturity, gCoupon, Spline,
        Settlement );
}


//
//   b o n d D i r t y P r i c e
//

double tBondMath::bondDirtyPrice( tDate Trade, tDate Maturity,
    double gCoupon, tInterestSpline& Spline )

{
    return bondPrice( toSettlement( Trade ),
        Maturity, gCoupon, Spline, false );
}


//
//   b o n d D i r t y P r i c e
//

double tBondMath::bondDirtyPrice( tDate Trade, tDate Maturity,
    double gCoupon, tInterestSpline& Spline,
    const tDayShift& Settlement )

{
    return bondPrice( toSettlement( Trade, Settlement ),
        Maturity, gCoupon, Spline, false );
}


//
//   b o n d C l e a n P r i c e
//

double tBondMath::bondCleanPrice( tDate Maturity, double gCoupon,
    tInterestSpline& Spline )

{
    return bondCleanPrice( tDate::today(), Maturity, gCoupon, Spline );
}


//
//   b o n d C l e a n P r i c e
//

double tBondMath::bondCleanPrice( tDate Maturity, double gCoupon,
    tInterestSpline& Spline, const tDayShift& Settlement )

{
    return bondCleanPrice( tDate::today(), Maturity, gCoupon, Spline,
        Settlement );
}


//
//   b o n d C l e a n P r i c e
//

double tBondMath::bondCleanPrice( tDate Trade, tDate Maturity,
    double gCoupon, tInterestSpline& Spline )

{
    return bondPrice( toSettlement( Trade ),
        Maturity, gCoupon, Spline, true );
}


//
//   b o n d C l e a n P r i c e
//

double tBondMath::bondCleanPrice( tDate Trade, tDate Maturity,
    double gCoupon, tInterestSpline& Spline,
    const tDayShift& Settlement )

{
    return bondPrice( toSettlement( Trade, Settlement ),
        Maturity, gCoupon, Spline, true );
}


//
//   b o n d D i r t y P r i c e
//

double tBondMath::bondDirtyPrice( tDate Maturity, double gCoupon,
    const tDrift& Drift )

{
    return bondDirtyPrice( tDate::today(), Maturity, gCoupon, Drift );
}


//
//   b o n d D i r t y P r i c e
//

double tBondMath::bondDirtyPrice( tDate Maturity, double gCoupon,
    const tDrift& Drift, const tDayShift& Settlement )

{
    return bondDirtyPrice( tDate::today(), Maturity, gCoupon, Drift,
        Settlement );
}


//
//   b o n d D i r t y P r i c e
//

double tBondMath::bondDirtyPrice( tDate Trade, tDate Maturity,
    double gCoupon, const tDrift& Drift )

{
    return bondPrice( toSettlement( Trade ),
        Maturity, gCoupon, Drift, false );
}


//
//   b o n d D i r t y P r i c e
//

double tBondMath::bondDirtyPrice( tDate Trade, tDate Maturity,
    double gCoupon, const tDrift& Drift, const tDayShift& Settlement )

{
    return bondPrice( toSettlement( Trade, Settlement ),
        Maturity, gCoupon, Drift, false );
}


//
//   b o n d C l e a n P r i c e
//

double tBondMath::bondCleanPrice( tDate Maturity, double gCoupon,
    const tDrift& Drift )

{
    return bondCleanPrice( tDate::today(), Maturity, gCoupon, Drift );
}


//
//   b o n d C l e a n P r i c e
//

double tBondMath::bondCleanPrice( tDate Maturity, double gCoupon,
    const tDrift& Drift, const tDayShift& Settlement )

{
    return bondCleanPrice( tDate::today(), Maturity, gCoupon, Drift,
        Settlement );
}


//
//   b o n d C l e a n P r i c e
//

double tBondMath::bondCleanPrice( tDate Trade, tDate Maturity,
    double gCoupon, const tDrift& Drift )

{
    return bondPrice( toSettlement( Trade ),
        Maturity, gCoupon, Drift, true );
}


//
//   b o n d C l e a n P r i c e
//

double tBondMath::bondCleanPrice( tDate Trade, tDate Maturity,
    double gCoupon, const tDrift& Drift, const tDayShift& Settlement )

{
    return bondPrice( toSettlement( Trade, Settlement ),
        Maturity, gCoupon, Drift, true );
}


//
//   b o n d D i r t y 2 Y i e l d
//

double tBondMath::bondDirty2Yield( tDate Maturity, double gCoupon,
    double gPrice )

{
    return bondDirty2Yield( tDate::today(), Maturity, gCoupon, gPrice );
}


//
//   b o n d D i r t y 2 Y i e l d
//

double tBondMath::bondDirty2Yield( tDate Maturity, double gCoupon,
    double gPrice, const tDayShift& Settlement )

{
    return bondDirty2Yield( tDate::today(), Maturity, gCoupon, gPrice,
        Settlement );
}


//
//   b o n d D i r t y 2 Y i e l d
//

double tBondMath::bondDirty2Yield( tDate Trade, tDate Maturity,
    double gCoupon, double gPrice )

{
    return bondYield( toSettlement( Trade ),
        Maturity, gCoupon, gPrice, false );
}


//
//   b o n d D i r t y 2 Y i e l d
//

double tBondMath::bondDirty2Yield( tDate Trade, tDate Maturity,
    double gCoupon, double gPrice, const tDayShift& Settlement )

{
    return bondYield( toSettlement( Trade, Settlement ),
        Maturity, gCoupon, gPrice, false );
}


//
//   b o n d C l e a n 2 Y i e l d
//

double tBondMath::bondClean2Yield( tDate Maturity, double gCoupon,
    double gPrice )

{
    return bondClean2Yield( tDate::today(), Maturity, gCoupon, gPrice );
}


//
//   b o n d C l e a n 2 Y i e l d
//

double tBondMath::bondClean2Yield( tDate Maturity, double gCoupon,
    double gPrice, const tDayShift& Settlement )

{
    return bondClean2Yield( tDate::today(), Maturity, gCoupon, gPrice,
        Settlement );
}


//
//   b o n d C l e a n 2 Y i e l d
//

double tBondMath::bondClean2Yield( tDate Trade, tDate Maturity,
    double gCoupon, double gPrice )

{
    return bondYield( toSettlement( Trade ),
        Maturity, gCoupon, gPrice, true );
}


//
//   b o n d C l e a n 2 Y i e l d
//

double tBondMath::bondClean2Yield( tDate Trade, tDate Maturity,
    double gCoupon, double gPrice, const tDayShift& Settlement )

{
    return bondYield( toSettlement( Trade, Settlement ),
        Maturity, gCoupon, gPrice, true );
}

MTG_END_NAMESPACE
