// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#if ! defined(_MTG_BOND_MATH_)
#define _MTG_BOND_MATH_

#include "MtgBondEquation.h"
#include "MtgDayShift.h"
#include "MtgDrift.h"
#include "MtgInterestSpline.h"

MTG_BEGIN_NAMESPACE


//
//   t B o n d M a t h
//

class tBondMath {

    tDayShift* m_pSettlement;

        // Different discount and yield conventions for
        // discount securities:

    tInterest m_Discount;
    tInterest m_MMYield;    // Money Market yield
    tInterest m_BEYield;    // Bond equivalent yield

        // Bond equation to use for coupon bearing bonds:

    tBondEquation m_BondEquation;

    void init();

protected:

    void copyFrom( const tBondMath& Math );

        // Initialization routines for subclasses:

    void setSettlement( const tDayShift& Shift );

    void setDiscount( const tInterest& Discount );
    void setMMYield( const tInterest& MMYield );
    void setBEYield( const tInterest& BEYield );

    void setBondEquation( const tBondEquation& BondEquation );

        // These functions use the registered bond equation to
        // compute clean/dirty price resp. yield-to-maturity.
        // They use createCompounder() to create an appropriate
        // compounding object first.

    double bondPrice( tDate Settlement, tDate Maturity, double gCoupon, 
        double gYield, bool bClean );

    double bondYield( tDate Settlement, tDate Maturity, double gCoupon, 
        double gPrice, bool bClean );

        // These functions do not use the registered bond
        // equation, but still calls createCompounder().

    double bondPrice( tDate Settlement, tDate Maturity, double gCoupon, 
        tInterestSpline& Spline, bool bClean );

    double bondPrice( tDate Settlement, tDate Maturity, double gCoupon, 
        const tDrift& Drift, bool bClean );

public:

    tBondMath();
    tBondMath( const tBondMath& Math );

    virtual ~tBondMath();

        // Advance trade date to settlement date.

    virtual tDate toSettlement( tDate Trade ) const;
    virtual tDate toSettlement( tDate Trade,
        const tDayShift& Settlement ) const;

        // This function is supposed to create a compounder for
        // a bond maturing at the specified date. The default
        // creates a coupon compounder with semiannaul coupon
        // payment. The compounder must be freed in bondPrice() or
        // bondYield() or whereever this funciton is called.

    virtual tCompounder* createCompounder( tDate Settlement,
        tDate Maturity ) const;

        // For discount securities, the calculations are virtually
        // the same for all markets and implemented here.
        // The principal is always assumed to be 100.

        // Convert the quoted discount of the security to
        // the dollar price:

    virtual double billPrice( tDate Maturity, double gDiscount );
    virtual double billPrice( tDate Maturity, double gDiscount,
        const tDayShift& Settlement );
    virtual double billPrice( tDate Trade, tDate Maturity,
        double gDiscount );
    virtual double billPrice( tDate Trade, tDate Maturity,
        double gDiscount, const tDayShift& Settlement );

        // Convert the dollar price of the discount security
        // to its discount:

    virtual double billDiscount( tDate Maturity, double gPrice );
    virtual double billDiscount( tDate Maturity, double gPrice,
        const tDayShift& Settlement );
    virtual double billDiscount( tDate Trade, tDate Maturity,
        double gPrice );
    virtual double billDiscount( tDate Trade, tDate Maturity,
        double gPrice, const tDayShift& Settlement );

        // Convert the dollar price of the discount security to
        // its equivalent money market yield:

    virtual double billMMYield( tDate Maturity, double gPrice );
    virtual double billMMYield( tDate Maturity, double gPrice,
        const tDayShift& Settlement );
    virtual double billMMYield( tDate Trade, tDate Maturity,
        double gPrice ); 
    virtual double billMMYield( tDate Trade, tDate Maturity,
        double gPrice, const tDayShift& Settlement );

        // Convert the dollar price of the discount security to the
        // bond yield equivalent:

    virtual double billBEYield( tDate Maturity, double gPrice );
    virtual double billBEYield( tDate Maturity, double gPrice,
        const tDayShift& Settlement );
    virtual double billBEYield( tDate Trade, tDate Maturity,
        double gPrice );
    virtual double billBEYield( tDate Trade, tDate Maturity,
        double gPrice, const tDayShift& Settlement );

        // Interest bearing securities are more diverse. For the
        // simplest incarnations, hoever, the following functions
        // suffice. (E.g., for the US treasury bonds and notes.)

    virtual double accruedInterest( tDate Maturity,
        double gCoupon ) const;
    virtual double accruedInterest( tDate Maturity,
        double gCoupon, const tDayShift& Settlement ) const;
    virtual double accruedInterest( tDate Trade, tDate Maturity,
        double gCoupon ) const;
    virtual double accruedInterest( tDate Trade, tDate Maturity,
        double gCoupon, const tDayShift& Settlement ) const;

        // Convert the bond yield to maturity to the dirty price:

    virtual double bondDirtyPrice( tDate Maturity, double gCoupon,
        double gYield );
    virtual double bondDirtyPrice( tDate Maturity, double gCoupon,
        double gYield, const tDayShift& Settlement );

    virtual double bondDirtyPrice( tDate Trade, tDate Maturity,
        double gCoupon, double gYield );
    virtual double bondDirtyPrice( tDate Trade, tDate Maturity,
        double gCoupon, double gYield, const tDayShift& Settlement );

        // Convert the bond yield to maturity to the clean price:

    virtual double bondCleanPrice( tDate Maturity, double gCoupon,
        double gYield );
    virtual double bondCleanPrice( tDate Maturity, double gCoupon,
        double gYield, const tDayShift& Settlement );

    virtual double bondCleanPrice( tDate Trade, tDate Maturity,
        double gCoupon, double gYield );
    virtual double bondCleanPrice( tDate Trade, tDate Maturity,
        double gCoupon, double gYield, const tDayShift& Settlement );

        // The following functions compute the price under a 
        // stepwise constant forward rate term structure.
        // All calculations are "true", as opposed to
        // the approximations used for the yield-to-maturity
        // conventions. The bond equation is not used!

        // Both dirty...

    virtual double bondDirtyPrice( tDate Maturity, double gCoupon,
        tInterestSpline& Spline );
    virtual double bondDirtyPrice( tDate Maturity, double gCoupon,
        tInterestSpline& Spline, const tDayShift& Settlement );

    virtual double bondDirtyPrice( tDate Trade, tDate Maturity,
        double gCoupon, tInterestSpline& Spline );
    virtual double bondDirtyPrice( tDate Trade, tDate Maturity,
        double gCoupon, tInterestSpline& Spline,
        const tDayShift& Settlement );

        // ...and clean:

    virtual double bondCleanPrice( tDate Maturity, double gCoupon,
        tInterestSpline& Spline );
    virtual double bondCleanPrice( tDate Maturity, double gCoupon,
        tInterestSpline& Spline, const tDayShift& Settlement );

    virtual double bondCleanPrice( tDate Trade, tDate Maturity,
        double gCoupon, tInterestSpline& Spline );
    virtual double bondCleanPrice( tDate Trade, tDate Maturity,
        double gCoupon, tInterestSpline& Spline,
        const tDayShift& Settlement );

        // Finally, the same services for general drift
        // objects. Again, all calculations are "true."

        // Both dirty...

    virtual double bondDirtyPrice( tDate Maturity, double gCoupon,
        const tDrift& Drift );
    virtual double bondDirtyPrice( tDate Maturity, double gCoupon,
        const tDrift& Drift, const tDayShift& Settlement );

    virtual double bondDirtyPrice( tDate Trade, tDate Maturity,
        double gCoupon, const tDrift& Drift );
    virtual double bondDirtyPrice( tDate Trade, tDate Maturity,
        double gCoupon, const tDrift& Drift, const tDayShift& Settlement );

        // ...and clean:

    virtual double bondCleanPrice( tDate Maturity, double gCoupon,
        const tDrift& Drift );
    virtual double bondCleanPrice( tDate Maturity, double gCoupon,
        const tDrift& Drift, const tDayShift& Settlement );

    virtual double bondCleanPrice( tDate Trade, tDate Maturity,
        double gCoupon, const tDrift& Drift );
    virtual double bondCleanPrice( tDate Trade, tDate Maturity,
        double gCoupon, const tDrift& Drift, const tDayShift& Settlement );

        // Convert the clean bond price to its yield-to-maturity.
        // Here, the bond equation is used again.

    virtual double bondDirty2Yield( tDate Maturity, double gCoupon,
        double gPrice );
    virtual double bondDirty2Yield( tDate Maturity, double gCoupon,
        double gPrice, const tDayShift& Settlement );

    virtual double bondDirty2Yield( tDate Trade, tDate Maturity,
        double gCoupon, double gPrice );
    virtual double bondDirty2Yield( tDate Trade, tDate Maturity,
        double gCoupon, double gPrice, const tDayShift& Settlement );

        // Convert the clean bond price to its yield-to-maturity:

    virtual double bondClean2Yield( tDate Maturity, double gCoupon,
        double gPrice );
    virtual double bondClean2Yield( tDate Maturity, double gCoupon,
        double gPrice, const tDayShift& Settlement );

    virtual double bondClean2Yield( tDate Trade, tDate Maturity,
        double gCoupon, double gPrice );
    virtual double bondClean2Yield( tDate Trade, tDate Maturity,
        double gCoupon, double gPrice, const tDayShift& Settlement );
};

MTG_END_NAMESPACE

#endif
