// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#include "MtgIncl.h"
#include "MtgCgi.h"

#if defined(_MTG_WITH_CGI)

#include <cgic.h>

MTG_BEGIN_NAMESPACE


//
//   v a r i a b l e s
//

bool tCgi::m_bIsInitialized = false;


//
//   t r i m
//

void tCgi::trim( char* sValue )

{
    int l = strlen( sValue );
    while( l > 0 && isspace( sValue[l - 1] ) )
        --l;
    sValue[l] = 0;

    if( isspace( *sValue ) ) {
        char* s = sValue;
        char* q = sValue;

        while( isspace( *s ) )
            ++s;

        while( *s )
            *q++ = *s++;
        *q = 0;
    }
}


//
//   i s A c t i v e
//

bool tCgi::isActive()

{
    int nIsCgi = 0;

    nIsCgi += getenv( "SERVER_NAME" ) != NULL;
    nIsCgi += getenv( "SERVER_PROTOCOL" ) != NULL;
    nIsCgi += getenv( "SERVER_PORT" ) != NULL;
    nIsCgi += getenv( "SCRIPT_NAME" ) != NULL;
    nIsCgi += getenv( "CONTENT_TYPE" ) != NULL;
    nIsCgi += getenv( "CONTENT_LENGTH" ) != NULL;                              

    if( nIsCgi >= 4 )
        return true;
    return false;
}


//
//   i n i t
//

tRetCode tCgi::init()

{
    if( m_bIsInitialized )
        return OK;
    if( cgiAllocResources() != 0 )
        return CGI_ERROR;
    m_bIsInitialized = true;
    return OK;
}


//
//   i n i t
//

tRetCode tCgi::init( const char* sEnvFile )

{
    tRetCode nRet;

    if( ( nRet = init() ) != OK )
        return nRet;
    if( sEnvFile != 0 )
        cgiReadEnvironment( const_cast<char*>( sEnvFile ) );
    return OK;
}


//
//   e x i t
//

void tCgi::exit()

{
    if( m_bIsInitialized ) {
        cgiFreeResources();
        m_bIsInitialized = false;
    }
}


//
//   r e a d S t a t e
//

void tCgi::readState( const char* sFileName )

{
    if( sFileName != 0 )
        cgiReadEnvironment( const_cast<char*>( sFileName ) );
}


//
//   w r i t e S t a t e
//

void tCgi::writeState( const char* sFileName )

{
    if( sFileName != 0 )
        cgiWriteEnvironment( const_cast<char*>( sFileName ) );
}


//
//   g e t S e r v e r S o f t w a r e
//

const char* tCgi::getServerSoftware()

{
    MTG_ASSERT( m_bIsInitialized );
    return cgiServerSoftware;
}


//
//   g e t S e r v e r N a m e
//

const char* tCgi::getServerName()

{
    MTG_ASSERT( m_bIsInitialized );
    return cgiServerName;
}


//
//   g e t G a t e w a y I n t e r f a c e
//

const char* tCgi::getGatewayInterface()

{
    MTG_ASSERT( m_bIsInitialized );
    return cgiGatewayInterface;
}


//
//   g e t S e r v e r P r o t o c o l
//

const char* tCgi::getServerProtocol()

{
    MTG_ASSERT( m_bIsInitialized );
    return cgiServerProtocol;
}


//
//   g e t S e r v e r P o r t
//

const char* tCgi::getServerPort()

{
    MTG_ASSERT( m_bIsInitialized );
    return cgiServerPort;
}


//
//   g e t R e q u e s t M e t h o d
//

const char* tCgi::getRequestMethod()

{
    MTG_ASSERT( m_bIsInitialized );
    return cgiRequestMethod;
}


//
//   g e t P a t h I n f o
//

const char* tCgi::getPathInfo()

{
    MTG_ASSERT( m_bIsInitialized );
    return cgiPathInfo;
}


//
//   g e t P a t h T r a n s l a t e d
//

const char* tCgi::getPathTranslated()

{
    MTG_ASSERT( m_bIsInitialized );
    return cgiPathTranslated;
}


//
//   g e t S c r i p t N a m e
//

const char* tCgi::getScriptName()

{
    MTG_ASSERT( m_bIsInitialized );
    return cgiScriptName;
}


//
//   g e t Q u e r y S t r i n g
//

const char* tCgi::getQueryString()

{
    MTG_ASSERT( m_bIsInitialized );
    return cgiQueryString;
}


//
//   g e t R e m o t e H o s t
//

const char* tCgi::getRemoteHost()

{
    MTG_ASSERT( m_bIsInitialized );
    return cgiRemoteHost;
}


//
//   g e t R e m o t e A d d r
//

const char* tCgi::getRemoteAddr()

{
    MTG_ASSERT( m_bIsInitialized );
    return cgiRemoteAddr;
}


//
//   g e t A u t h T y p e
//

const char* tCgi::getAuthType()

{
    MTG_ASSERT( m_bIsInitialized );
    return cgiAuthType;
}


//
//   g e t R e m o t e U s e r
//

const char* tCgi::getRemoteUser()

{
    MTG_ASSERT( m_bIsInitialized );
    return cgiRemoteUser;
}


//
//   g e t R e m o t e I d e n t
//

const char* tCgi::getRemoteIdent()

{
    MTG_ASSERT( m_bIsInitialized );
    return cgiRemoteIdent;
}


//
//   g e t C o n t e n t T y p e
//

const char* tCgi::getContentType()

{
    MTG_ASSERT( m_bIsInitialized );
    return cgiContentType;
}


//
//   g e t A c c e p t
//

const char* tCgi::getAccept()

{
    MTG_ASSERT( m_bIsInitialized );
    return cgiAccept;
}


//
//   g e t U s e r A g e n t
//

const char* tCgi::getUserAgent()

{
    MTG_ASSERT( m_bIsInitialized );
    return cgiUserAgent;
}


//
//   g e t V a r
//

tRetCode tCgi::getVar( const char* sName, char*& sValue, bool bTrim )

{
    MTG_ASSERT( m_bIsInitialized );

    cgiFormResultType nRes;
    int nLength;

    nRes = cgiFormStringSpaceNeeded( const_cast<char*>( sName ), &nLength );
    if( nRes != cgiFormSuccess )
        return NOT_FOUND;

    if( nLength == 1 )
        return EMPTY;

    sValue = new char[nLength];
    nRes = cgiFormString( const_cast<char*>( sName ), sValue, nLength );
    MTG_ASSERT( nRes == cgiFormSuccess );

    if( bTrim )
        trim( sValue );

    if( *sValue == 0 ) {
        delete sValue;
        sValue = 0;
        return EMPTY;
    }

    return OK;
}


//
//   g e t V a r
//

tRetCode tCgi::getVar( const char* sName, tDate& Value )

{
    tRetCode nRet;
    char* s = 0;

    if( ( nRet = getVar( sName, s, true ) ) != OK )
        return nRet;

    if( *s == 0 ) {
        delete s;
        return EMPTY;
    }

    if( strcmp( sName, "today" ) == 0 ) {
        Value = tDate::today();
    }
    else {
        try {
            Value = s;
        }
        catch( tException e ) {
            delete s;
            return e.ret();
        }
    }

    delete s;
    return OK;
}


//
//   g e t V a r
//

tRetCode tCgi::getVar( const char* sName, tDate& Value, tDate Default )

{
    tRetCode nRet;

    if( ( nRet = getVar( sName, Value ) ) == OK )
        return nRet;
    if( nRet == EMPTY ) {
        Value = Default;
        return OK;
    }
    return nRet;
}


//
//   g e t V a r
//

tRetCode tCgi::getVar( const char* sName, int& nValue )

{
    MTG_ASSERT( m_bIsInitialized );

    switch( cgiFormInteger( const_cast<char*>( sName ), &nValue, 0 ) ) {
        case cgiFormBadType :  return FORMAT_ERROR;
        case cgiFormEmpty :    return EMPTY;
        case cgiFormNotFound : return NOT_FOUND;
        default :              break;
    }

    return OK;
}


//
//   g e t V a r
//

tRetCode tCgi::getVar( const char* sName, int& nValue, int nDefault )

{
    tRetCode nRet;

    if( ( nRet = getVar( sName, nValue ) ) == OK )
        return nRet;
    if( nRet == EMPTY ) {
        nValue = nDefault;
        return OK;
    }
    return nRet;
}


//
//   g e t V a r
//

tRetCode tCgi::getVar( const char* sName, double& gValue )

{
    MTG_ASSERT( m_bIsInitialized );

    switch( cgiFormDouble( const_cast<char*>( sName ), &gValue, 0 ) ) {
        case cgiFormBadType :  return FORMAT_ERROR;
        case cgiFormEmpty :    return EMPTY;
        case cgiFormNotFound : return NOT_FOUND;
        default :              break;
    }

    return OK;
}


//
//   g e t V a r
//

tRetCode tCgi::getVar( const char* sName, double& gValue, double gDefault )

{
    tRetCode nRet;

    if( ( nRet = getVar( sName, gValue ) ) == OK )
        return nRet;
    if( nRet == EMPTY ) {
        gValue = gDefault;
        return OK;
    }
    return nRet;
}


//
//   g e t V a r
//

tRetCode tCgi::getVar( const char* sName, int& nIndex,
    const tHeap<char*>& Candidates )

{
    MTG_ASSERT( m_bIsInitialized );

    switch( cgiFormSelectSingle( const_cast<char*>( sName ),
                &Candidates[0], Candidates.numOfElems(),
                &nIndex, -1 ) ) {
        case cgiFormNotFound :     return NOT_FOUND;
        case cgiFormNoSuchChoice : return FORMAT_ERROR;
        default :                  break;
    }

    return OK;
}

    
//
//   g e t V a r
//
    
tRetCode tCgi::getVar( const char* sName, int& nIndex,
    const tHeap<char*>& Candidates, int nDefault )

{
    tRetCode nRet;

    if( ( nRet = getVar( sName, nIndex, Candidates ) ) == OK )
        return nRet;
    if( nRet == NOT_FOUND ) {
        nIndex = nDefault;
        return OK;
    }
    return nRet;
}


//
//   g e t V a r
//

tRetCode tCgi::getVar( const char* sName, int& nIndex,
    const char* sCand0, const char* sCand1 )

{
    MTG_ASSERT( m_bIsInitialized );

    char* s[2];

    s[0] = const_cast<char*>( sCand0 );
    s[1] = const_cast<char*>( sCand1 );

    switch( cgiFormSelectSingle( const_cast<char*>( sName ),
                s, 2, &nIndex, -1 ) ) {
        case cgiFormNotFound :     return NOT_FOUND;
        case cgiFormNoSuchChoice : return FORMAT_ERROR;
        default :                  break;
    }

    return OK;
}


//
//   g e t V a r
//

tRetCode tCgi::getVar( const char* sName, int& nIndex,
    const char* sCand0, const char* sCand1, int nDefault )

{
    tRetCode nRet;

    if( ( nRet = getVar( sName, nIndex, sCand0, sCand1 ) ) == OK )
        return nRet;
    if( nRet == NOT_FOUND ) {
        nIndex = nDefault;
        return OK;
    }
    return nRet;
}


//
//   o u t H e a d e r C o n t e n t T y p e
//

tRetCode tCgi::outHeaderContentType( const char* sType,
    const char* sSubType )

{
    MTG_ASSERT( m_bIsInitialized );

    fprintf( cgiOut, "Content-type: %s/%s%c%c", sType, sSubType, 10, 10);
    return OK;
}


//
//   o u t
//

tRetCode tCgi::out( const char* sFmt, ... )

{
    MTG_ASSERT( m_bIsInitialized );

    va_list Marker;

    va_start( Marker, sFmt );
    vfprintf( cgiOut, sFmt, Marker );
    return OK;
}

MTG_END_NAMESPACE

#endif