// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#include "MtgIncl.h"
#include "MtgClaim.h"
#include "MtgCashflow.h"
#include "MtgCustomClaim.h"
#include "MtgEvGenerator.h"
#include "MtgFactor.h"
#include "MtgFDEngine.h"
#include "MtgMCEngine.h"
#include "MtgParser.h"
#include "MtgStdClaim.h"
#include "MtgUSTBond.h"

MTG_BEGIN_NAMESPACE


//
//   v a r i a b l e s
//

int tClaim::m_nNextId = 0;


//
//   i n i t
//

void tClaim::init()

{
    m_nId = m_nNextId++;
    m_sComment = 0;

    m_nBirth = -1;
    m_nMaturity = -1;
    m_bIsPriced = false;
    m_gUnitBidPrice = -1;
    m_gUnitAskPrice = -1;
    m_gUnitPriceShift = 0;
    m_gMultiplier = 1;
    m_bHasUpBarrier = false;
    m_gUpBarrier = -1;
    m_bHasDownBarrier = false;
    m_gDownBarrier = -1;
    m_bMonitor = false;
    m_bResolved = false;

    m_nIndex = -1;
    m_nSlot = -1;
}


//
//   p a r s e L i f e t i m e
//

tRetCode tClaim::parseLifetime( tParser& Parser )

{
    MTG_ASSERT( Parser.curToken() == xTokBirth ||
                Parser.curToken() == xTokMaturity );

    tRetCode nRet;

    switch( Parser.curToken() ) {
        case xTokBirth :
            if( m_nBirth >= 0 )
                return Parser.setError( "'birth' defined more than once" );
            if( ( nRet = Parser.readToken() ) != OK )
                return nRet;
            if( Parser.tryDate( m_BirthDate ) ) {
                if( m_BirthDate < system().base() )
                    return Parser.setError( INVALID_DATE );
                m_nBirth = system().dateIndex( m_BirthDate );
                if( ( nRet = Parser.readToken() ) != OK )
                    return nRet;
            }
            else {
                if( ( nRet = Parser.scanInteger( m_nBirth, 0 ) ) != OK )
                    return nRet;
                m_BirthDate = system().base() + m_nBirth;
            }
            if( m_nMaturity >= 0 && m_nMaturity <= m_nBirth )
                return Parser.setError( "invalid birth/maturity combination" );
            break;

        case xTokMaturity :
            if( m_nMaturity >= 0 )
                return Parser.setError( "'maturity' defined more than once" );
            if( ( nRet = Parser.readToken() ) != OK )
                return nRet;
            if( Parser.tryDate( m_MaturityDate ) ) {
                if( m_MaturityDate < system().base() )
                    return Parser.setError( INVALID_DATE );
                m_nMaturity = system().dateIndex( m_MaturityDate );
                if( ( nRet = Parser.readToken() ) != OK )
                    return nRet;
            }
            else {
                if( ( nRet = Parser.scanInteger( m_nMaturity, 1 ) ) != OK )
                    return nRet;
                m_MaturityDate = system().base() + m_nMaturity;
            }
            if( m_nBirth >= 0 && m_nMaturity <= m_nBirth )
                return Parser.setError( "invalid birth/maturity combination" );
            break;

        default :
            break;
    }

    return OK;
}


//
//   p a r s e M u l t i p l i e r
//

tRetCode tClaim::parseMultiplier( tParser& Parser, bool& bMultiplier )
    
{
    tRetCode nRet;

    if( bMultiplier )
        return Parser.setError( "'multiplier' defined more than once" );

    switch( Parser.curToken() ) {
        case xTokMultiplier :
        case xTokPosition :
            if( ( nRet = Parser.readToken() ) != OK ||
                ( nRet = Parser.scanDouble( m_gMultiplier ) ) != OK ) {
                return nRet;
            }
            break;

        case xTokLong :
            if( ( nRet = Parser.readToken() ) != OK )
                return nRet;
            if( Parser.curToken() == xTokNumber ) {
                if( ( nRet = Parser.scanDouble( m_gMultiplier ) ) != OK )
                    return nRet;
            }
            else {
                m_gMultiplier = 1;
            }
            break;

        case xTokShort :
            if( ( nRet = Parser.readToken() ) != OK )
                return nRet;
            if( Parser.curToken() == xTokNumber ) {
                if( ( nRet = Parser.scanDouble( m_gMultiplier ) ) != OK )
                    return nRet;
                m_gMultiplier = -m_gMultiplier;
            }
            else {
                m_gMultiplier = -1;
            }
            break;

        default :
            if( ( nRet = Parser.scanDouble( m_gMultiplier ) ) != OK )
                return nRet;
            break;
    }

    bMultiplier = true;
    return OK;
}


//
//   p a r s e M a r k e t
//

tRetCode tClaim::parsePrice( tParser& Parser )

{
    MTG_ASSERT( Parser.curToken() == xTokPrice ||
                Parser.curToken() == xTokBidPrice ||
                Parser.curToken() == xTokAskPrice );

    tRetCode nRet;
    
    switch( Parser.curToken() ) {
        case xTokPrice :
            if( m_gUnitBidPrice >= 0 || m_gUnitAskPrice >= 0 )
                return Parser.setError( "can only define one price" );
            if( ( nRet = Parser.readToken() ) != OK ||
                ( nRet = Parser.scanNonNegDouble( m_gUnitBidPrice ) ) != OK ) {
                return nRet;
            }
            m_gUnitAskPrice = m_gUnitBidPrice;
            break;

        case xTokBidPrice :
            if( m_gUnitBidPrice >= 0 )
                return Parser.setError( "can only define one price" );
            if( ( nRet = Parser.readToken() ) != OK ||
                ( nRet = Parser.scanNonNegDouble( m_gUnitBidPrice ) ) != OK ) {
                return nRet;
            }
            if( m_gUnitAskPrice >= 0 && m_gUnitAskPrice < m_gUnitBidPrice )
                return Parser.setError( "invalid bid/ask spread" );
            break;

        case xTokAskPrice :
            if( m_gUnitAskPrice >= 0 )
                return Parser.setError( "can only define one price" );
            if( ( nRet = Parser.readToken() ) != OK ||
                ( nRet = Parser.scanNonNegDouble( m_gUnitAskPrice ) ) != OK ) {
                return nRet;
            }
            if( m_gUnitBidPrice >= 0 && m_gUnitAskPrice < m_gUnitBidPrice )
                return Parser.setError( "invalid bid/ask spread" );
            break;

        default :
            break;
    }

    return OK;
}


//
//   p a r s e B a r r i e r
//

tRetCode tClaim::parseBarrier( tParser& Parser )
    
{
    MTG_ASSERT( Parser.curToken() == xTokUpAndOut ||
                Parser.curToken() == xTokDownAndOut );

    tRetCode nRet;

    switch( Parser.curToken() ) {
        case xTokUpAndOut :
            if( m_bHasUpBarrier ) {
                return Parser.setError( "'up_and_out' barrier "
                                    "defined more than once" );
            }
            if( ( nRet = Parser.readToken() ) != OK ||
                ( nRet = Parser.scanNonNegDouble( m_gUpBarrier ) ) != OK ) {
                return nRet;
            }
            m_bHasUpBarrier = true;
            break;

        case xTokDownAndOut :
            if( m_bHasDownBarrier ) {
                return Parser.setError( "'down_and_out' barrier "
                                    "defined more than once" );
            }
            if( ( nRet = Parser.readToken() ) != OK ||
                ( nRet = Parser.scanNonNegDouble( m_gDownBarrier ) ) != OK ) {
                return nRet;
            }
            m_bHasDownBarrier = true;
            break;

        default :
            break;
    }

    return OK;
}


//
//   s e t I n d e x
//

void tClaim::setIndex( int nIndex )

{
    MTG_ASSERT( nIndex >= 0 );
    m_nIndex = nIndex;
}


//
//   s e t S l o t
//

void tClaim::setSlot( int nSlot )

{
    MTG_ASSERT( nSlot >= 0 );
    m_nSlot = nSlot;
}


//
//   c o p y F r o m
//

void tClaim::copyFrom( const tClaim& Claim )

{
    if( &Claim == this )
        return;

    m_nBirth = Claim.m_nBirth;
    m_BirthDate = Claim.m_BirthDate;
    m_nMaturity = Claim.m_nMaturity;
    m_MaturityDate = Claim.m_MaturityDate;
    m_bIsPriced = Claim.m_bIsPriced;
    m_gUnitBidPrice = Claim.m_gUnitBidPrice;
    m_gUnitAskPrice = Claim.m_gUnitAskPrice;
    m_gUnitPriceShift = Claim.m_gUnitPriceShift;
    m_Settlement = Claim.m_Settlement;
    m_gMultiplier = Claim.m_gMultiplier;
    m_bHasUpBarrier = Claim.m_bHasUpBarrier;
    m_gUpBarrier = Claim.m_gUpBarrier;
    m_bHasDownBarrier = Claim.m_bHasDownBarrier;
    m_gDownBarrier = Claim.m_gDownBarrier;
    m_bMonitor = Claim.m_bMonitor;

    if( m_sComment != 0 )
        delete m_sComment;
    m_sComment = StrCopy( Claim.m_sComment );

    for( int k = 0; k < m_Cashflow.numOfElems(); ++k )
        delete m_Cashflow[k];

    m_Cashflow.reset();
    for( MTG_FOR_INIT( int ) k = 0; k < Claim.m_Cashflow.numOfElems(); ++k )
        m_Cashflow.append( Claim.m_Cashflow[k]->clone() );

    m_bResolved = Claim.m_bResolved;
    if( Claim.isFinalized() )
        m_Access = Claim.m_Access;
    else
        m_Access.reset();

    super::copyFrom( Claim );
}


//
//   s e t M o n i t o r
//

void tClaim::setMonitor()

{
    MTG_ASSERT( ! m_bMonitor );

    m_bMonitor = true;
    touch();
}


//
//   s e t U p B a r r i e r
//

void tClaim::setUpBarrier( double gBarrier )

{
    MTG_ASSERT( ! m_bHasUpBarrier );

    m_bHasUpBarrier = true;
    m_gUpBarrier = gBarrier;    // may be negative, in which case
                                // the derived class supplies its
                                // own barrier
}


//
//   s e t D o w n B a r r i e r
//

void tClaim::setDownBarrier( double gBarrier )

{
    MTG_ASSERT( ! m_bHasDownBarrier );

    m_bHasDownBarrier = true;
    m_gDownBarrier = gBarrier;  // may be negative, in which case
                                // the derived class supplies its
                                // own barrier
}


//
//   r e g C a s h f l o w
//

void tClaim::regCashflow( tCashflow* pCashflow )

{
    MTG_ASSERT( pCashflow != 0 );

    int nDay = pCashflow->day();
    int k = m_Cashflow.numOfElems();

    ++m_Cashflow;
    while( k > 0 && m_Cashflow[k - 1]->day() > nDay ) {
        m_Cashflow[k] = m_Cashflow[k - 1];
        --k;
    }

    m_Cashflow[k] = pCashflow;
    touch();
}


//
//   m o d i f y M u l t i p l i e r
//

void tClaim::modifyMultiplier( double gMultiplier )

{
    m_gMultiplier = gMultiplier;
    touch();
}


//
//   m o d i f y M a t u r i t y
//

void tClaim::modifyMaturity( tDate MaturityDate )

{
    if( MaturityDate < system().base() )
        MaturityDate = system().base();

    if( MaturityDate != m_MaturityDate ) {
        m_MaturityDate = MaturityDate;
        m_nMaturity = system().dateIndex( m_MaturityDate );
        touch();
    }
}


//
//   m o d i f y U n i t B i d P r i c e
//

void tClaim::modifyUnitBidPrice( double gUnitBidPrice )

{
    m_gUnitBidPrice = gUnitBidPrice;
    touch();
}


//
//   m o d i f y U n i t A s k P r i c e
//

void tClaim::modifyUnitAskPrice( double gUnitAskPrice )

{
    m_gUnitAskPrice = gUnitAskPrice;
    touch();
}


//
//   s e t U n i t P r i c e S h i f t
//

void tClaim::setUnitPriceShift( double gUnitPriceShift )

{
    m_gUnitPriceShift = gUnitPriceShift;
    touch();
}


//
//   s e t S e t t l e m e n t
//

void tClaim::setSettlement( tDate Settlement )

{
    m_Settlement = Settlement;
    touch();
}


//
//   g e t E v e n t s
//

void tClaim::getEvents( tSeqEvGenerator& EvGen ) const

{
    if( birth() != 0 )
        EvGen += birth();

    for( int i = 0; i < m_Cashflow.numOfElems(); ++i ) {
        int t = m_Cashflow[i]->day();
        if( t != birth() && t != maturity() )
            EvGen += t;
    }

    if( maturity() != 0 )
        EvGen += maturity();
}


//
//   g e t B a r r i e r s
//

void tClaim::getBarriers( const tFactor* pFactor, bool bMain,
    tHeap<double>& Barrier ) const

{
        // main == dominant
    if( bMain ) {
            // The derived class may set the barrier flags, but 
            // supply its own barrier, in which case the barriers
            // are < 0.
        if( m_bHasUpBarrier && m_gUpBarrier >= 0 )
            Barrier.append( m_gUpBarrier );
        if( m_bHasDownBarrier && m_gDownBarrier >= 0 )
            Barrier.append( m_gDownBarrier );
    }
}


//
//   p a r s e P r e f i x
//

tRetCode tClaim::parsePrefix( tParser& Parser, tParseInfoStub& Info )

{
    //tParseInfo& I = static_cast<tParseInfo&>( Info );

    m_Settlement = system().base();
    return super::parsePrefix( Parser, Info );
}


//
//   p a r s e P a r a m
//

tRetCode tClaim::parseParam( tParser& Parser, tParseInfoStub& Info )

{
    tRetCode nRet;

    tParseInfo& I = static_cast<tParseInfo&>( Info );

    switch( Parser.curToken() ) {
        case xTokBirth :
        case xTokMaturity :
            if( ( nRet = parseLifetime( Parser ) ) != OK )
                return nRet;
            break;

        case xTokMultiplier :
        case xTokPosition :
        case xTokLong :
        case xTokShort :
            if( ( nRet = parseMultiplier( Parser, I.m_bMultiplier ) ) != OK )
                return nRet;
            break;

        case xTokPrice :
        case xTokBidPrice :
        case xTokAskPrice :
            if( ( nRet = parsePrice( Parser ) ) != OK ) {
                return nRet;
            }
            break;

        case xTokUpAndOut :
        case xTokDownAndOut :
            if( ( nRet = parseBarrier( Parser ) ) != OK )
                return nRet;
            break;

        case xTokComment :
            if( m_sComment != 0 )
                return Parser.setError( ATTR_REDEFINITION );
            if( ( nRet = Parser.readToken() ) != OK ||
                ( nRet = Parser.scanString( m_sComment ) ) != OK ) {
                return nRet;
            }
            break;

        default :
            if( Parser.beginOfNumber() ) {
                if( ( nRet = parseMultiplier( Parser,
                        I.m_bMultiplier ) ) != OK ) {
                    return nRet;
                }
            }
            else {
                return super::parseParam( Parser, Info );
            }
            break;
    }

    return OK;
}


//
//   p a r s e P o s t f i x
//

tRetCode tClaim::parsePostfix( tParser& Parser, tParseInfoStub& Info )

{
    //tParseInfo& I = static_cast<tParseInfo&>( Info );

    if( m_nMaturity < 0 )
        return Parser.setError( MISSING_MATURITY );

    if( m_nBirth >= 0  ) {
        if( m_nBirth > 0 )
            return Parser.setError( "invalid nonzero birth" );
    }
    else {
        m_nBirth = 0;
        m_BirthDate = system().base();
    }

    if( m_gUnitBidPrice >= 0 || m_gUnitAskPrice >= 0 ) {
        if( m_gUnitBidPrice < 0 )
            m_gUnitBidPrice = m_gUnitAskPrice;
        if( m_gUnitAskPrice < 0 )
            m_gUnitAskPrice = m_gUnitBidPrice;
        m_bIsPriced = true;
    }

    return super::parsePostfix( Parser, Info );
}


//
//   t C l a i m
//

tClaim::tClaim()

{
    init();
}


//
//   t C l a i m
//

tClaim::tClaim( const tClaim& Claim )

{
    init();
    copyFrom( Claim );
}


//
//   ~ t C l a i m
//

tClaim::~tClaim()

{
    if( m_sComment != 0 )
        delete m_sComment;
    for( int k = 0; k < m_Cashflow.numOfElems(); ++k )
        delete m_Cashflow[k];
}


//
//   f i n a l i z e
//

tRetCode tClaim::finalize()

{
    int k, n, nDay;
    tAccess A, Null;

    if( isFinalized() )
        return OK;

    k = 0;
    n = m_Cashflow.numOfElems();

    Null.m_nFirst = -1;
    Null.m_nLast = -1;
    nDay = -1;

    while( k < n ) {
        int nDay1 = m_Cashflow[k]->day(); 
        tDate Date1 = m_Cashflow[k]->date();

        if( m_nBirth > nDay1 || m_BirthDate > Date1 ) {
            m_nBirth = nDay1;
            m_BirthDate = Date1;
        }
        if( m_nMaturity < nDay1 || m_MaturityDate < Date1 ) {
            m_nMaturity = nDay1;
            m_MaturityDate = Date1;
        }

        if( nDay < nDay1 - 1 )
            m_Access.append( nDay1 - 1, Null );
        nDay = nDay1;
        A.m_nFirst = k;
        while( k + 1 < n && m_Cashflow[k + 1]->day() == nDay )
            ++k;
        A.m_nLast = k;
        m_Access.append( nDay, A );
        ++k;
    }

    if( nDay < m_nMaturity )
        m_Access.append( m_nMaturity, Null );

#if defined(_DEBUG)
    //MTG_TRACE( "Claim %d: pos %lg, life %d-%d",
    //    m_nId, m_gMultiplier, m_nBirth, m_nMaturity );
    //if( m_bIsPriced ) 
    //    MTG_TRACE( ", price %lg-%lg", m_gUnitBidPrice, m_gUnitAskPrice );
    //if( m_bHasUpBarrier )
    //    MTG_TRACE( ", u&o %lg", m_gUpBarrier );
    //if( m_bHasDownBarrier )
    //    MTG_TRACE( ", d&o %lg", m_gDownBarrier );
    //if( m_bMonitor )
    //    MTG_TRACE( ", mon" );
    //for( k = m_nBirth; k <= m_nMaturity; ++k ) {
    //    tAccess& A = m_Access[k];
    //    if( A.m_nFirst >= 0 )
    //        MTG_TRACE( ", day %d: %d",
    //            k, A.m_nLast - A.m_nFirst + 1 );
    //}
    //MTG_TRACE( "\n" );
#endif

    return super::finalize();
}


//
//   r e s o l v e
//

tRetCode tClaim::resolve( tLookup<int>& Claim,
    tLookup<int>& Factor, tHeap<tFactor*>& FactorPtr, tPortfolio& Pf )

{
    m_bResolved = true;
    return OK;
}


//
//   c o m m e n t
//

const char* tClaim::comment()

{
    if( m_sComment == 0 ) {
        if( isAnonymous() ) {
            char sComment[16];
            sprintf( sComment, "claim%d", m_nId );
            m_sComment = StrCopy( sComment );
        }
        else {
            return name();
        }
    }
    return m_sComment;
}


//
//   p a y o f f
//

double tClaim::payoff( tEngine& Engine )

{
    return 0;
}


//
//   k n o c k o u t P a y o f f
//

double tClaim::knockoutPayoff( tEngine& Engine )

{
    return 0;
}


//
//   e x e r c i s e P a y o f f
//

double tClaim::exercisePayoff( tEngine& Engine )

{
    return payoff( Engine );
}


//
//   u p B a r r i e r
//

bool tClaim::upBarrier( tEngine& Engine, double& gBarrier )

{
    tFDEngine* pFD = dynamic_cast<tFDEngine*>( &Engine );

    MTG_ASSERT( pFD == 0 || pFD->day() >= m_nBirth && pFD->day() <= m_nMaturity );

    if( m_bHasUpBarrier ) {
        MTG_ASSERT( m_gUpBarrier >= 0 );
        gBarrier = m_gUpBarrier;
        return true;
    }
    return false;
}


//
//   d o w n B a r r i e r
//

bool tClaim::downBarrier( tEngine& Engine, double& gBarrier )

{
    tFDEngine* pFD = dynamic_cast<tFDEngine*>( &Engine );

    MTG_ASSERT( pFD == 0 || pFD->day() >= m_nBirth && pFD->day() <= m_nMaturity );

    if( m_bHasDownBarrier ) {
        MTG_ASSERT( m_gDownBarrier >= 0 );
        gBarrier = m_gDownBarrier;
        return true;
    }
    return false;
}


//
//   m o n i t o r
//

tExPolicy tClaim::monitor( tEngine& Engine, double gUnitValue ) 

{
    tFDEngine* pFD = dynamic_cast<tFDEngine*>( &Engine );

    if( pFD == 0 )
        throw tException( NOT_AVAILABLE );

    MTG_ASSERT( pFD->day() >= m_nBirth && pFD->day() <= m_nMaturity );

    return xDontExercise;
}


//
//   h a s C a s h f l o w
//

bool tClaim::hasCashflow( tEngine& Engine )

{
    tFDEngine* pFD = dynamic_cast<tFDEngine*>( &Engine );

    if( pFD != 0 ) {
        MTG_ASSERT( pFD->day() >= m_nBirth && pFD->day() <= m_nMaturity );

        tAccess& A = m_Access[pFD->day()];
        return A.m_nFirst >= 0 && A.m_nFirst <= A.m_nLast;
    }

    tMCEngine* pMC = dynamic_cast<tMCEngine*>( &Engine );

    if( pMC == 0 )
        throw tException( NOT_AVAILABLE );

    return m_Cashflow.numOfElems() > 0;
}


//
//   c a s h f l o w
//

double tClaim::cashflow( tEngine& Engine )

{
    double gCash = 0;

    tFDEngine* pFD = dynamic_cast<tFDEngine*>( &Engine );

    if( pFD != 0 ) {
        MTG_ASSERT( pFD->day() >= m_nBirth && pFD->day() <= m_nMaturity );

        tAccess& A = m_Access[pFD->day()];
 
        if( A.m_nFirst >= 0 ) {
            for( int k = A.m_nFirst; k <= A.m_nLast; ++k )
                gCash += m_Cashflow[k]->generate( Engine, *this );
        }
    }
    else {
        tMCEngine* pMC = dynamic_cast<tMCEngine*>( &Engine );

        if( pMC == 0 )
            throw tException( NOT_AVAILABLE );

        for( int i = 0; i < m_Cashflow.numOfElems(); ++i ) {
            if( m_Cashflow[i]->date() >= pMC->settlement() ) {
                gCash +=
                    Engine.presentValue(
                        pMC->settlement(), m_Cashflow[i]->date() ) *
                    m_Cashflow[i]->generate( Engine, *this );
            }
        }
    }

    return gCash;
}


//
//   a d d P a y m e n t S t r e a m
//

bool tClaim::addPaymentStream( tInterestSpline& Spline ) const

{
    return false;
}


//
//   p a r s e
//

tRetCode tClaim::parse( tParser& Parser, tSystem& System, tObject*& pObj )

{
    tRetCode nRet;
    tClaim* pClaim;
    tToken nType;
    tParseInfo Info;

    Info.m_bMultiplier = false;
    Info.m_bParam1 = false;
    Info.m_bParam2 = false;

    if( ( nRet = parseType( Parser, nType, xTokCustom ) ) != OK )
        return nRet;

    switch( nType ) {
        case xTokCustom :
            pClaim = new tCustomClaim;
            break;

        case xTokEuropeanCall :
            pClaim = new tStdClaim( tStdClaim::xLinearCall, false );
            break;

        case xTokEuropeanPut :
            pClaim = new tStdClaim( tStdClaim::xLinearPut, false );
            break;

        case xTokAmericanCall :
            pClaim = new tStdClaim( tStdClaim::xLinearCall, true );
            break;

        case xTokAmericanPut :
            pClaim = new tStdClaim( tStdClaim::xLinearPut, true );
            break;

        case xTokUSTBond :
            pClaim = new tUSTBond;
            break;

        default :
            return Parser.setError( INVALID_KEYWORD );
    }

    pClaim->setSystem( System );

    if( ( nRet = pClaim->super::parse( Parser, &Info ) ) != OK ) {
        delete pClaim;
        return nRet;
    }
    pObj = pClaim;

    return OK;
}

MTG_END_NAMESPACE
