// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#if ! defined(_MTG_CLAIM_)
#define _MTG_CLAIM_

#include "MtgHeap.h"
#include "MtgLookup.h"
#include "MtgMap.h"
#include "MtgObject.h"

MTG_BEGIN_NAMESPACE

class tCashflow;
class tEngine;
class tFactor;
class tInterestSpline;
class tParser;
class tPortfolio;
class tSeqEvGenerator;


//
//   t C l a i m
//
    
class tClaim : public tObject {

    typedef tObject super;

protected :

    struct tParseInfo : super::tParseInfoStub {
        bool m_bMultiplier;
        bool m_bParam1;         // additional parameters
        bool m_bParam2;         // for subclasses
    };

private :

    struct tAccess {
        int m_nFirst;   // First position in m_aCashflow;
        int m_nLast;    // Last position in m_aCashflow;
    };

        // Caution: when basic instruments are assigned (operator=)
        // or copied (passed by value), new ids are generated for
        // the copies!

    int m_nId;
    static int m_nNextId;

        // The following two values are used for maintenance
        // by a portfolio object that contains the claim
        // (claims are copied by portfolio objects).

    int m_nIndex;            // Index, used for access to
                             // supporting arrays like multipliers
    int m_nSlot;             // Slot, used for finite difference
                             // methods

    int m_nBirth;            // Default: 0
    tDate m_BirthDate;

    int m_nMaturity;         // Default: 0
    tDate m_MaturityDate;

        // The price is set via "price" attributes. How to
        // interpret it is up to the instruments; if necessary,
        // an adjustment can be made in parseFinish() by calling
        // modifyUnitBidPrice() and modifyUnitAskPrice(), After
        // finalization, however, the price must be the dollar
        // price of the instrument!

    bool m_bIsPriced;           // Default: false
    double m_gUnitBidPrice;     // Default: 0
    double m_gUnitAskPrice;     // Default: 0

        // Sometimes, the computed price must be shifted to
        // a higher or lower value (for instance, to switch
        // from the - computed - dirty price to the clean
        // price). This constant must be set by the subclass
        // with setPriceShitft(), in addition to modification
        // to the unit bid and ask price.

    double m_gUnitPriceShift;   // Default: 0

        // Also, priced instruments must provide a settlement
        // date on which the price of the instrument is exchanged.
        // This date is set by setSettlement(). Not necessary
        // for instruments which are not priced.

    tDate m_Settlement;         // Default: system().base()

    double m_gMultiplier;       // Default: 1 

    bool m_bHasUpBarrier;       // Default: false
    double m_gUpBarrier;        // Default: 0

    bool m_bHasDownBarrier;     // Default: false
    double m_gDownBarrier;      // Default: 0

    bool m_bMonitor;            // Default: false

    bool m_bResolved;

    char* m_sComment;

    void init();

    tRetCode parseLifetime( tParser& Parser );
    tRetCode parseMultiplier( tParser& Parser, bool& bMultiplier );     
    tRetCode parsePrice( tParser& Parser );
    tRetCode parseBarrier( tParser& Parser );        

        // The portfolio calls these functions:

    void setIndex( int nIndex );
    void setSlot( int nSlot );

protected:

    tHeap<tCashflow*> m_Cashflow;
    tMap<tAccess,3,10> m_Access;
   
    void copyFrom( const tClaim& Claim );

        // Tools for subclasses to change come variables. These
        // functions must be called after parsePostfix()  has
        // completed. A good place is finalize().

    void setMonitor();
    void setUpBarrier( double gBarrier = -1 );
    void setDownBarrier( double gBarrier = -1 );
    void regCashflow( tCashflow *pCashflow );

    void modifyMultiplier( double gMultiplier );
    void modifyMaturity( tDate MaturityDate );
    void modifyUnitBidPrice( double gUnitBidPrice );
    void modifyUnitAskPrice( double gUnitAskPrice );

    void setUnitPriceShift( double gUnitPriceShift );
    void setSettlement( tDate Settlement );

    void touch() {
        super::touch();
        m_bResolved = false;
    }

    bool isResolved() const {
        return m_bResolved;
    }

    virtual void getEvents( tSeqEvGenerator& EvGen ) const;

    virtual void getBarriers( const tFactor* pFactor, bool bMain,
        tHeap<double>& Barrier ) const;

    tRetCode parsePrefix( tParser& Parser, tParseInfoStub& Info );
    tRetCode parseParam( tParser& Parser, tParseInfoStub& Info );
    tRetCode parsePostfix( tParser& Parser, tParseInfoStub& Info );

public:

    tClaim();
    tClaim( const tClaim& Claim );

    virtual ~tClaim();

    tRetCode finalize();

    virtual tRetCode resolve( tLookup<int>& Claim,
        tLookup<int>& Factor, tHeap<tFactor*>& FactorPtr, tPortfolio& Pf );

    bool isAuxiliary() const      { return false; }
    bool isPriced() const         { return m_bIsPriced; }
    bool isMonitored() const      { return m_bMonitor; }

    bool hasUpBarrier() const     { return m_bHasUpBarrier; }
    bool hasDownBarrier() const   { return m_bHasDownBarrier; }

    int id() const                { return m_nId; }
    int birth() const             { return m_nBirth; }
    tDate birthDate() const       { return m_BirthDate; }
    int maturity() const          { return m_nMaturity; }
    tDate maturityDate() const    { return m_MaturityDate; }
    double unitAskPrice() const   { return m_gUnitAskPrice; }
    double unitBidPrice() const   { return m_gUnitBidPrice; }
    double unitPriceShift() const { return m_gUnitPriceShift; }
    tDate settlement() const      { return m_Settlement; }
    double multiplier() const     { return m_gMultiplier; }

    int index() const             { return m_nIndex; }
    int slow() const              { return m_nSlot; }

    const char* comment();

        // There are three kinds of payoff: payoff at maturity, 
        // payoff at knockout (usually 0), and payoff at early
        // exercise. Only the payoff at maturity has to be
        // specified; the other two are optional.

        // NOTE THAT IN PAYOFF AND CASHFLOW FUNCTIONS, THE PAYOFF
        // AND CASHFLOW IS PER UNIT.

        // These functions are expected to throw exceptions if
        // anything goes wrong.

    virtual double payoff( tEngine& Engine );
    virtual double knockoutPayoff( tEngine& Engine );
    virtual double exercisePayoff( tEngine& Engine );

        // The following functions return barriers for the dominant
        // factor, if existent.

    virtual bool upBarrier( tEngine& Engine, double& gBarrier );
    virtual bool downBarrier( tEngine& Engine, double& gBarrier );

        // If m_bMonitor is set, the following function is called to 
        // determine the local exercise policy. This is the only place
        // in which a decision may depend on the current value of the
        // instrument. (It is not called at maturity.)

    virtual tExPolicy monitor( tEngine& Engine, double gUnitValue );

        // Finally, the following functions return whatever
        // cashflow might arise. For finite difference methods,
        // the function cashflow() is potentially called for every
        // note (not at maturity, however). For Monte Carlo,
        // cashflow() should return the cashflow over the entire
        // lifetime of the instrument, PROPERLY DISCOUNTED TO
        // Engine.base()!

    bool hasCashflow( tEngine& Engine );
    double cashflow( tEngine& Engine );

        // Sometimes claims can be used to come up with an initial
        // forward rate curve (with the bootstrapping method, for
        // instance). The following functions checks whether this
        // is possible for the claim and creates the payment stream,
        // if possible.

    virtual bool addPaymentStream( tInterestSpline& Spline ) const;

    static tRetCode parse( tParser& Parser, tSystem& System, tObject*& pObj );
        
    friend class tPortfolio;
};

MTG_END_NAMESPACE

#endif
