// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#if ! defined(_MTG_COMPOUNDER_)
#define _MTG_COMPOUNDER_

#include "MtgDayCount.h"
#include "MtgDayShift.h"
#include "MtgHeap.h"

MTG_BEGIN_NAMESPACE


//
//   t C o m p o u n d e r
//

class tCompounder {

        // "Left open" and "Right open" refer to the boundaries of
        // compounding periods. "Left open" means a compounding date
        // belongs to the earlier compounding period. "Right open"
        // means a compounding date belongs to the later compounding
        // period.

        // The default in virtually all cases is "right open", i.e. false.

    bool m_bLeftOpen;

    void init();

protected:

    void copyFrom( const tCompounder& Compounder );

    bool leftOpen() const {
        return m_bLeftOpen;
    }

    bool rightOpen() const {
        return ! m_bLeftOpen;
    }

    void setLeftOpen();
    void setRightOpen();

public:

    tCompounder();
    tCompounder( const tCompounder& Compounder );

    virtual ~tCompounder();

    virtual tCompounder* clone() const = 0;

        // The number of compounding periods per year is the most
        // important information which the compounder must always
        // provide (this is the only information which the period
        // compounder, see below, provides). 0 or negativ means
        // "continuous compounding."

    virtual int numOfPeriodsPerYear() const;

    virtual bool isCompoundingDate( tDate Date ) const = 0;

    virtual int numOfCompoundingDates( tDate Start, tDate End ) const = 0;

    virtual void getCompoundingPeriod( tDate Date,
        tDate& Start, tDate& End ) const = 0;

        // Iterate through compounding periods.

    virtual void getNextCompoundingPeriod( tDate& Start,
        tDate& End ) const = 0;

        // Sometimes the day count fraction is based on a year, and
        // sometimes on the length of the underlying accrual period,
        // which can be shorter than a year. To normalize yield and
        // fraction, the following function supplies the appropriate
        // multiplicative factors. It returns false if the factors
        // cannot be computed for some reason.

    virtual bool getNormFactor( const tDayCount& DayCount,
        double& gYieldFactor, double& gFractionFactor ) const;

        // While getNormFactor() returns a fraction factor
        // which scales day count fractions to a period, the
        // following function returns a fraction factor that
        // scales day count fractions to a year. It basically
        // divides by the number of periods per year.

    virtual bool getYearFactor( const tDayCount& DayCount,
        double& gFractionFactor ) const;
};


//
//   t P e r i o d C o m p o u n d e r
//

class tPeriodCompounder : public tCompounder {

    typedef tCompounder super;

        // A period compounder does NOT contain a sequence
        // of actual compounding days, but only the number
        // of compounding events that occur in the course
        // of a year. The period compounder is used only
        // in a limited number of situations.

    int m_nNumOfPeriods;

    void init( int nNumOfPeriods );

protected:

    void copyFrom( const tPeriodCompounder& Compounder );

public:

    tPeriodCompounder();
    tPeriodCompounder( int nNumOfPeriods );
    tPeriodCompounder( const tPeriodCompounder& Compounder );

    ~tPeriodCompounder();

    tPeriodCompounder& operator=( const tPeriodCompounder& Compounder );

    tCompounder* clone() const;

    void set( int nNumOfPeriods );

    int numOfPeriodsPerYear() const;

        // These functions lead to a NOT_IMPLEMENTED exception:

    bool isCompoundingDate( tDate Date ) const;
    int numOfCompoundingDates( tDate Start, tDate End ) const;
    void getCompoundingPeriod( tDate Date, tDate& Start, tDate& End ) const;
    void getNextCompoundingPeriod( tDate& Start, tDate& End ) const;
};


//
//   t C y c l i c C o m p o u n d e r
//

class tCyclicCompounder : public tCompounder {

    typedef tCompounder super;

        // A cyclic compounder is a full-featured compounder
        // which generates actual compounding dates based
        // on date templates. The number of date templates
        // determines the number of compounding periods per
        // year..

    struct tDateTemplate {
        int m_nDay;             // day of compounding
        int m_nMonth;           // month of compounding
        tDayShift *m_pShift;    // massage the instantiated
    };                          // date 

    tHeap<tDateTemplate> m_DateTemplate;

    void init();

    int nextCompoundingDate( const tDate& Date, tDate& Compounding ) const;
    tDate prepCompoundingDate( int nIndex, int nYear ) const;

protected:

    void copyFrom( const tCyclicCompounder& Compounder );
    void resetDateTemplates();

public:

    tCyclicCompounder();
    tCyclicCompounder( const tCyclicCompounder& Compounder );

    ~tCyclicCompounder();

    tCyclicCompounder& operator=( const tCyclicCompounder& Compounder );

    tCompounder* clone() const;

    int numOfPeriodsPerYear() const;

        // A valid date template has a month value between 1 and 12
        // and a day value between 1 and 31 (for all months!). When 
        // the actual compounding day is generated, days 31, 30 and 29
        // are reduced if they don't exist.

    bool isValidDateTemplate( int nDay, int nMonth ) const;

        // When the object is initialized, date templates are
        // registered with the following function.

    bool addDateTemplate( int nDay, int nMonth );
    bool addDateTemplate( int nDay, int nMonth,
        const tDayShift& Shift );

    bool isCompoundingDate( tDate Date ) const;
    int numOfCompoundingDates( tDate Start, tDate End ) const;
    void getCompoundingPeriod( tDate Date, tDate& Start, tDate& End ) const;
    void getNextCompoundingPeriod( tDate& Start, tDate& End ) const;        
};


//
//   t C o u p o n C o m p o u n d e r
//

class tCouponCompounder : public tCyclicCompounder {

    typedef tCyclicCompounder super;

        // A coupon compounder is basically a cyclic compounder
        // which generates the date templates automatically
        // from the maturity date (and a specific coupon date).

        // It has a "reference" coupon date and a maturity.
        // Note that if the "reference" coupon date is the
        // last day of the month, all derived coupon dates
        // will also automatically be the last day of their
        // respective months.

        // If no reference coupon date is defined, the maturity
        // date is the reference coupon date. (The reference
        // coupon date can differ for add-last-coupon securities,
        // for instance.)

    tDate m_Coupon;
    tDate m_Maturity;

        // m_Maturity and/or some coupon dates might fall on
        // a non-businessday. Should they be advanced to the
        // next businessday? The default is NO, but this or
        // many other policies are possible:

    tDayShift* m_pShift;

    int m_nNumOfPeriods;

    void init();
    void setDateTemplates();

protected:

    void copyFrom( const tCouponCompounder& Compounder );

public:

    tCouponCompounder();
    tCouponCompounder( tDate Maturity, int nNumOfPeriods = 1 );
    tCouponCompounder( tDate Maturity, const tDayShift& Shift,
        int nNumOfPeriods = 1 );
    tCouponCompounder( tDate Maturity, tDate Coupon, 
        int nNumOfPeriods = 1 );
    tCouponCompounder( tDate Maturity, tDate Coupon, 
        const tDayShift& Shift, int nNumOfPeriods = 1 );
    tCouponCompounder( const tCouponCompounder& Compounder );

    ~tCouponCompounder();

    tCouponCompounder& operator=( const tCouponCompounder& Compounder );

    tCompounder* clone() const;

        // The number of periods can be 1, 2, 3, 4, 6 or 12.
        // Everything else will lead to an exception.

    void set( tDate Maturity, int nNumOfPeriods = 1 );
    void set( tDate Maturity, const tDayShift& Shift,
        int nNumOfPeriods = 1 );
    void set( tDate Maturity, tDate Coupon, int nNumOfPeriods = 1 );
    void set( tDate Maturity, tDate Coupon, 
        const tDayShift& Shift, int nNumOfPeriods = 1 );

        // The maturity can be queried. Note that m_bBusinessday is true,
        // the value returned can be different from the one set in
        // the constructor or the set() functions.

    tDate maturity() const {
        return m_Maturity;
    }

        // The following functions retain their functionality except
        // that there is no compounding date after maturity.
        // Note that if the maturity is not covered by the date
        // templates, the maturity is not a compounding date!
        // (This can happen if an explicit reference coupon date
        // is specified.)

    bool isCompoundingDate( tDate Date ) const;
    int numOfCompoundingDates( tDate Start, tDate End ) const;        

        // Note that getCompoundingPeriod() and getNextCompoundingPeriod()
        // remain unaltered and return compounding periods BEYOND the
        // maturity, i.e. virtual compounding periods. This is important
        // because for odd-last-coupon instruments, the first virtual
        // compounding period after maturity must be known.
};

MTG_END_NAMESPACE

#endif
