// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#include "MtgIncl.h"
#include "MtgCurve.h"
#include "MtgClaim.h"
#include "MtgCurveInstance.h"
#include "MtgDataScanner.h"
#include "MtgForwardCurve.h"
#include "MtgPathSpace.h"
#include "MtgPortfolio.h"
#include "MtgYieldCurve.h"

MTG_BEGIN_NAMESPACE


//
//   i n i t
//

void tCurve::init()

{
    m_pPathSpace = 0;
    m_pPortfolio = 0;
    m_nNumOfGnuBins = 0;
    m_bHasMaturity = false;
    m_gCutoffYear = 0;
    m_nCutoffClaim = 0;
}


//
//   s a v e G n u T i c s
//

tRetCode tCurve::saveGnuTics( ofstream& Out, int nNumOfBins ) const

{
    MTG_ASSERT( nNumOfBins > 0 );

    static const char* M[] = {
        "", "Jan", "Feb", "Mar", "Apr", "May", "Jun",
            "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"
    };

    Out.fill( '0' );
    Out << "set xtics (\"" << M[base().month()] << " "
        << setw( 2 ) << base().year() % 100 << "\" 0";

    double gWidth = (double) numOfSamples() / nNumOfBins;

    for( int k = 1; k < nNumOfBins; ++k ) {
        tDate D = getClosestDateSI( gWidth * k );

        Out << ", \"" << M[D.month()] << " "
            << setw( 2 ) << D.year() % 100 << "\" " << gWidth * k;
    }

        // we could be farther than m_Maturity says

    tDate D = getClosestDateSI( numOfSamples() );

    if( dayCount().fraction( base(), D ) >= 9.5 ) {
        Out << ", \"" << D.year() << "\" "
            << numOfSamples() << ")" << endl;
    }
    else {
        Out << ", \"" << M[D.month()] << " "
            << setw( 2 ) << D.year() % 100 << "\" "
            << numOfSamples() << ")" << endl;
    }

    return Out.good() ? OK : WRITE_ERROR;
}


//
//   s a v e G n u T i c s
//

tRetCode tCurve::saveGnuTics( ofstream& Out, int nNumOfBins,
    const tPortfolio& Portfolio ) const

{
    MTG_ASSERT( nNumOfBins > 0 );

    static const char* M[] = {
        "", "Jan", "Feb", "Mar", "Apr", "May", "Jun",
            "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"
    };

    double gWidth = (double) numOfSamples() / nNumOfBins;
    double gLastSub = 0;

    Out.fill( '0' );
    Out << "set xtics (\"" << M[base().month()] << " "
        << setw( 2 ) << base().year() % 100 << "\" 0";

        // claims are sorted by maturity, descending

    gWidth *= 0.75;

        // we could be farther than m_Maturity says

    tDate Maturity = getClosestDateSI( numOfSamples() );

    for( int k = Portfolio.numOfClaims() - 1; k >= 0; --k ) {
        tDate D = Portfolio.claim( k ).maturityDate();

        if( D > base() && D < Maturity ) {
            double gSub = getSI( D );

                // make sure texts don't overlap
    
            if( gSub >= gLastSub + gWidth &&
                gSub <= numOfSamples() - gWidth ) {
                gLastSub = gSub;
                Out << ", \"" << M[D.month()] << " "
                    << setw( 2 ) << D.year() % 100 << "\" " << gSub;
            }
            else {
                Out << ", \"\" " << gSub;
            }
        }
    }

    if( dayCount().fraction( base(), Maturity ) >= 9.5 ) {
        Out << ", \"" << Maturity.year() << "\" "
            << numOfSamples() << ")" << endl;
    }
    else {
        Out << ", \"" << M[Maturity.month()] << " "
            << setw( 2 ) << Maturity.year() % 100 << "\" "
            << numOfSamples() << ")" << endl;
    }

    return Out.good() ? OK : WRITE_ERROR;
}


//
//   a u t o S a v e
//

tRetCode tCurve::autoSave( const tHeap<double>& Sample ) const

{
    MTG_ASSERT( isFinalized() );
    MTG_ASSERT( Sample.numOfElems() == numOfSamples() );
                
    tRetCode nRet;

        // output samples only

    if( m_OutSampleFile.isValid() ) {
        ofstream Out;

        if( ( nRet = m_OutSampleFile.openWrite( Out,
                        "curve samples" ) ) != OK ) {
            return nRet;
        }

        if( ( nRet = Sample.save( Out, 1 ) ) != OK )
            return nRet;

        if( ! Out.good() )
            return WRITE_ERROR;
    }

        // output samples as drift

    if( m_OutDriftFile.isValid() ) {
        ofstream Out;

        if( ( nRet = m_OutDriftFile.openWrite( Out ) ) != OK )
            return nRet;

        Out << "drift " << name() << " {\n"
            << "    "
            << static_cast<const tDateRange&>( *this ) << ",\n";

        Out << setprecision( DBL_DIG );
        for( int k = 0; k < Sample.numOfElems(); ++k )
            Out << "    forward " << k + 1 << " " << Sample[k] << ",\n";
        Out << "}" << endl;

        if( ! Out.good() )
            return WRITE_ERROR;
    }

        // output samples for Gnuplot

    if( m_OutGnuDataFile.isValid() ) {
        ofstream Out;

        if( ( nRet = m_OutGnuDataFile.openWrite( Out ) ) != OK )
            return nRet;

            // for Gnuplot, we multiply the samples by 100

        if( Sample.numOfElems() > 0 ) {
            for( int k = 0; k < Sample.numOfElems(); ++k )
                Out << k << " " << 100 * Sample[k] << "\n";

                // extrapolate last point (has no effect for step
                // plots, only for line plots)
            if( Sample.numOfElems() == 1 ) {
                Out << Sample.numOfElems() << " "
                    << 100 * Sample.last() << endl;
            }
            else {
                double d = Sample.last() - Sample[Sample.numOfElems() - 2];

                Out << Sample.numOfElems() << " "
                    << 100 * ( Sample.last() + d ) << endl;
            }
        }

        if( ! Out.good() )
            return WRITE_ERROR;
    }

    if( m_OutGnuTicsFile.isValid() ) {
        ofstream Out;

        if( ( nRet = m_OutGnuTicsFile.openWrite( Out ) ) != OK )
            return nRet;

        if( m_nNumOfGnuBins <= 0 ) {
            int n = -m_nNumOfGnuBins;

            if( n == 0 )
                n = 6;

            if( m_pPortfolio != 0 )
                nRet = saveGnuTics( Out, n, *m_pPortfolio );
            else
            if( m_pPathSpace != 0 && m_pPathSpace->hasPortfolio() )
                nRet = saveGnuTics( Out, n, m_pPathSpace->portfolio() );
            else
                nRet = saveGnuTics( Out, n );
        }
        else {
            nRet = saveGnuTics( Out, m_nNumOfGnuBins );
        }

        if( nRet != OK )
            return nRet;
    }

    return OK;
}


//
//   p a r s e S a m p l e
//

tRetCode tCurve::parseSample( tParser& Parser )

{
    MTG_ASSERT( Parser.curToken() == xTokSample );

    tRetCode nRet;

    if( m_InSampleFile.isValid() || numOfSamples() > 0 )
        return Parser.setError( ATTR_REDEFINITION );

    if( ( nRet = Parser.readToken() ) != OK )
        return nRet;

    if( Parser.beginOfObj() ) {
        if( ( nRet = Parser.scanTable( m_Sample ) ) != OK )
            return nRet;
        setNumOfSamples( m_Sample.numOfElems() );
        if( numOfSamples() == 0 )
            return Parser.setError( MISSING_SAMPLES );
    } 
    else {
        if( ( nRet = m_InSampleFile.parseRead( Parser ) ) != OK )
            return nRet;

        tDataScanner Scanner;

        if( ( nRet = Scanner.setSource( m_InSampleFile ) ) != OK ||
            ( nRet = Scanner.append( m_Sample ) ) != OK ) {
            setNumOfSamples( m_Sample.numOfElems() );
            if( nRet != GO )
                return Parser.setError( nRet );
        }
        else {
            setNumOfSamples( m_Sample.numOfElems() );
            if( ! Scanner.atEOF() )
                return INVALID_CONSTANT;
            if( numOfSamples() == 0 && ! m_InSampleFile.isTry() )
                return MISSING_SAMPLES;
        }
    }

    return OK;
}


//
//   p a r s e S a v e
//

tRetCode tCurve::parseSave( tParser& Parser )

{
    MTG_ASSERT( Parser.curToken() == xTokSave );

    tRetCode nRet;

    if( ( nRet = Parser.readToken() ) != OK )
        return nRet;

    switch( Parser.curToken() ) {
        case xTokSample :
            if( m_OutSampleFile.isValid() )
                return Parser.setError( ATTR_REDEFINITION );
            if( ( nRet = Parser.readToken() ) != OK ||
                ( nRet = m_OutSampleFile.parseWrite( Parser ) ) != OK ) {                    
                return nRet;
            }
            break;

        case xTokDrift :
            if( m_OutDriftFile.isValid() )
                return Parser.setError( ATTR_REDEFINITION );
            if( ( nRet = Parser.readToken() ) != OK ||
                ( nRet = m_OutDriftFile.parseWrite( Parser ) ) != OK ) {
                return nRet;
            }
            break;

        case xTokGnuData :
            if( m_OutGnuDataFile.isValid() )
                return Parser.setError( ATTR_REDEFINITION );
            if( ( nRet = Parser.readToken() ) != OK ||
                ( nRet = m_OutGnuDataFile.parseWrite( Parser ) ) != OK ) {
                return nRet;
            }
            break;

        case xTokGnuTics :
            if( m_OutGnuTicsFile.isValid() )
                return Parser.setError( ATTR_REDEFINITION );
            if( ( nRet = Parser.readToken() ) != OK ||
                ( nRet = m_OutGnuTicsFile.parseWrite( Parser ) ) != OK ) {
                return nRet;
            }
            if( Parser.beginOfNumber() ) {
                if( ( nRet = Parser.scanInteger(
                        m_nNumOfGnuBins, -100 ) ) != OK ) {
                    return nRet;
                }
            }
            else {
                m_nNumOfGnuBins = -1;
            }
            break;

        default :
            if( m_OutFile.isValid() )
                return Parser.setError( ATTR_REDEFINITION );
            if( Parser.curToken() == xTokCurve ) {
                if( ( nRet = Parser.readToken() ) != OK )
                    return nRet;
            }
            if( ( nRet = m_OutFile.parseWrite( Parser ) ) != OK )
                return nRet;
            break;
    }

    return OK;
}


//
//   p a r s e M a t u r i t y
//

tRetCode tCurve::parseCutoff( tParser& Parser )

{
    MTG_ASSERT( Parser.curToken() == xTokCutoff );

    tRetCode nRet;

    if( ( nRet = Parser.readToken() ) != OK )
        return nRet;

    switch( Parser.curToken() ) {
        case xTokClaim :
            if( m_nCutoffClaim != 0 )
                return Parser.setError( ATTR_REDEFINITION );
            if( ( nRet = Parser.readToken() ) != OK ||
                ( nRet = Parser.scanInteger(
                    m_nCutoffClaim, INT_MIN ) ) != OK ) {
                return nRet;
            }
            if( m_nCutoffClaim == 0 )
                return Parser.setError( INVALID_CONSTANT );
            break;

        case xTokYear :
            if( m_gCutoffYear != 0 )
                return Parser.setError( ATTR_REDEFINITION );
            if( ( nRet = Parser.readToken() ) != OK ||
                ( nRet = Parser.scanPosDouble( m_gCutoffYear ) ) != OK ) {
                return nRet;
            }
            break;

        default :
            return Parser.setError( INVALID_KEYWORD );
    }

    return OK;
}


//
//   c o p y F r o m
//

void tCurve::copyFrom( const tCurve& Curve )

{
    if( &Curve == this )
        return;

    m_Sample = Curve.m_Sample;
    m_InSampleFile = Curve.m_InSampleFile;
    m_OutSampleFile = Curve.m_OutSampleFile;
    m_OutDriftFile = Curve.m_OutDriftFile;
    m_OutGnuDataFile = Curve.m_OutGnuDataFile;
    m_OutGnuTicsFile = Curve.m_OutGnuTicsFile;
    m_nNumOfGnuBins = Curve.m_nNumOfGnuBins;
    m_bHasMaturity = Curve.m_bHasMaturity;
    m_Maturity = Curve.m_Maturity;
    m_gCutoffYear = Curve.m_gCutoffYear;
    m_nCutoffClaim = Curve.m_nCutoffClaim;
    m_OutFile = Curve.m_OutFile;
    
    setObjRefToZeroOrSo( m_pPathSpace, Curve.m_pPathSpace );
    setObjRefToZeroOrSo( m_pPortfolio, Curve.m_pPortfolio );

    super::copyFrom( Curve );
    tDateRange::copyFrom( Curve );
}


//
//   p a r s e P r e f i x
//

tRetCode tCurve::parsePrefix( tParser& Parser, tParseInfoStub& Info )

{
    setBase( system().base() );
    setDayCount( system().dayCount() );

    return super::parsePrefix( Parser, Info );
}


//
//   p a r s e P a r a m
//

tRetCode tCurve::parseParam( tParser& Parser, tParseInfoStub& Info )

{
    tRetCode nRet;
    tObject* pObj;
    tPathSpace* pPathSpace;
    tPortfolio* pPortfolio;
    tDayCount DayCount;

    tParseInfo& I = static_cast<tParseInfo&>( Info );

    nRet = DayCount.parse( Parser );

    if( nRet == OK ) {
        if( I.m_bDayCount ) 
            return Parser.setError( ATTR_REDEFINITION );
        setDayCount( DayCount );
        I.m_bDayCount = true;
        return nRet;
    }

    if( nRet != GO )
        return nRet;

    switch( Parser.curToken() ) {
        case xTokBase :
            if( I.m_bBase )
                return Parser.setError( ATTR_REDEFINITION );
            if( ( nRet = tDateRange::parse( Parser ) ) != OK )
                return nRet;
            I.m_bBase = true;
            break;

        case xTokScale :
        case xTokYear :
        case xTokDay :
            if( I.m_bScale )
                return Parser.setError( ATTR_REDEFINITION );
            if( ( nRet = tDateRange::parse( Parser ) ) != OK )
                return nRet;
            I.m_bScale = true;            
            break;

        case xTokPathSpace :
            if( m_pPathSpace != 0 )
                return Parser.setError( ATTR_REDEFINITION );
            if( ( nRet = Parser.readToken() ) != OK )
                return nRet;
            if( Parser.curToken() != xTokId )
                return Parser.setError( "path space expected" );
            if( ( pObj = Parser.findObject() ) == 0 )
                return Parser.setError( NOT_FOUND );
            if( ( pPathSpace = dynamic_cast<tPathSpace*>( pObj ) ) == 0 ) 
                return Parser.setError( "path space expected" );
            setObjRefToZeroOrSo( m_pPathSpace, pPathSpace );
            if( ( nRet = Parser.readToken() ) != OK )
                return nRet;
            break;

        case xTokPortfolio :
            if( m_pPortfolio != 0 )
                return Parser.setError( ATTR_REDEFINITION );
            if( ( nRet = Parser.readToken() ) != OK )
                return nRet;
            if( Parser.curToken() != xTokId )
                return Parser.setError( "portfolio expected" );
            if( ( pObj = Parser.findObject() ) == 0 )
                return Parser.setError( NOT_FOUND );
            if( ( pPortfolio = dynamic_cast<tPortfolio*>( pObj ) ) == 0 ) 
                return Parser.setError( "portfolio expected" );
            setObjRefToZeroOrSo( m_pPortfolio, pPortfolio );
            if( ( nRet = Parser.readToken() ) != OK )
                return nRet;
            break;

        case xTokMaturity :
            if( m_bHasMaturity )
                return Parser.setError( ATTR_REDEFINITION );
            if( ( nRet = Parser.readToken() ) != OK ||
                ( nRet = Parser.scanDate( m_Maturity ) ) != OK ) {
                return nRet;
            }
            m_bHasMaturity = true;
            break;

        case xTokCutoff :
            if( ( nRet = parseCutoff( Parser ) ) != OK )
                return nRet;
            break;

        case xTokSample :
            if( ( nRet = parseSample( Parser ) ) != OK )
                return nRet;
            break;

        case xTokSave :
            if( ( nRet = parseSave( Parser ) ) != OK )
                return nRet;
            break;

        default :
            if( ( nRet = super::parseParam( Parser, Info ) ) != OK )
                return nRet;
    }

    return OK;
}


//
//   p a r s e P o s t f i x
//

tRetCode tCurve::parsePostfix( tParser& Parser, tParseInfoStub& Info )

{
    tParseInfo& I = static_cast<tParseInfo&>( Info );

    if( ! I.m_bBase ) {
            // take the one of the path space
        if( m_pPathSpace != 0 )
            setBase( m_pPathSpace->base() );
    }

    if( ! I.m_bScale ) {
            // take the one of the path space
        if( m_pPathSpace != 0 )
            setScale( m_pPathSpace->scale(), m_pPathSpace->numOfPeriods() );
    }

    if( ! I.m_bDayCount ) {
            // take the one of the path space
        if( m_pPathSpace != 0 )
            setDayCount( m_pPathSpace->dayCount() );
    }

    tDate CutDate;
    bool bCut = false;

    if( m_nCutoffClaim != 0 ) {
        if( m_pPortfolio == 0 )
            return Parser.setError( MISSING_PORTFOLIO );

        int n = m_pPortfolio->numOfClaims();
        bCut = true;

            // cut from front or back

        if( m_nCutoffClaim > 0 ) {
            if( n >= m_nCutoffClaim ) {
                CutDate =
                    m_pPortfolio->claim( n - m_nCutoffClaim ).maturityDate();
            }
            else {
                bCut = false;
            }
        }
        else {
            if( n > -m_nCutoffClaim ) {
                CutDate =
                    m_pPortfolio->claim( -m_nCutoffClaim ).maturityDate();
            }
            else {
                CutDate = base();
            }
        }
    }

    if( m_gCutoffYear > 0 ) {
        tDate M = getClosestDateSI( m_gCutoffYear / dtSI() );

        if( ! bCut || M < CutDate )
            CutDate = M;
        bCut = true;
    }

        // find maturity if possible

    if( ! m_bHasMaturity ) {
        if( m_pPathSpace != 0 && numOfSamples() == 0 ) {
            m_Maturity = m_pPathSpace->maturity();
        }
        else 
        if( m_pPortfolio != 0 && numOfSamples() == 0 ) {
            m_Maturity = m_pPortfolio->maturityDate();
        }
        else
        if( bCut ) {
            if( numOfSamples() > 0 ) {
                tDate M = getClosestDateSI( numOfSamples() );

                if( M < CutDate )
                    CutDate = M;
            }
            m_Maturity = CutDate;
        }
        else
        if( numOfSamples() == 0 ) {
            return Parser.setError( MISSING_MATURITY );
        }
        m_bHasMaturity = true;
    }

    if( bCut && m_Maturity > CutDate )
        m_Maturity = CutDate;

    if( m_Maturity < base() )
        return Parser.setError( INVALID_MATURITY );

    return super::parsePostfix( Parser, Info );
}


//
//   t C u r v e
//

tCurve::tCurve()

{
    init();
}


//
//   t C u r v e
//
    
tCurve::tCurve( tScale nScale )
    : tDateRange( nScale )

{
    init();
}


//
//   t C u r v e
//

tCurve::tCurve( tScale nScale, int nNumOfPeriods )
    : tDateRange( nScale, nNumOfPeriods )

{
    init();
}


//
//   t C u r v e
//

tCurve::tCurve( tScale nScale, const tDayCount& DayCount )
    : tDateRange( nScale, DayCount )

{
    init();
}


//
//   t C u r v e
//

tCurve::tCurve( tScale nScale, int nNumOfPeriods, const tDayCount& DayCount )
    : tDateRange( nScale, nNumOfPeriods, DayCount )

{
    init();
}


//
//   t C u r v e
//

tCurve::tCurve( tScale nScale, const tDayCount& DayCount, tDate Base )
    : tDateRange( nScale, DayCount, Base )

{
    init();
}


//
//   t C u r v e
//

tCurve::tCurve( tScale nScale, int nNumOfPeriods,
    const tDayCount& DayCount, tDate Base )
    : tDateRange( nScale, nNumOfPeriods, DayCount, Base )

{
    init();
}


//
//   t C u r v e
//

tCurve::tCurve( const tCurve& Curve )

{
    init();
    copyFrom( Curve );
}


//
//   ~ t C u r v e
//

tCurve::~tCurve()

{
    setObjRefToZeroOrSo( m_pPathSpace );
    setObjRefToZeroOrSo( m_pPortfolio );
}


//
//   f i n a l i z e
//

tRetCode tCurve::finalize()

{
    int n = numOfSamples();
    
    if( m_bHasMaturity ) {
        setNumOfSamples( m_Maturity );
        m_Sample.numOfElems( numOfSamples() );   
    }

    if( n < numOfSamples() ) {
        double g = ( n == 0 ) ? 0 : m_Sample[n - 1];
        for( int i = n; i < numOfSamples(); ++i )
            m_Sample[i] = g;
    }

    return super::finalize();
}


//
//   m o d i f y S a m p l e
//

tRetCode tCurve::modifySample( const tHeap<double>& Sample )

{
    MTG_ASSERT( isFinalized() );
    MTG_ASSERT( Sample.numOfElems() == numOfSamples() );

    m_Sample = Sample;
    return OK;
}


//
//   m o d i f y S a m p l e
//

tRetCode tCurve::modifySample( const tCurveInstance& Curve )

{
    return modifySample( Curve.sample() );
}


//
//   s a v e
//

tRetCode tCurve::save( ostream& Out, const tHeap<double>* pSample ) const

{
    MTG_ASSERT( isFinalized() );
    MTG_ASSERT( pSample == 0 ||
                pSample->numOfElems() == numOfSamples() );

    tRetCode nRet;

    if( pSample == 0 )
        pSample = &m_Sample;

    Out << "curve " << name() << " {\n"
        << "    "
        << static_cast<const tDateRange&>( *this ) << "," << endl;

    if( pSample->numOfElems() == 0 && m_bHasMaturity )
        Out << "    maturity \"" << m_Maturity << "\"," << endl;

    if( pSample->numOfElems() > 0 ) {
        Out << "    sample {\n" << setprecision( DBL_DIG );
        if( ( nRet = pSample->save( Out, 4, "        " ) ) != OK )
            return nRet;
        Out << "    }" << endl;
    }
    
    Out << "}" << endl;

    if( ! Out.good() )
        return WRITE_ERROR;

    return autoSave( *pSample );
}


//
//   s a v e
//

tRetCode tCurve::save( ostream& Out, const tHeap<double>& Sample ) const

{
    return save( Out, &Sample );
}


//
//   s a v e
//

tRetCode tCurve::save( ostream& Out, const tCurveInstance& Curve ) const

{
    return save( Out, Curve.sample() );
}


//
//   s a v e
//

tRetCode tCurve::save( ostream& Out ) const

{
    return save( Out, 0 );
}


//
//   s a v e
//

tRetCode tCurve::save( tFileName& Out, const tHeap<double>* pSample ) const

{
    if( ! Out.isValid() )
        return save( pSample );

    ofstream S;
    tRetCode nRet;

    if( ( nRet = Out.openWrite( S, 0 ) ) != OK )
        return nRet;

    return save( S, pSample );
}


//
//   s a v e
//

tRetCode tCurve::save( tFileName& Out, const tHeap<double>& Sample ) const

{
    return save( Out, &Sample );
}


//
//   s a v e
//

tRetCode tCurve::save( tFileName& Out, const tCurveInstance& Curve ) const

{
    return save( Out, Curve.sample() );
}


//
//   s a v e
//

tRetCode tCurve::save( tFileName& Out ) const

{
    return save( Out, 0 );
}


//
//   s a v e
//

tRetCode tCurve::save( const tHeap<double>* pSample ) const

{
    if( ! m_OutFile.isValid() )
        return autoSave( *pSample );

    ofstream S;
    tRetCode nRet;

    if( ( nRet = m_OutFile.openWrite( S, 0 ) ) != OK )
        return nRet;

    return save( S, pSample );
}


//
//   s a v e
//

tRetCode tCurve::save( const tHeap<double>& Sample ) const

{
    return save( &Sample );
}


//
//   s a v e
//

tRetCode tCurve::save( const tCurveInstance& Curve ) const

{
    return save( Curve.sample() );
}


//
//   s a v e
//

tRetCode tCurve::save() const

{
    return save( 0 );
}


//
//   p a r s e
//

tRetCode tCurve::parse( tParser& Parser, tSystem& System, tObject*& pObj )

{
    tRetCode nRet;
    tCurve* pCurve;
    tToken nType;
    tParseInfo Info;

    Info.m_bBase = false;
    Info.m_bScale = false;
    Info.m_bDayCount = false;
    Info.m_bParam1 = false;
    Info.m_bParam2 = false;

    if( ( nRet = parseType( Parser, nType, xTokForward ) ) != OK )
        return nRet;

    switch( nType ) {
        case xTokForward :
            pCurve = new tForwardCurve;
            break;

        case xTokYield :
            pCurve = new tYieldCurve;
            break;

        default :
            return Parser.setError( INVALID_KEYWORD );
    }

    pCurve->setSystem( System );

    if( ( nRet = pCurve->super::parse( Parser, &Info ) ) != OK ) {
        delete pCurve;
        return nRet;
    }

    pObj = pCurve;
    return OK;
}

MTG_END_NAMESPACE
