// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#include "MtgIncl.h"
#include "MtgCurveInstance.h"
#include "MtgCurve.h"
#include "MtgDateBase.h"
#include "MtgEngine.h"

#include <float.h>

MTG_BEGIN_NAMESPACE


//
//   t C u r v e I n s t a n c e
//

tCurveInstance::tCurveInstance( const tCurve& Curve, tEngine& Engine )
    : m_Curve( Curve ), m_Engine( Engine )

{
    m_gAlpha = 1;
    m_gBeta = 0;
}


//
//   ~ t C u r v e I n s t a n c e
//

tCurveInstance::~tCurveInstance()

{
}


//
//   p r e p a r e
//

tRetCode tCurveInstance::prepare()

{
    tRetCode nRet;

    const tDateBase& DB = m_Engine.dateBase();

    //if( DB.base() > m_Curve.base() )
    //    return OUT_OF_RANGE;

    if( ( nRet = m_Curve.getTranslationSI( DB, m_gAlpha, m_gBeta ) ) != OK )
        return nRet;

    int n = m_Curve.numOfSamples();

    m_nFirstSample = 0;
    while( m_nFirstSample < n &&
           ! m_Engine.isCoveredSI( xlat( m_nFirstSample ) ) ) {
        ++m_nFirstSample;
    }

    m_nLastSample = n - 1;
    while( m_nLastSample >= m_nFirstSample &&
           ! m_Engine.isCoveredSI( xlat( m_nLastSample ) ) ) {
        --m_nLastSample;
    }

    m_Sample.numOfElems( n );
    m_Sample.fill( 0 );

    m_Accu.reset();

    return OK;
}


//
//   p o s t P r o c e s s
//

tRetCode tCurveInstance::postProcess( double gWeight )

{
    if( m_nFirstSample <= m_nLastSample ) {
        for( int i = 0; i < m_nFirstSample; ++i )
            m_Sample[i] = m_Sample[m_nFirstSample];
        for( int j = m_nLastSample + 1; j < m_Sample.numOfElems(); ++j )
            m_Sample[j] = m_Sample[m_nLastSample];
    }

    if( gWeight >= 0 ) {
        if( m_Accu.numOfElems() == 0 ) {
            m_Accu = m_Sample;
                for( int i = 0; i < m_Accu.numOfElems(); ++i )
                m_Accu[i] *= gWeight;
        }
        else {
            for( int i = 0; i < m_Accu.numOfElems(); ++i )
                m_Accu[i] += gWeight * m_Sample[i];
        }

            // prepare for next round:
        zero();
    }

    return OK;
}


//
//   z e r o
//

void tCurveInstance::zero()

{
    m_Sample.fill( 0 );
}


//
//   r e w e i g h
//

void tCurveInstance::reweigh( double gBadWeight )

{
    if( gBadWeight > 0 ) {
        gBadWeight = 1 / ( 1 - gBadWeight );
        for( int i = 0; i < m_Accu.numOfElems(); ++i )
            m_Accu[i] *= gBadWeight;
    }
}


//
//   c o n t a i n s E v a l u a t i o n O f
//

bool tCurveInstance::containsEvaluationOf(
    const tCurveInstance& Instance ) const

{
    return false;
}


//
//   s t e a l E v a l u a t i o n F r o m
//

tRetCode tCurveInstance::stealEvaluationFrom( const tCurveInstance& Instance )

{
    MTG_ASSERT( Instance.containsEvaluationOf( *this ) );

    int n = m_Curve.numOfSamples();

    m_gAlpha = Instance.m_gAlpha;
    m_gBeta = Instance.m_gBeta;

    m_nFirstSample = Instance.m_nFirstSample;
    m_nLastSample = Instance.m_nLastSample;

    if( m_nFirstSample > n )
        m_nFirstSample = n;
    if( m_nLastSample > n )
        m_nLastSample = n - 1;

    m_Sample.numOfElems( n );
    for( int i = 0; i < n; ++i )
        m_Sample[i] = Instance.m_Sample[i];

    return OK;
}


//
//   s a m p l e
//

const tHeap<double>& tCurveInstance::sample() const

{
    if( m_Accu.numOfElems() == m_Sample.numOfElems() )
        return m_Accu;
    return m_Sample;
}

MTG_END_NAMESPACE
