// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#if ! defined(_MTG_DATA_SCANNER_)
#define _MTG_DATA_SCANNER_

#include "MtgFileName.h"
#include "MtgHeap.h"
#include "MtgScanner.h"
#include "MtgTclKernel.h"

#include <float.h>

MTG_BEGIN_NAMESPACE


//
//   t D a t a S c a n n e r
//

class tDataScanner : protected tScanner {

        // Note that we declare tScanner protected, not public.
        // A data scanner has a very rigid interface.

    typedef tScanner super;

        // It's possible to initialize the scanner with a
        // source that it owns, by passing a file name.
        // This source will be deleted at destruction.

    tFileSource* m_pAutoSrc;

        // Scanners must not be copied, as access to the source
        // is not synchronized. They are throw-away objects.

    tDataScanner( const tDataScanner& ) {
        throw tException( INTERNAL_ERROR );
    }

public:

    tDataScanner();

    tDataScanner( tSource& Src );
    tDataScanner( tSource* pSrc );
    tDataScanner( const tFileName& FName );

    ~tDataScanner();

        // The source is not owned; simply a reference
        // is recorded. Override protection.

    tRetCode setSource( tSource& Src ) {
        return super::setSource( Src );
    }

    tRetCode setSource( tSource* pSrc = 0 ) {
        return super::setSource( pSrc );
    }

    tRetCode setSource( const tFileName& FName ) {
        return setSource( &FName );
    }

    tRetCode setSource( const tFileName* pFName );

    const char* getErrorMsg() const {
        return super::getErrorMsg();
    }

        // Some functions of super also offered by the data scanner.

    tRetCode scanInteger( int& nValue, int nLowerBound = 1 ) {
        return super::scanInteger( nValue, nLowerBound );
    }

    tRetCode scanInteger( long& nValue, long nLowerBound = 1 ) {
        return super::scanInteger( nValue, nLowerBound );
    }

    tRetCode scanDouble( double& gValue ) {
        return super::scanDouble( gValue );
    }

    tRetCode scanDate( tDate& Date ) {
        return super::scanDate( Date );
    }

        // scanString() scans for a string OR for an id which
        // is interpreted as a string:

    tRetCode scanString( char*& sString );

#if defined(_MTG_WITH_TCL)

        // scanObject() takes anything and makes a new Tcl
        // object out of it:

    tRetCode scanObject( Tcl_Obj*& pObj );

#endif

        // Append to Data while numbers in input stream.

    tRetCode append( tHeap<double>& Data );
    tRetCode append( tHeap<double>& Data, double gLoBound,
        double gHiBound = DBL_MAX );

#if defined(_MTG_WITH_TCL)
    tRetCode append( tHeap<Tcl_Obj*>& Data );
#endif

        // Fill Data; size of Data remains fixed.

    tRetCode fill( tHeap<double>& Data );
    tRetCode fill( tHeap<double>& Data, double gLoBound,
        double gHiBound = DBL_MAX );

    bool atEOF() const;
};

MTG_END_NAMESPACE


#endif
