// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#include "MtgIncl.h"
#include "MtgDate.h"

MTG_BEGIN_NAMESPACE


//
//   v a r i a b l e s
//

const int tDate::m_nBaseYear = 1600;  // Divisible by 400!
const int tDate::m_nBaseWeekday = 6;  // Weekday of 1/1/m_nBaseYear.

const int tDate::m_DayCount[12] = {
    31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31
};                  

const int tDate::m_AccDayCount[12] = {
    0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334
};


//
//   d a y I n Y e a r
//

int tDate::dayInYear() const

{
        // January first is day zero:
    int n = m_AccDayCount[m_nMonth - 1] + m_nDay - 1;
    return ( isLeapYear() && m_nMonth >= 3 ) ? n + 1 : n;
}


//
//   d a y s S i n c e B a s e
//

long tDate::daysSinceBase() const

{
    MTG_ASSERT( m_nBaseYear % 400 == 0 );

    int d = m_nYear - m_nBaseYear;
    long n = d * 365;

        // Now take care of leap years:

    n += ( d + 3 ) / 4;
    n -= ( d + 99 ) / 100;
    n += ( d + 399 ) / 400;

    return n + dayInYear();
}


//
//   s e t I n d e x
//

void tDate::setIndex()

{
    m_nIndex = daysSinceBase();
}


//
//   f r o m I n d e x
//

void tDate::fromIndex()

{
        // There are 97 leap years in every 400 year span starting
        // in a year which is divisible by 400; 25 leap years in the
        // first every 100 day period starting in a year which is divisible
        // by 400, 24 leap years in every subsequent 100 year period,
        // and so on...

    static const long nDaysPer400 = 400 * 365 + 97;
    static const long nDaysPer100 = 100 * 365 + 25;
    static const long nDaysPer4 = 4 * 365 + 1;

        // Find the year:

    m_nYear = m_nBaseYear + (int) ( m_nIndex / nDaysPer400 ) * 400;
    long n = m_nIndex % nDaysPer400;

    if( n >= nDaysPer100 ) {
        m_nYear += 100;
        n -= nDaysPer100;        
        m_nYear += 100 * (int) ( n / ( nDaysPer100 - 1 ) );
        n %= nDaysPer100 - 1;
    }

    long k = isLeapYear() ? nDaysPer4 : nDaysPer4 - 1;

    if( n >= k ) {
        m_nYear += 4;
        n -= k;
        m_nYear += 4 * (int) ( n / nDaysPer4 );
        n %= nDaysPer4;
    }

    k = isLeapYear() ? 366 : 365;

    if( n >= k ) {
        ++m_nYear;
        n -= k;
        m_nYear += (int) ( n / 365 );
        n %= 365;
    }

        // Now find month and day:

    int i = 1;
    while( i < 12 && m_AccDayCount[i] <= n )
        ++i;

    m_nMonth = i;
    m_nDay = (int) n - m_AccDayCount[i - 1] + 1;

        // Handle leap years:

    if( m_nMonth >= 3 && isLeapYear() ) {
        if( m_nDay == 1 ) {
            if( m_nMonth == 3 ) {
                m_nMonth = 2;
                m_nDay = 29;
            }
            else {
                --m_nMonth;
                m_nDay = m_DayCount[m_nMonth - 1];
            }
        }
        else {
            --m_nDay;
        }
    }
}


//
//   n u m O f L e a p Y e a r s
//

int tDate::numOfLeapYears( int nFromYear, int nToYear )

{
    if( nFromYear > nToYear )
        return 0;

    int n = 0;

        // nFromYear is included, so is nToYear

    n += nToYear / 4 - nFromYear / 4;
    if( nFromYear % 4 == 0 )
        ++n;

    n -= nToYear / 100 - nFromYear / 100;
    if( nFromYear % 100 == 0 )
        --n;

    n += nToYear / 400 - nFromYear / 400;
    if( nFromYear % 400 == 0 )
        ++n;

    return n;
}


//
//   s k i p S p a c e
//

void tDate::skipSpace( const char*& sText )

{
    while( isspace( *sText ) )
        ++sText;
}


//
//   s c a n N u m b e r
//

bool tDate::scanNumber( const char*& sText, int& nNumber )

{
    MTG_ASSERT( isdigit( *sText ) );

    nNumber = 0;
    while( isdigit( *sText ) ) 
        nNumber = nNumber * 10 + ( *sText++ - '0' );
    return true;
}


//
//   s c a n W o r d
//

bool tDate::scanWord( const char*& sText, int& nWeekday, int& nMonth )

{
    MTG_ASSERT( isalpha( *sText ) );

        // Compare as little as possible (be fault-tolerant).

    int w = -1;
    int m = -1;

    switch( tolower( *sText ) ) {
        case 'w' : w = 3;  break;           // Wednesday
        case 'o' : m = 10; break;           // October
        case 'n' : m = 11; break;           // November
        case 'd' : m = 12; break;           // December

        case 'f' : 
            if( ! *++sText )
                 return false;

            switch( tolower( *sText ) ) {
                case 'r' : w = 5; break;    // Friday
                case 'e' : m = 2; break;    // February
                default  : return false;
            }
            break;

        case 's' :
            if( ! *++sText )
                 return false;

            switch( tolower( *sText ) ) {
                case 'a' : w = 6; break;    // Saturday
                case 'u' : w = 0; break;    // Sunday
                case 'e' : m = 9; break;    // September
                default  : return false;
            }
            break;

        case 't' : 
            if( ! *++sText )
                 return false;

            switch( tolower( *sText ) ) {
                case 'u' : w = 2; break;    // Tuesday
                case 'h' : w = 4; break;    // Thursday
                default  : return false;
            }
            break;

        case 'a' :
            if( ! *++sText )
                 return false;

            switch( tolower( *sText ) ) {
                case 'p' : m = 4; break;    // April
                case 'u' : m = 8; break;    // August
                default  : return false;
            }
            break;

        case 'm' : 
            if( ! *++sText )
                 return false;

            switch( tolower( *sText ) ) {
                case 'o' : 
                    w = 1;                          // Monday
                    break;                          

                case 'a' : 
                    if( ! *++sText )
                        return false;

                    switch( tolower( *sText ) ) {
                        case 'r' : m = 3; break;    // March
                        case 'y' : m = 5; break;    // May
                        default  : return false;
                    }
                    break;

                default :
                    return false;
            }
            break;

        case 'j' :
            if( ! *++sText )
                 return false;

            switch( tolower( *sText ) ) {
                case 'a' : 
                    m = 1;                          // January
                    break;                          

                case 'u' : 
                    if( ! *++sText )
                        return false;

                    switch( tolower( *sText ) ) {
                        case 'n' : m = 6; break;    // June
                        case 'l' : m = 7; break;    // July
                        default  : return false;
                    }
                    break;

                default :
                    return false;
            }
            break;

        default:
            return false;
    }

    while( isalpha( *sText ) )
        ++sText;

    if( w >= 0 ) {
        if( nWeekday >= 0 )
            return false;
        nWeekday = w;
    }

    if( m >= 0 ) {
        if( nMonth >= 0 )
            return false;
        nMonth = m;
    }

    return true;
}


//
//   t D a t e
//

tDate::tDate( long nIndex )

{
    MTG_ASSERT( nIndex >= 0 );

    m_nIndex = nIndex;
    fromIndex();
}


//
//   t D a t e
//

tDate::tDate()

{
    m_nDay = 1;
    m_nMonth = 1;
    m_nYear = m_nBaseYear;
    m_nIndex = 0;
}


//
//   t D a t e
//

tDate::tDate( const tDate& Date )

{
    m_nDay = Date.m_nDay;
    m_nMonth = Date.m_nMonth;
    m_nYear = Date.m_nYear;
    m_nIndex = Date.m_nIndex;
}


//
//   t D a t e
//

tDate::tDate( int nDay, int nMonth, int nYear )

{
    set( nDay, nMonth, nYear );
}


//
//   t D a t e
//

tDate::tDate( const struct tm& Time )

{
    set( Time );
}


//
//   t D a t e
//

tDate::tDate( const char* sText )

{
    operator=( sText );
}


//
//   ~ t D a t e
//

tDate::~tDate()

{
}


//
//   o p e r a t o r =
//

tDate& tDate::operator=( const tDate& Date )

{
    if( &Date != this ) {
        m_nDay = Date.m_nDay;
        m_nMonth = Date.m_nMonth;
        m_nYear = Date.m_nYear;
        m_nIndex = Date.m_nIndex;
    }
    return *this;
}


//
//   t D a t e
//

tDate& tDate::operator=( const char* sText )

{
    const char* s = set( sText );

    if( s != 0 ) {
        while( isspace( *s ) )
            ++s;
    }

    if( s == 0 || *s != 0 )
        throw tException( FORMAT_ERROR );

    return *this;
}


//
//   w e e k d a y
//

int tDate::weekday() const

{
    return ( m_nBaseWeekday + m_nIndex ) % 7;
}


//
//   i s B u s i n e s s d a y
//

bool tDate::isBusinessday() const

{
    int w = weekday();
    return w >= 1 && w <= 5;
}


//
//   i s W e e k e n d
//

bool tDate::isWeekend() const

{
    return ! isBusinessday();
}


//
//   i s L e a p Y e a r
//

bool tDate::isLeapYear() const

{
    return isLeapYear( m_nYear );
}


//
//   i s L e a p Y e a r
//

bool tDate::isLeapYear( int nYear )

{
    return nYear % 4 == 0 && ( nYear % 100 != 0 || nYear % 400 == 0 );
}


//
//   l a s t D a y I n F e b r u a r y
//

int tDate::lastDayInFebruary() const

{
    return lastDayInFebruary( m_nYear );
}


//
//   l a s t D a y I n F e b r u a r y
//

int tDate::lastDayInFebruary( int nYear )

{
    return isLeapYear( nYear ) ? 29 : 28;
}


//
//   i s L a s t D a y I n F e b r u a r y
//

bool tDate::isLastDayInFebruary() const

{
    return m_nMonth == 2 && m_nDay == lastDayInFebruary( m_nYear );
}


//
//   i s L a s t D a y I n M o n t h
//

bool tDate::isLastDayInMonth() const

{
    if( m_nMonth == 2 )
        return isLastDayInFebruary();
    return m_nDay == m_DayCount[m_nMonth - 1];
}


//
//   s e t
//

void tDate::set( int nDay, int nMonth, int nYear )

{
    if( nYear <= 38 )
        nYear += 2000;
    else
        if( nYear < 100 )
            nYear += 1900;

    if( nDay < 1 || nMonth < 1 || nMonth > 12 || nYear < m_nBaseYear )
        throw tException( FORMAT_ERROR );

    if( nMonth == 2 && isLeapYear( nYear ) ) {
        if( nDay > 29 )
            throw tException( FORMAT_ERROR );
    }
    else {
        if( nDay > m_DayCount[nMonth - 1] )
            throw tException( FORMAT_ERROR );
    }

    m_nDay = nDay;
    m_nMonth = nMonth;
    m_nYear = nYear;
    setIndex();
}


//
//   s e t
//

void tDate::set( const struct tm& Time )

{
    set( Time.tm_mday, Time.tm_mon + 1, Time.tm_year + 1900 );
}


//
//   s e t
//

void tDate::set( time_t UTC )

{
    set( *localtime( &UTC ) );
}


//
//   s e t
//

const char* tDate::set( const char* sText )

{
    MTG_ASSERT( sText != 0 );

    int d = -1;
    int m = -1;
    int y = -1;
    int w = -1;

    skipSpace( sText );

    while( isalpha( *sText ) ) {
        int w1 = w;

        if( ! scanWord( sText, w, m ) )
            return 0;

        if( w1 != w ) {
                // It's the weekday, which can come only first.
            if( m >= 0 )
                return 0;
                // Skip comma if present.
            skipSpace( sText );
            if( *sText == ',' ) {
                ++sText;
                skipSpace( sText );
            }
        }
        else {
            skipSpace( sText );
            if( *sText == '/' || *sText == '-' ) {
                ++sText;
                skipSpace( sText );
            }
        }
    }

    if( m >= 0 ) {
            // Nov 6 1994   or   Nov 6, 1994   or   Nov/6/1994

        if( ! isdigit( *sText ) || ! scanNumber( sText, d ) )
            return 0;

        skipSpace( sText );
        if( *sText == '/' || *sText == '-' || *sText == ',' ) {
            ++sText;
            skipSpace( sText );
        }

    }
    else {
            // 6 Nov 1994   or   6-Nov-1994   or
            // 6/Nov/1994   or   11/6/1994

        int a;

        if( ! isdigit( *sText ) || ! scanNumber( sText, a ) )
            return 0;

        skipSpace( sText );
        if( *sText == '/' || *sText == '-' ) {
            ++sText;
            skipSpace( sText );
        }

        if( isalpha( *sText ) ) {
                // 6 Nov 1994   or   6-Nov-1994   or
                // 6/Nov/1994

            if( ! scanWord( sText, w, m ) || m < 0 )
                return 0;
            d = a;
        }
        else {
                // 11/6/1994

            if( ! isdigit( *sText ) || ! scanNumber( sText, d ) )
                return 0;
            m = a;
        }

        skipSpace( sText );
        if( *sText == '/' || *sText == '-' ) {
            ++sText;
            skipSpace( sText );
        }
    }

        // Now scan year.
    if( ! isdigit( *sText ) || ! scanNumber( sText, y ) )
        return 0;

    set( d, m, y );
    return sText;
}


//
//   s e t T o d a y
//

void tDate::setToday()

{
    set( time( NULL ) );
}


//
//   t o d a y
//

tDate tDate::today()

{
    tDate d;
    d.setToday();
    return d;
}


//
//   o p e r a t o r + +
//

tDate& tDate::operator++()

{
    return operator+=( 1 );
}


//
//   o p e r a t o r - -
//

tDate& tDate::operator--()

{
    return operator+=( -1 );
}


//
//   o p e r a t o r + =
//

tDate& tDate::operator+=( long nDays )

{
    if( m_nIndex + nDays < 0 )
        throw tException( FORMAT_ERROR );

    m_nIndex += nDays;
    fromIndex();

    return *this;
}


//
//   o p e r a t o r - =
//

tDate& tDate::operator-=( long nDays )

{
    return operator+=( -nDays );
}


//
//   o p e r a t o r +
//

tDate tDate::operator+( long nDays ) const

{
    if( m_nIndex + nDays < 0 )
        throw tException( FORMAT_ERROR );
    return tDate( m_nIndex + nDays );
}


//
//   o p e r a t o r -
//

tDate tDate::operator-( long nDays ) const

{
    if( m_nIndex - nDays < 0 )
        throw tException( FORMAT_ERROR );
    return tDate( m_nIndex - nDays );
}


//
//   o p e r a t o r -
//

long tDate::operator-( const tDate& Date ) const

{
    return m_nIndex - Date.m_nIndex;
}


//
//   n u m O f L e a p D a y s T o 
//

int tDate::numOfLeapDaysTo( const tDate& Date ) const

{
    if( Date < *this )
        return 0;

    int n = numOfLeapYears( m_nYear, Date.m_nYear );

    if( isLeapYear() && tDate( 29, 2, m_nYear ) < *this )
        --n;
    if( Date.isLeapYear() && tDate( 29, 2, Date.m_nYear ) > Date )
        --n;

    return n;
}


//
//   n u m O f D a y s I n L e a p Y e a r s T o
//

long tDate::numOfDaysInLeapYearsTo( const tDate& Date ) const

{
    if( Date < *this )
        return 0;

    long n = numOfLeapYears( m_nYear, Date.m_nYear ) * (long) 366;

    if( isLeapYear() )
        n -= dayInYear();
    if( Date.isLeapYear() )
        n -= 365 - Date.dayInYear();    // 365 is right!

    return n;
}


//
//   n u m O f D a y s I n N o n L e a p Y e a r s T o
//

long tDate::numOfDaysInNonLeapYearsTo( const tDate& Date ) const

{
    if( Date < *this )
        return 0;

    int y = Date.m_nYear - m_nYear + 1;
    long n = ( y - numOfLeapYears( m_nYear, Date.m_nYear ) ) * (long) 365;

    if( ! isLeapYear() )
        n -= dayInYear();
    if( ! Date.isLeapYear() )
        n -= 364 - Date.dayInYear();    // 364 is right!

    return n;
}


//
//   o p e r a t o r = =
//

bool tDate::operator==( const tDate& Date ) const

{
    return m_nIndex == Date.m_nIndex;
}


//
//   o p e r a t o r ! =
//

bool tDate::operator!=( const tDate& Date ) const

{
    return m_nIndex != Date.m_nIndex;
}


//
//   o p e r a t o r > =
//

bool tDate::operator>=( const tDate& Date ) const

{
    return m_nIndex >= Date.m_nIndex;
}


//
//   o p e r a t o r < =
//

bool tDate::operator<=( const tDate& Date ) const

{
    return m_nIndex <= Date.m_nIndex;
}


//
//   o p e r a t o r >
//

bool tDate::operator>( const tDate& Date ) const

{
    return m_nIndex > Date.m_nIndex;
}


//
//   o p e r a t o r <
//

bool tDate::operator<( const tDate& Date ) const

{
    return m_nIndex < Date.m_nIndex;
}


//
//   n e x t B u s i n e s s d a y
//

tDate tDate::nextBusinessday( int nAdvance ) const

{
    MTG_ASSERT( nAdvance >= 0 );

    int n, w;

        // If it's the weekend, we advance no matter what.

    switch( ( w = weekday() ) ) {
        case 6 : n = 2; w = 1; break;   // Saturday
        case 0 : n = 1; w = 1; break;   // Sunday
        default: n = 0;                 // Any weekday
    }

    if( n > 0 && nAdvance > 0 )
        --nAdvance;

    if( nAdvance > 0 ) {
            // Advance to following Monday, if necessary.
        if( w != 1 ) {
            if( w + nAdvance > 5 ) {
                n += 8 - w;           // Skip the weekend.
                nAdvance -= 6 - w;    // Don't count the weekend!
            }
            else {
                n += nAdvance;
                nAdvance = 0;
            }
        }

        if( nAdvance > 0 ) {
            n += 7 * ( nAdvance / 5 );
            n += nAdvance % 5;
        }
    }

    MTG_ASSERT( ( *this + n ).isBusinessday() );
    return *this + n;
}


//
//   p r e v B u s i n e s s d a y
//

tDate tDate::prevBusinessday( int nBacktrack ) const

{
    MTG_ASSERT( nBacktrack >= 0 );

    int n, w;

        // If it's the weekend, we backtrack no matter what
        // (to Friday, that is).

    switch( ( w = weekday() ) ) {
        case 6 : n = 1; w = 5; break;   // Saturday
        case 0 : n = 2; w = 5; break;   // Sunday
        default: n = 0;                 // Any weekday
    }

    if( n > 0 && nBacktrack > 0 )
        --nBacktrack;

    if( nBacktrack > 0 ) {
            // Backtrack to last Friday, if necessary.
        if( w != 5 ) {
            if( w - nBacktrack < 1 ) {
                n += w + 2;         // Skip the weekend.
                nBacktrack -= w;    // Don't count the weekend!
            }
            else {
                n += nBacktrack;
                nBacktrack = 0;
            }
        }

        if( nBacktrack > 0 ) {
            n += 7 * ( nBacktrack / 5 );
            n += nBacktrack % 5;
        }
    }

    MTG_ASSERT( ( *this - n ).isBusinessday() );
    return *this - n;
}


//
//   o p e r a t o r < <
//

ostream& operator<<( ostream& Out, const tDate& Date )

{
    char c = Out.fill( '0' );

    Out << setw( 2 ) << Date.month() << "/"
        << setw( 2 ) << Date.day() << "/"
        << setw( 4 ) << Date.year();

    Out.fill( c );
    return Out;
}

MTG_END_NAMESPACE


//#define _TEST
#if defined(_TEST)

#if defined(_WIN32)
    #include <conio.h>
#else
    #define getche getchar
#endif

MTG_USING_NAMESPACE

//
//   m a i n
//

void main( int argc, char *argv[] )

{
    printf( "\nTest tDate\n\n" );

    int w = 6;
    long n = 0;

    tDate z( 1, 1, 1600 );
    tDate r( 13, 10, 1954 );

    for( int y = 1600; y <= 2500; ++y ) {
        printf( "\r%d..", y );

        int nFeb = 28 + ( tDate::isLeapYear( y ) ? 1 : 0 );

        for( int m = 1; m <= 12; ++m ) {
            for( int d = 1; d <= 31; ++d ) {
                if( d == 31 && 
                    ( m == 4 || m == 6 || m == 9 || m == 11 ) ) {
                    continue;
                }
                if( m == 2 && d > nFeb )
                    continue;

                tDate a( d, m, y );

                if( a.weekday() != w || a.index() != n || a != z )
                    printf( "\rError: %d.%d.%d\n", d, m, y );

                if( z.day() != d || z.month() != m || z.year() != y )
                    printf( "\rError: %d.%d.%d\n", d, m, y );
                
                tDate e = a + 1;

                if( e <= a )
                    printf( "\rError: %d.%d.%d\n", d, m, y );

                e += 2; e -= 3; e = e + 10; e = e - 5; --e; e -= 4;

                if( e != a )
                    printf( "\rError: %d.%d.%d\n", d, m, y );

                e.set( a.day(), a.month(), a.year() );

                if( e < a || e > a )
                    printf( "\rError: %d.%d.%d\n", d, m, y );

                long l = r - e;
                e = r - l;

                if( e != a )
                    printf( "\rError: %d.%d.%d\n", d, m, y );

                ++z;
                ++n;
                if( ++w == 7 )
                    w = 0;
            }
        }
    }

    if( tDate( 12, 7, 1998 ).weekday() != 0 )
        printf( "\rWrong weekday calibration\n" );
    else
        printf( "\r" );

    char sBuf[256];
    const char* s;
    int cCmd;
    tDate d;

    bool bGo = true;

    while( bGo ) { 
        printf( "<S>et E<x>it: " );
        cCmd = getche();
        printf( "\n" );

        switch( cCmd ) {
            case 's' :
            case 'S' :
                printf( "Date: " );
                gets( sBuf );
                if( ( s = d.set( sBuf ) ) == 0 ) {
                    printf( "Format error\n" );
                }
                else {
                    printf( "%d/%d/%d, Rest [%s]\n",
                        d.month(), d.day(), d.year(), s );
                }
                break;

            case 'x' :
            case 'X' :
                bGo = false;
                break;
        }
    }

#if ! defined(_WIN32)
    printf( "\n" );
#endif
}

#endif
