// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#if ! defined(_MTG_DATE_)
#define _MTG_DATE_

MTG_BEGIN_NAMESPACE


//
//   t D a t e
//

class tDate {

    int m_nDay;     // 1-31
    int m_nMonth;   // 1-12
    int m_nYear;    // full year

    long m_nIndex;  // Number of days from base year.

        // Helper variables:

    static const int m_nBaseYear;
    static const int m_nBaseWeekday;
    static const int m_DayCount[12];
    static const int m_AccDayCount[12];

        // Helper functions:

    void init();

    int dayInYear() const;
    long daysSinceBase() const;

    void setIndex();
    void fromIndex();

    static int numOfLeapYears( int nFromYear, int nToYear );

    static void skipSpace( const char*& sText );
    static bool scanNumber( const char*& sText, int& nNumber );
    static bool scanWord( const char*& sText, int& nWeekday, int& nMonth );

    tDate( long nIndex );

public:

    tDate();
    tDate( const tDate& Date );
    tDate( int nDay, int nMonth, int nYear );
    tDate( const struct tm& Time );
    tDate( const char* sText );

    ~tDate();

    tDate& operator=( const tDate& Date );    
    tDate& operator=( const char* sText );    

        // Data retrieval:

    int day() const    { return m_nDay; }
    int month() const  { return m_nMonth; }
    int year() const   { return m_nYear; }

        // For debugging purposes:

    long index() const { return m_nIndex; }

        // Sunday == 0, Monday == 1, etc

    int weekday() const;

    bool isBusinessday() const;
    bool isWeekend() const;

    bool isLeapYear() const;
    static bool isLeapYear( int nYear );

        // For the current or given year, what is the last day
        // in February? 28 or 29?

    int lastDayInFebruary() const;
    static int lastDayInFebruary( int nYear );

        // Is the current day the last day in February? The
        // last day of the month?

    bool isLastDayInFebruary() const;
    bool isLastDayInMonth() const;

        // Modifications. The first version of the set() functions
        // throws a FORMAT_ERROR exception if the arguments do
        // not form a proper date.

    void set( int nDay, int nMonth, int nYear );
    void set( const struct tm& Time );
    void set( time_t UTC );

        // The following function is *very* lenient with
        // date formats; note however, that the European 
        // formats dd.mm.yy are not supported. Supported
        // formats are:
        //     Nov 6 1994
        //     Nov 6, 1994
        //     Nov/6/1994 
        //     6 Nov 1994
        //     6-Nov-1994
        //     6/Nov/1994
        //     11/6/1994
        // and all these with the weekday prepended (i.e.,
        // Sun Nov 6 1994). Also, these formats may be
        // slightly distored (i.e., Nov/6 1994) as long
        // as unambiguity is maintained.

        // On return, 0 means "format error" and a non-zero
        // value points to the first character after the
        // date.

    const char* set( const char* sText );

        // Set the date to the system date, resp. return
        // the system date:

    void setToday();
    static tDate today();

    tDate& operator++();
    tDate& operator--();

    tDate& operator+=( long nDays );
    tDate& operator-=( long nDays );

        // Non-destructive operations:

    tDate operator+( long nDays ) const;
    tDate operator-( long nDays ) const;

        // Difference in days between two dates:

    long operator-( const tDate& Date ) const;

        // More arithmeric:

        // Number of leap days between current date and
        // argument date, including (!) first and last days.

    int numOfLeapDaysTo( const tDate& Date ) const;

        // Number of days in the period belonging to leap
        // resp. non-leap years. First and last day are
        // included!

    long numOfDaysInLeapYearsTo( const tDate& Date ) const;
    long numOfDaysInNonLeapYearsTo( const tDate& Date ) const;

        // Comparison:

    bool operator==( const tDate& Date ) const;
    bool operator!=( const tDate& Date ) const;
    bool operator>=( const tDate& Date ) const;
    bool operator<=( const tDate& Date ) const;
    bool operator>( const tDate& Date ) const;
    bool operator<( const tDate& Date ) const;

        // Weekday functions:

        // If it's the weekend the following function first
        // avances to the next Monday and subtracts one from
        // the argument. Otherwise, the argument remains    
        // untouched. Then, if the argument is still greater
        // then 0, the date is advanced by a corresponding
        // number of business days.

    tDate nextBusinessday( int nAdvance = 0 ) const;

        // This is the equivalent function for moving
        // backwards in time:

    tDate prevBusinessday( int nBacktrack = 0 ) const;
};


//
//   f u n c t i o n s
//

    // Some member functions are also accessible via non-member
    // functions, depending on individual taste:

inline int NumOfLeapDays( tDate A, tDate B ) {
    return A.numOfLeapDaysTo( B );
}

inline long NumOfDaysInLeapYears( tDate A, tDate B ) {
    return A.numOfDaysInLeapYearsTo( B );
}

inline long NumOfDaysInNonLeapYears( tDate A, tDate B ) {
    return A.numOfDaysInNonLeapYearsTo( B );
}

    // Output in the format "mm/dd/yyyy":

ostream& operator<<( ostream& Out, const tDate& Date );

MTG_END_NAMESPACE

#endif
