// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#include "MtgIncl.h"
#include "MtgDateBase.h"
#include "MtgParser.h"

MTG_BEGIN_NAMESPACE


//
//   i n i t
//

void tDateBase::init( tScale nScale, int nNumOfPeriods, 
    const tDayCount& DayCount, tDate Base )

{
    m_nScale = nScale;
    m_nNumOfPeriods = nNumOfPeriods;
    m_DayCount = DayCount;
    m_Base = Base;
    deriveDt();
}


//
//   d e r i v e D t
//

void tDateBase::deriveDt()

{
    m_gDtDay = m_DayCount.fraction(
        tDate( 15, 6, 1998 ), tDate( 16, 6, 1998 ) );

    if( m_nScale == xDay ) {
        m_gDtTU = m_gDtSI = m_gDtDay;
    }
    else {
        m_gDtTU = 1;
        m_gDtSI = 1.0 / m_nNumOfPeriods;
    }
}


//
//   c o p y F r o m
//

void tDateBase::copyFrom( const tDateBase& DB )

{
    if( this == &DB )
        return;

    m_nScale = DB.m_nScale;
    m_nNumOfPeriods = DB.m_nNumOfPeriods;
    m_DayCount = DB.m_DayCount;
    m_Base = DB.m_Base;
    m_gDtTU = DB.m_gDtTU;
    m_gDtSI = DB.m_gDtSI;
    m_gDtDay = DB.m_gDtDay;
}


//
//   p a r s e
//

tRetCode tDateBase::parse( tParser& Parser )

{
    tRetCode nRet;
    tScale nScale;
    int nNumOfPeriods;

    nRet = m_DayCount.parse( Parser );
    if( nRet == OK || nRet != GO ) {
        deriveDt();
        return nRet;
    }

    switch( Parser.curToken() ) {
        case xTokBase :
            if( ( nRet = Parser.readToken() ) != OK ||
                ( nRet = Parser.scanDate( m_Base ) ) != OK ) {
                return nRet;
            }
            break;

        case xTokScale :
        case xTokYear :
        case xTokDay :
            if( Parser.curToken() == xTokScale ) {
                if( ( nRet = Parser.readToken() ) != OK )
                    return nRet;
                if( Parser.curToken() != xTokYear &&
                    Parser.curToken() != xTokDay ) {
                    return Parser.setError( INVALID_KEYWORD );
                }
            }

            nScale = Parser.curToken() == xTokYear ? xYear : xDay;
            if( ( nRet = Parser.readToken() ) != OK )
                return nRet;
            if( nScale == xYear && Parser.beginOfNumber() ) {
                if( ( nRet =
                        Parser.scanInteger( nNumOfPeriods, 1 ) ) != OK ) {
                    return nRet;
                }
            }
            else {
                nNumOfPeriods = 1;
            }
            setScale( nScale, nNumOfPeriods );
            break;

        default :
            return GO;
    }

    return OK;
}


//
//   g e t T U
//

void tDateBase::getTU( tDate Date, long& nUnit ) const

{
    switch( m_nScale ) {
        case xDay :
            nUnit = m_DayCount.numOfDays( m_Base, Date );
            break;

        case xYear :
            throw tException( INTERNAL_ERROR );

        default :
            throw tException( INTERNAL_ERROR );
    }
}


//
//   g e t T U
//

void tDateBase::getTU( tDate Date, int& nUnit ) const

{
    long n;

    getSI( Date, n );
    nUnit = (int) n;
}


//
//   g e t T U
//

void tDateBase::getTU( tDate Date, double& gUnit ) const

{
    switch( m_nScale ) {
        case xDay :
            gUnit = m_DayCount.numOfDays( m_Base, Date );
            break;

        case xYear :
            gUnit = m_DayCount.fraction( m_Base, Date );
            break;

        default :
            throw tException( INTERNAL_ERROR );
    }
}


//
//   g e t S I
//

void tDateBase::getSI( tDate Date, long& nSub ) const

{
    switch( m_nScale ) {
        case xDay :
            nSub = m_DayCount.numOfDays( m_Base, Date );
            break;

        case xYear :
            throw tException( INTERNAL_ERROR );

        default :
            throw tException( INTERNAL_ERROR );
    }
}


//
//   g e t S I
//

void tDateBase::getSI( tDate Date, int& nSub ) const

{
    long n;

    getSI( Date, n );
    nSub = (int) n;
}


//
//   g e t S I
//

void tDateBase::getSI( tDate Date, double& gSub ) const

{
    switch( m_nScale ) {
        case xDay :
            gSub = m_DayCount.numOfDays( m_Base, Date );
            break;

        case xYear :
            gSub = m_DayCount.fraction( m_Base, Date ) * m_nNumOfPeriods;
            break;

        default :
            throw tException( INTERNAL_ERROR );
    }
}


//
//   t D a t e B a s e
//

tDateBase::tDateBase()

{
    init( xDay, 1, DayCountACT_365, tDate::today() );
}


//
//   t D a t e B a s e
//

tDateBase::tDateBase( tScale nScale )

{
    init( nScale, 1, DayCountACT_365, tDate::today() );
}


//
//   t D a t e B a s e
//

tDateBase::tDateBase( tScale nScale, int nNumOfPeriods )

{
    MTG_ASSERT( nScale != xDay || nNumOfPeriods == 1 );
    init( nScale, nNumOfPeriods, DayCountACT_365, tDate::today() );
}


//
//   t D a t e B a s e
//

tDateBase::tDateBase( tScale nScale, const tDayCount& DayCount )

{
    init( nScale, 1, DayCount, tDate::today() );
}


//
//   t D a t e B a s e
//

tDateBase::tDateBase( tScale nScale, int nNumOfPeriods,
    const tDayCount& DayCount )

{
    MTG_ASSERT( nScale != xDay || nNumOfPeriods == 1 );
    init( nScale, nNumOfPeriods, DayCount, tDate::today() );
}


//
//   t D a t e B a s e
//

tDateBase::tDateBase( tScale nScale, const tDayCount& DayCount,
    tDate Base )

{
    init( nScale, 1, DayCount, Base );
}


//
//   t D a t e B a s e
//

tDateBase::tDateBase( tScale nScale, int nNumOfPeriods,
    const tDayCount& DayCount, tDate Base )

{
    MTG_ASSERT( nScale != xDay || nNumOfPeriods == 1 );
    init( nScale, nNumOfPeriods, DayCount, Base );
}


//
//   t D a t e B a s e
//

tDateBase::tDateBase( const tDateBase& DB )

{
    copyFrom( DB );
}


//
//   ~ t D a t e B a s e
//

tDateBase::~tDateBase()

{
}


//
//   o p e r a t o r =
//

tDateBase& tDateBase::operator=( const tDateBase& DB )

{
    if( this != &DB )
        copyFrom( DB );
    return *this;
}


//
//   s e t S c a l e
//

void tDateBase::setScale( tScale nScale )

{
    m_nScale = nScale;
    m_nNumOfPeriods = 1;
    deriveDt();
}


//
//   s e t S c a l e
//

void tDateBase::setScale( tScale nScale, int nNumOfPeriods )

{
    MTG_ASSERT( nScale != xDay || nNumOfPeriods == 1 );

    m_nScale = nScale;
    m_nNumOfPeriods = nNumOfPeriods;
    deriveDt();
}


//
//   s e t D a y C o u n t
//

void tDateBase::setDayCount( const tDayCount& DayCount )

{
    m_DayCount = DayCount;
    deriveDt();
}


//
//   s e t B a s e
//

void tDateBase::setBase( tDate Base )

{
    m_Base = Base;
}


//
//   s a m e D a t e B a s e A s
//

bool tDateBase::sameDateBaseAs( const tDateBase& Base ) const

{
    return m_nScale == Base.m_nScale &&
           m_nNumOfPeriods == Base.m_nNumOfPeriods &&
           m_DayCount == Base.m_DayCount &&
           m_Base == Base.m_Base;
}


//
//   c o m p a t i b l e D a t e B a s e W i t h
//

bool tDateBase::compatibleDateBaseWith( const tDateBase& Base ) const

{
    return m_nScale == Base.m_nScale &&
           m_DayCount == Base.m_DayCount &&
           m_Base == Base.m_Base;
}


//
//   i s B e t w e e n
//

bool tDateBase::isBetween( tDate Date, tDate End ) const

{
    if( Date < base() || Date > End )
        return false;
    return true;
}


//
//   g e t N u m O f S t e p s
//

int tDateBase::getNumOfSteps( tDate Maturity, int nExtraSteps ) const

{
    MTG_ASSERT( Maturity >= m_Base );

    int nNumOfSteps;

    if( m_nScale == xDay ) {
        nNumOfSteps = m_DayCount.numOfDays( m_Base, Maturity );
    }
    else {
        double f = m_nNumOfPeriods *  m_DayCount.fraction( m_Base, Maturity );

        nNumOfSteps = (int) floor( f );
        if( floor( f ) < f )
            ++nNumOfSteps;
    }

    return nNumOfSteps + nExtraSteps;
}


//
//   g e t T r a n s l a t i o n S I
//

tRetCode tDateBase::getTranslationSI( const tDateBase& DB, double& gAlpha,
    double& gBeta ) const

{
    if( ! ( m_DayCount == DB.m_DayCount ) )
        return DATEBASE_MISMATCH;

    if( m_nScale == xDay ) {
        if( DB.m_nScale == xDay )
            gAlpha = 1;
        else
            gAlpha = m_gDtDay * DB.m_nNumOfPeriods;
    }
    else {
        if( DB.m_nScale == xDay )
            gAlpha = 1 / ( DB.m_gDtDay * m_nNumOfPeriods );
        else
            gAlpha = (double) DB.m_nNumOfPeriods / m_nNumOfPeriods;
    }

    if( DB.m_nScale == xDay ) {
        if( m_Base <= DB.m_Base )
            gBeta = -m_DayCount.numOfDays( m_Base, DB.m_Base );
        else
            gBeta = m_DayCount.numOfDays( DB.m_Base, m_Base );
    }
    else {
        if( m_Base <= DB.m_Base )
            gBeta = -m_DayCount.fraction( m_Base, DB.m_Base );
        else
            gBeta = m_DayCount.fraction( DB.m_Base, m_Base );
        gBeta *= DB.m_nNumOfPeriods;
    }

    return OK;
}


//
//   g e t C l o s e s t D a t e S I
//

tDate tDateBase::getClosestDateSI( double gSub ) const

{
    if( gSub == 0 )
        return m_Base;

    if( m_nScale == xDay ) {
        tDate Guess = m_Base + (int) floor( gSub + 0.5 );
        long n = m_DayCount.numOfDays( m_Base, Guess );

        if( n < gSub ) {
            gSub -= 0.5;    // round
            if( n < gSub ) {
                do {
                    ++Guess;
                } while( m_DayCount.numOfDays( m_Base, Guess ) < gSub );
            }
        }
        else
        if( n > gSub ) {
            gSub += 0.5;
            if( n > gSub ) {
                do {
                    --Guess;
                } while( m_DayCount.numOfDays( m_Base, Guess ) > gSub );
            }
        }

        return Guess;
    }

    gSub /= m_nNumOfPeriods;    // make it into a day count fraction

    tDate Guess = m_Base + (int) floor( gSub / m_gDtDay + 0.5 );
    double f = m_DayCount.fraction( m_Base, Guess );

    if( f < gSub ) {
        gSub -= m_gDtDay / 2;      // round
        if( f < gSub ) {
            do {
                ++Guess;
            } while( m_DayCount.fraction( m_Base, Guess ) < gSub );
        }
    }
    else
    if( f > gSub ) {
        gSub += m_gDtDay / 2;      // round
        if( f > gSub ) {
            do {
                --Guess;
            } while( m_DayCount.fraction( m_Base, Guess ) > gSub );
        }
    }

    return Guess;
}


//
//   g e t F r a c t i o n S I
//

double tDateBase::getFractionSI( double gSub ) const

{
    return getFractionSI( gSub, m_DayCount );
}


//
//   g e t F r a c t i o n S I
//

double tDateBase::getFractionSI( double gSub,
    const tDayCount& DayCount ) const

{
    if( m_nScale == xYear )
        return gSub / m_nNumOfPeriods;

    if( DayCount == m_DayCount )
        return gSub * m_gDtDay;

    tDate D = getClosestDateSI( gSub );
    return DayCount.fraction( m_Base, D );
}


//
//   o p e r a t o r < <
//

ostream& operator<<( ostream& Out, const tDateBase& DB )

{
    Out << "base \"" << DB.base() << "\", ";

    if( DB.scale() == xYear )
        Out << "scale year " << DB.numOfPeriods() << ", ";
    else
        Out << "scale day, ";

    Out << DB.dayCount();

    return Out;
}

MTG_END_NAMESPACE