// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#if ! defined(_MTG_DAY_COUNT_)
#define _MTG_DAY_COUNT_

#include "MtgDate.h"

MTG_BEGIN_NAMESPACE

class tParser;


//
//    o b j e c t s
//

extern class tDayCount DayCountACT_ACT;
extern class tDayCount DayCountACT_365;
extern class tDayCount DayCountACT_365_25;
extern class tDayCount DayCountACT_365_NL;
extern class tDayCount DayCountACT_365_ISDA;
extern class tDayCount DayCountACT_360;
extern class tDayCount DayCount30_360_ISDA;
extern class tDayCount DayCount30_360_PSA;
extern class tDayCount DayCount30_360_SIA;
extern class tDayCount DayCount30E_360;
        

//
//   t D a y C o u n t
//

class tDayCount {

public:

    enum tPolicy {
        xACT_ACT,
        xACT_365,       // Default
        xACT_365_25,
        xACT_365_NL,
        xACT_365_ISDA,
        xACT_360,
        x30_360_ISDA,
        x30_360_PSA,
        x30_360_SIA,
        x30E_360
    };

private:

    tPolicy m_nPolicy;

    long numOfDaysACT_ACT( tDate Settlement, tDate Maturity,
        bool bCouponOnLastDayOfFeb ) const;

    long numOfDaysACT_365( tDate Settlement, tDate Maturity,
        bool bCouponOnLastDayOfFeb ) const;

    long numOfDaysACT_365_25( tDate Settlement, tDate Maturity,
        bool bCouponOnLastDayOfFeb ) const;

    long numOfDaysACT_365_NL( tDate Settlement, tDate Maturity,
        bool bCouponOnLastDayOfFeb ) const;

    long numOfDaysACT_365_ISDA( tDate Settlement, tDate Maturity,
        bool bCouponOnLastDayOfFeb ) const;

    long numOfDaysACT_360( tDate Settlement, tDate Maturity,
        bool bCouponOnLastDayOfFeb ) const;

    long numOfDays30_360_ISDA( tDate Settlement, tDate Maturity,
        bool bCouponOnLastDayOfFeb ) const;

    long numOfDays30_360_PSA( tDate Settlement, tDate Maturity,
        bool bCouponOnLastDayOfFeb ) const;

    long numOfDays30_360_SIA( tDate Settlement, tDate Maturity,
        bool bCouponOnLastDayOfFeb ) const;

    long numOfDays30E_360( tDate Settlement, tDate Maturity,
        bool bCouponOnLastDayOfFeb ) const;

    double fractionACT_ACT( tDate Settlement, tDate Maturity,
        tDate PeriodStart, tDate PeriodEnd,
        bool bCouponOnLastDayOfFeb ) const;

    double fractionACT_365( tDate Settlement, tDate Maturity,
        bool bCouponOnLastDayOfFeb ) const;

    double fractionACT_365_25( tDate Settlement, tDate Maturity,
        bool bCouponOnLastDayOfFeb ) const;

    double fractionACT_365_NL( tDate Settlement, tDate Maturity,
        bool bCouponOnLastDayOfFeb ) const;

    double fractionACT_365_ISDA( tDate Settlement, tDate Maturity,
        bool bCouponOnLastDayOfFeb ) const;

    double fractionACT_360( tDate Settlement, tDate Maturity,
        bool bCouponOnLastDayOfFeb ) const;

    double fraction30_360_ISDA( tDate Settlement, tDate Maturity,
        bool bCouponOnLastDayOfFeb ) const;

    double fraction30_360_PSA( tDate Settlement, tDate Maturity,
        bool bCouponOnLastDayOfFeb ) const;

    double fraction30_360_SIA( tDate Settlement, tDate Maturity,
        bool bCouponOnLastDayOfFeb ) const;

    double fraction30E_360( tDate Settlement, tDate Maturity,
        bool bCouponOnLastDayOfFeb ) const;

public:

    tDayCount();
    tDayCount( tPolicy nPolicy );
    tDayCount( const tDayCount& Count );

    tDayCount& operator=( tDayCount Count );
    tDayCount& operator=( tPolicy nPolicy );

        // The crucial function follows. Start and Value denote
        // the time span during which interest is accrued.
        // PeriodStart and PeriodEnd denote the compounding period
        // and must include Start and Value.
        // bCouponOnLastDayOfFeb is only needed for 30/360 SIA and
        // indicates whether the underlying bond "pays a coupon on
        // the last day of February."

        // Note that PeriodStart and PeriodEnd are ignored by all
        // day count conventions but ACT/ACT, but are included
        // for generality nevertheless.

    double fraction( tDate Start, tDate End,
        tDate PeriodStart, tDate PeriodEnd,
        bool bCouponOnLastDayOfFeb = false ) const;

        // Since for all policies but ACT/ACT the compounding period
        // can be omitted, the following function offers the same
        // service without PeriodStart and PeriodEnd. For ACT/ACT,
        // ACT/ACT, the year starting at Start and ending one year
        // later is supplied, so even for that convention a 
        // consistent result is returned.

    double fraction( tDate Start, tDate End,
        bool bCouponOnLastDayOfFeb = false ) const;

        // If only the numerator of the day count fraction is needed
        // (i.e., the number of days between Start and End according
        // to the day count convention), then the following function
        // can be used:

    long numOfDays( tDate Start, tDate End,
        bool bCouponOnLastDayOfFeb = false ) const;

    bool operator==( tDayCount Count ) const;  
    bool operator==( tPolicy nPolicy ) const; 

        // A tDayCount object can be parsed from a text source.
        // The following function does this (it returns OK if
        // a day count has been parsed, and GO it not at all, or
        // error if partially):

    tRetCode parse( tParser& Parser );

        // Generate a listing:
        
    friend ostream& operator<<( ostream& Out, const tDayCount& DayCount );
};


//
//   f u n c t i o n s
//

ostream& operator<<( ostream& Out, const tDayCount& DayCount );

MTG_END_NAMESPACE

#endif
