// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#if ! defined(_MTG_DAY_SHIFT_)
#define _MTG_DAY_SHIFT_

#include "MtgDate.h"

MTG_BEGIN_NAMESPACE

class tParser;


//
//    o b j e c t s
//


//
//   t D a y S h i f t
//

class tDayShift {

        // tDayShift represents a policy to shift from a
        // given date to some other date. It is used to 
        // find the settlement date from the trade date,
        // or the payment date for ocupons, for instance.

protected:

    void copyFrom( const tDayShift& Shift );

public:

    tDayShift();
    tDayShift( const tDayShift& Shift );

    virtual ~tDayShift();

    virtual tDayShift* clone() const = 0;

        // This is the service performed by the tDayShift
        // object:

    virtual tDate shift( tDate Date ) const = 0;

        // A tDayShift object can be parsed from a text source.
        // The following function does this:

    static tRetCode parse( tParser& Parser, tDayShift*& pShift );
};


//
//   t I d e n t i t y D a y S h i f t
//

class tIdentityDayShift : public tDayShift {

    typedef tDayShift super;

        // tIdentityDayShift leaves the date unchanged.

protected:

    void copyFrom( const tIdentityDayShift& Shift );

public:

    tIdentityDayShift();
    tIdentityDayShift( const tIdentityDayShift& Shift );

    ~tIdentityDayShift();

    tDayShift* clone() const;

    tDate shift( tDate Date ) const;
};


//
//   t F i x e d D a y S h i f t
//

class tFixedDayShift : public tDayShift {

    typedef tDayShift super;

        // tFixedDayShift always sets the date fixed in
        // the object, regardless of trade date.

    tDate m_FixedDate;

protected:

    void copyFrom( const tFixedDayShift& Shift );

public:

    tFixedDayShift();
    tFixedDayShift( tDate FixedDate );
    tFixedDayShift( const tFixedDayShift& Shift );

    ~tFixedDayShift();

    tDayShift* clone() const;

    tDate shift( tDate Date ) const;
};


//
//   t F o l l o w i n g D a y S h i f t
//

class tFollowingDayShift : public tDayShift {

    typedef tDayShift super;

        // tFollowingDayShift implements the FOLLOWING policy:
        // The m_nAdvance-th next businessday is picked.
        // If m_nAdvance < -1 then the policy turns into
        // IDENTITY. If m_nAdvance == 0 the policy is to 
        // pick the next businessday from a non-businessday,
        // but leave the date unaltered if it is already a
        // businessday.
        // The default is m_nAdvance == 0.

    int m_nAdvance;

protected:

    void copyFrom( const tFollowingDayShift& Shift );

public:

    tFollowingDayShift();
    tFollowingDayShift( int nAdvance );
    tFollowingDayShift( const tFollowingDayShift& Shift );

    ~tFollowingDayShift();

    tFollowingDayShift& operator=( const tFollowingDayShift& Shift );

    tDayShift* clone() const;

    void set( int nAdvance );

    tDate shift( tDate Date ) const;
};


//
//   t P r e c e d i n g D a y S h i f t
//

class tPrecedingDayShift : public tDayShift {

    typedef tDayShift super;

        // tPrecedingDayShift implements the PRECEDING policy:
        // The m_nBacktrack-th previous businessday is picked.
        // If m_nBacktrack < -1 then the policy turns into
        // IDENTITY. If m_nBacktrack == 0 the policy is to 
        // pick the previous businessday from a non-businessday,
        // but leave the date unaltered if it is already a
        // businessday.
        // The default is m_nBacktrack == 0.

    int m_nBacktrack;

protected:

    void copyFrom( const tPrecedingDayShift& Shift );

public:

    tPrecedingDayShift();
    tPrecedingDayShift( int nBacktrack );
    tPrecedingDayShift( const tPrecedingDayShift& Shift );

    ~tPrecedingDayShift();

    tPrecedingDayShift& operator=( const tPrecedingDayShift& Shift );

    tDayShift* clone() const;

    void set( int nBacktrack );

    tDate shift( tDate Date ) const;
};


//
//   t M o d i f i e d D a y S h i f t
//

class tModifiedDayShift : public tDayShift {

    typedef tDayShift super;

        // tModifiedDayShift implements the MODIFIED FOLLOWING
        // policy: if the date is a weekday, nothing happens.
        // Otherwise, the next businessday is picked.
        // If this day rolls into the next month, however,
        // it is moved back to the last businessday before
        // the date.

protected:

    void copyFrom( const tModifiedDayShift& Shift );

public:

    tModifiedDayShift();
    tModifiedDayShift( const tModifiedDayShift& Shift );

    ~tModifiedDayShift();

    tModifiedDayShift& operator=( const tModifiedDayShift& Shift );

    tDayShift* clone() const;

    tDate shift( tDate Date ) const;
};

MTG_END_NAMESPACE

#endif
