// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#include "MtgIncl.h"
#include "MtgDrift.h"
#include "MtgParser.h"
#include "MtgStepDrift.h"
#include "MtgUSBondMath.h"

MTG_BEGIN_NAMESPACE


//
//   v a r i a b l e s
//

const int tDrift::m_nMaxIter = 50;


//
//   c o p y F r o m
//

void tDrift::copyFrom( const tDrift& Drift )

{
    if( this == &Drift )
        return;

    super::copyFrom( Drift );
    tDateRange::copyFrom( Drift );
}


//
//   p a r s e P r e f i x
//

tRetCode tDrift::parsePrefix( tParser& Parser, tParseInfoStub& Info )

{
    setBase( system().base() );
    setDayCount( system().dayCount() );

    return super::parsePrefix( Parser, Info );
}


//
//   p a r s e P a r a m
//

tRetCode tDrift::parseParam( tParser& Parser, tParseInfoStub& Info )

{
    tRetCode nRet;
    tDayCount DayCount;

    tParseInfo& I = static_cast<tParseInfo&>( Info );

    nRet = DayCount.parse( Parser );

    if( nRet == OK ) {
        if( I.m_bDayCount ) 
            return Parser.setError( ATTR_REDEFINITION );
        setDayCount( DayCount );
        I.m_bDayCount = true;
        return nRet;
    }

    if( nRet != GO )
        return nRet;

    switch( Parser.curToken() ) {
        case xTokBase :
            if( I.m_bBase )
                return Parser.setError( ATTR_REDEFINITION );
            if( ( nRet = tDateRange::parse( Parser ) ) != OK )
                return nRet;
            I.m_bBase = true;
            break;

        case xTokScale :
        case xTokYear :
        case xTokDay :
            if( I.m_bScale )
                return Parser.setError( ATTR_REDEFINITION );
            if( ( nRet = tDateRange::parse( Parser ) ) != OK )
                return nRet;
            I.m_bScale = true;            
            break;

        default :
            return super::parseParam( Parser, Info );
    }

    return OK;
}


//
//   t D r i f t
//

tDrift::tDrift()

{
}


//
//   t D r i f t
//

tDrift::tDrift( tScale nScale )
    : tDateRange( nScale )

{
}


//
//   t D r i f t
//

tDrift::tDrift( tScale nScale, int nNumOfPeriods )
    : tDateRange( nScale, nNumOfPeriods )

{
}


//
//   t D r i f t
//

tDrift::tDrift( tScale nScale, const tDayCount& DayCount )
    : tDateRange( nScale, DayCount )

{
}


//
//   t D r i f t
//

tDrift::tDrift( tScale nScale, int nNumOfPeriods, const tDayCount& DayCount )
    : tDateRange( nScale, nNumOfPeriods, DayCount )

{
}


//
//   t D r i f t
//

tDrift::tDrift( tScale nScale, const tDayCount& DayCount, tDate Base )
    : tDateRange( nScale, DayCount, Base )

{
}


//
//   t D r i f t
//

tDrift::tDrift( tScale nScale, int nNumOfPeriods,
    const tDayCount& DayCount, tDate Base )
    : tDateRange( nScale, nNumOfPeriods, DayCount, Base )

{
}


//
//   t D r i f t
//

tDrift::tDrift( const tDrift& Drift )

{
    copyFrom( Drift );
}


//
//   ~ t D r i f t
//

tDrift::~tDrift()

{
}


//
//   f o r w a r d T U
//

double tDrift::forwardTU( int nUnit ) const

{
    if( scale() == xDay )
        return forwardSI( nUnit );
    return forwardSI( nUnit * numOfPeriods() );
}


//
//   f o r w a r d T U
//

double tDrift::forwardTU( int nFromUnit, int nToUnit ) const

{
    if( scale() == xDay )
        return forwardSI( nFromUnit, nToUnit );
    return forwardSI( nFromUnit * numOfPeriods(), nToUnit * numOfPeriods() );
}


//
//   f o r w a r d T U
//

double tDrift::forwardTU( double gFromUnit, double gToUnit ) const

{
    if( scale() == xDay )
        return forwardSI( gFromUnit, gToUnit );
    return forwardSI( gFromUnit * numOfPeriods(), gToUnit * numOfPeriods() );
}


//
//   i m p l i e d T U
//

double tDrift::impliedTU( int nMaturity ) const

{
    if( scale() == xDay )
        return impliedSI( nMaturity );
    return impliedSI( nMaturity * numOfPeriods() );
}


//
//   f o r w a r d
//

double tDrift::forward( tDate Date ) const

{
    if( scale() == xDay ) {
        int nSub;

        getSI( Date, nSub );
        return forwardSI( nSub );
    }

    return forwardSI( (int) floor( getSI( Date ) ) );
}


//
//   f o r w a r d
//

double tDrift::forward( tDate Start, tDate End ) const

{
    if( scale() == xDay ) {
        int nFromSub, nToSub;

        getSI( Start, nFromSub );
        getSI( End, nToSub );
        return forwardSI( nFromSub, nToSub );
    }

    return forwardSI( getSI( Start ), getSI( End ) );
}


//
//   i m p l i e d
//

double tDrift::implied( tDate Maturity ) const

{
    return forward( base(), Maturity );
}


//
//   p r e s e n t V a l u e T U
//

double tDrift::presentValueTU( int nFromUnit, int nToUnit ) const

{
    double gFwd = forwardTU( nFromUnit, nToUnit );
    return exp( -gFwd * (double) ( nToUnit - nFromUnit ) * dtTU() );
}


//
//   p r e s e n t V a l u e S I
//

double tDrift::presentValueSI( int nFromSub, int nToSub ) const

{
    double gFwd = forwardSI( nFromSub, nToSub );
    return exp( -gFwd * (double) ( nToSub - nFromSub ) * dtSI() );
}


//
//   f u t u r e V a l u e T U
//

double tDrift::futureValueTU( int nFromUnit, int nToUnit ) const

{
    return 1 / presentValueTU( nFromUnit, nToUnit );
}


//
//   f u t u r e V a l u e S I
//

double tDrift::futureValueSI( int nFromSub, int nToSub ) const

{
    return 1 / presentValueSI( nFromSub, nToSub );
}


//
//   p r e s e n t V a l u e T U
//

double tDrift::presentValueTU( double gFromUnit, double gToUnit ) const

{
    MTG_ASSERT( scale() != xDay );

    double gFwd = forwardTU( gFromUnit, gToUnit );
    return exp( -gFwd * ( gToUnit - gFromUnit ) );
}


//
//   p r e s e n t V a l u e S I
//

double tDrift::presentValueSI( double gFromSub, double gToSub ) const

{
    MTG_ASSERT( scale() != xDay );

    double gFwd = forwardSI( gFromSub, gToSub );
    return exp( -gFwd * ( gToSub - gFromSub ) / numOfPeriods() );
}


//
//   f u t u r e V a l u e T U
//

double tDrift::futureValueTU( double gFromUnit, double gToUnit ) const

{
    return 1 / presentValueTU( gFromUnit, gToUnit );
}


//
//   f u t u r e V a l u e S I
//

double tDrift::futureValueSI( double gFromSub, double gToSub ) const

{
    return 1 / presentValueSI( gFromSub, gToSub );
}


//
//   p r e s e n t V a l u e
//

double tDrift::presentValue( tDate Start, tDate End ) const

{
    double gFwd = forward( Start, End );
    return exp( -gFwd * dayCount().fraction( Start, End ) );
}


//
//   f u t u r e V a l u e
//

double tDrift::futureValue( tDate Start, tDate End ) const

{
    return 1 / presentValue( Start, End );
}


//
//   p r e s e n t V a l u e
//

double tDrift::presentValue( tDate Start,
    const tHeap<tPayment>& Payment ) const

{
    double gPV = 0;

    for( int i = 0; i < Payment.numOfElems(); ++i ) {
        gPV += Payment[i].m_gAmount *
                presentValue( Start, Payment[i].m_Date );
    }
    return gPV;
}


//
//   p r e s e n t V a l u e
//

double tDrift::presentValue( tDate Start, tDate End,
    double gRedemption, double gCoupon,
    const tCompounder& Compounder ) const

{
    tHeap<tPayment> Payment;

    tPayment::create( Start, End, gRedemption, gCoupon, Compounder, Payment );
    return presentValue( Start, Payment );
}


//
//   p r e s e n t V a l u e
//

double tDrift::presentValue( tDate Start, double gRedemption,
    double gCoupon, const tCouponCompounder& Compounder ) const 

{
    return presentValue( Start, Compounder.maturity(), gRedemption,
        gCoupon, Compounder );
}


//
//   s w a p R a t e
//

double tDrift::swapRate( tDate Start, tDate End, tBondMath& Math ) const

{
    if( Start == End )
        return 100 * forward( Start );

    double gRate = 5;
    tIdentityDayShift Id;

    if( Math.bondDirtyPrice( Start, End, 0, *this, Id ) > 100 )
        throw tException( OUT_OF_RANGE );

        // find initial bracket for bisection

    double gLeft = 0;
    double gPV = Math.bondDirtyPrice( Start, End, gRate, *this, Id );

    while( gPV < 100 ) {
        gLeft = gRate;
        gRate *= 2;
        gPV = Math.bondDirtyPrice( Start, End, gRate, *this, Id );
    }

    double gRight = gRate;
    double gLastError = 0;
    int nIter = 0;

    while( true ) {
            // next probe
        double gNewRate = ( gLeft + gRight ) / 2;
        double gNewPV = Math.bondDirtyPrice( Start, End, gNewRate, *this, Id );
        double gError = fabs( gNewPV / 100 - 1 );

        if( gError == 0 ) {
            gRate = gNewRate;
            break;
        }

        ++nIter;

        if( gLastError > 0 &&
            gError >= gLastError &&
            nIter > m_nMaxIter / 2 ) {
                // No improvement; be content with last guess.
                // (This escape only after a considerable number
                // of iterations.)
            break;
        }

        gRate = gNewRate;

        if( gError < 1e-10 ) {
                // Be content with 10 digits accuracy.
            break;
        }

        if( nIter >= m_nMaxIter )
            throw tException( NO_CONVERGENCE );

        gLastError = gError;

        if( gNewPV < 100 )
            gLeft = gNewRate;
        else
            gRight = gNewRate;
    }

    return gRate;
}


//
//   s w a p R a t e
//

double tDrift::swapRate( tDate Start, tDate End, int nPeriods ) const

{
    tUSBondMath Math( nPeriods );
    return swapRate( Start, End, Math );
}


//
//   p a r s e
//

tRetCode tDrift::parse( tParser& Parser, tSystem& System, tObject*& pObj )

{
    tRetCode nRet;
    tDrift* pDrift;
    tParseInfo Info;

    Info.m_bBase = false;
    Info.m_bScale = false;
    Info.m_bDayCount = false;
    Info.m_bParam1 = false;
    Info.m_bParam2 = false;

    if( ( nRet = Parser.scanBeginOfObj() ) != OK )
        return nRet;

    pDrift = new tStepDrift;
    pDrift->setSystem( System );

    if( ( nRet = pDrift->super::parse( Parser, &Info ) ) != OK ) {
        delete pDrift;
        return nRet;
    }

    pObj = pDrift;
    return OK;
}

#if defined(_MTG_WITH_TCL)


//
//   t c l T e s t
//

bool tDrift::tclTest( const tObject* pObj )

{
    return dynamic_cast<const tDrift*>( pObj ) != 0;
}


//
//   t c l C o m m a n d
//

int tDrift::tclCommand( ClientData clientData, Tcl_Interp *pInterp,
    int objc, Tcl_Obj *CONST objv[] )

{
    tDrift* pDrift = static_cast<tDrift*>( clientData );

    if( objc < 4 ) {
        Tcl_WrongNumArgs( pInterp, 1, objv,
            "command [options] startdate enddate" );
        return TCL_ERROR;
    }

    int nLength;
    char* sCmd =
        StrToLower( Tcl_GetStringFromObj( objv[1], &nLength ) );

    int nCmd = -1;

    if( strcmp( sCmd, "presentvalue" ) == 0 )
        nCmd = 0;
    else
    if( strcmp( sCmd, "futurevalue" ) == 0 )
        nCmd = 1;
    else
    if( strcmp( sCmd, "swaprate" ) == 0 )
        nCmd = 2;
    else
    if( strcmp( sCmd, "forward" ) == 0 || strcmp( sCmd, "yield" ) == 0 )
        nCmd = 3;

    delete sCmd;
    if( nCmd < 0 )
        return TCL_ERROR;

    int nNextObj = 2;
    int nPeriods = 2;
    tBondMath* pMath = 0;

    if( nCmd == 2 ) {
        if( ! getBondMath( pInterp, objc, objv, nNextObj, pMath ) )
            return TCL_ERROR;

        if( pMath == 0 ) {
            char* sString = Tcl_GetStringFromObj( objv[nNextObj], &nLength );

            if( nLength > 0 && *sString == '-' ) { 
                sString = StrToLower( sString );

                if( strcmp( sString, "-periods" ) == 0 ) {
                    delete sString;
                    if( ++nNextObj == objc ||
                        Tcl_GetIntFromObj( pInterp,
                            objv[nNextObj], &nPeriods ) != TCL_OK ) {
                        return TCL_ERROR;
                    }
                    ++nNextObj;
                }
                else {
                    delete sString;
                    return TCL_ERROR;
                }
            }
        }
    }

    if( objc - nNextObj != 2 ) {
        Tcl_WrongNumArgs( pInterp, 1, objv,
            "command [options] startdate enddate" );
        return TCL_ERROR;
    }

    tDate Start, End;

    if( ! getDate( pInterp, objc, objv, nNextObj, Start ) ||
        ! getDate( pInterp, objc, objv, nNextObj, End ) ) {
        return TCL_ERROR;
    }

    if( Start > End ) {
        tDate T = Start;
        Start = End;
        End = T;
    }

    double gValue;

    switch( nCmd ) {
        case 0 :    // presentvalue
            try { 
                gValue = pDrift->presentValue( Start, End );
            }
            catch( tException ) {
                return TCL_ERROR;
            }
            break;

        case 1 :    // futurevalue
            try { 
                gValue = pDrift->futureValue( Start, End );
            }
            catch( tException ) {
                return TCL_ERROR;
            }
            break;

        case 2 :    // swaprate
            try { 
                if( pMath == 0 )
                    gValue = pDrift->swapRate( Start, End, nPeriods );
                else
                    gValue = pDrift->swapRate( Start, End, *pMath );
            }
            catch( tException ) {
                return TCL_ERROR;
            }
            break;

        case 3 :    // forward
            try { 
                gValue = pDrift->forward( Start, End );
            }
            catch( tException ) {
                return TCL_ERROR;
            }
            break;

        default :
            throw tException( INTERNAL_ERROR );
    }

    Tcl_SetObjResult( pInterp, Tcl_NewDoubleObj( gValue ) );
    return TCL_OK;
}


//
//   t c l D e l e t e
//

void tDrift::tclDelete( ClientData clientData )

{
    tDrift* pDrift = static_cast<tDrift*>( clientData );

    if( pDrift != 0 )
        delete pDrift;
}

#endif

MTG_END_NAMESPACE