// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#if ! defined(_MTG_DRIFT_)
#define _MTG_DRIFT_

#include "MtgObject.h"
#include "MtgCompounder.h"
#include "MtgDateRange.h"
#include "MtgPayment.h"

#if defined(_MTG_WITH_TCL)
    #include "MtgTclTools.h"
#endif

MTG_BEGIN_NAMESPACE

class tBondMath;


//
//   t D r i f t
//

#if defined(_MTG_WITH_TCL)
class tDrift : public tObject, public tDateRange, tTclTools {
#else
class tDrift : public tObject, public tDateRange {
#endif

    typedef tObject super;

        // Max number of iterations in bisection method
        // for swap rate calculation:

    static const int m_nMaxIter;

protected:

    struct tParseInfo : super::tParseInfoStub {
        bool m_bBase;
        bool m_bScale;
        bool m_bDayCount;
        bool m_bParam1;
        bool m_bParam2;
    };

    void copyFrom( const tDrift& Drift );

    tRetCode parseParam( tParser& Parser, tParseInfoStub& Info );
    tRetCode parsePrefix( tParser& Parser, tParseInfoStub& Info );

public:

    tDrift();

        // Passed on to the tDateBase constructor:

    tDrift( tScale nScale );
    tDrift( tScale nScale, int nNumOfPeriods );
    tDrift( tScale nScale, const tDayCount& DayCount );
    tDrift( tScale nScale, int nNumOfPeriods,
        const tDayCount& DayCount );
    tDrift( tScale nScale, const tDayCount& DayCount,
        tDate Base );
    tDrift( tScale nScale, int nNumOfPeriods,
        const tDayCount& DayCount, tDate Base );

    tDrift( const tDrift& Drift );

    ~tDrift();

    virtual tRetCode operator-=( const tDrift& Drift ) = 0;
    virtual tRetCode operator^=( const tDrift& Drift ) = 0;

    virtual void getFwdRange( double& gMin, double& gMax ) const = 0;

        // First, numerical based retrieval functions.

        // Retrieval based on time unit and scaling:

    virtual double forwardTU( int nUnit ) const;
    virtual double forwardTU( int nFromUnit, int nToUnit ) const;
    virtual double forwardTU( double gFromUnit, double gToUnit ) const;

    virtual double impliedTU( int nMaturity ) const;

        // Retrieval independent of time unit and scaling:

    virtual double forwardSI( int nSub ) const = 0;
    virtual double forwardSI( int nFromSub, int nToSub ) const = 0;
    virtual double forwardSI( double gFromSub, double gToSub ) const = 0;

    virtual double impliedSI( int nMaturity ) const = 0;

        // By default, retrieval is based on subintervals:

    virtual double forward( int nSub ) const {
        return forwardSI( nSub );
    }

    virtual double forward( int nFromSub, int nToSub ) const {
        return forwardSI( nFromSub, nToSub );
    }

    virtual double forward( double gFromSub, double gToSub ) const {
        return forwardSI( gFromSub, gToSub );
    }

    virtual double implied( int nMaturity ) const {
        return impliedSI( nMaturity );
    }

    virtual int constantUntil() const = 0;
    virtual int maturity() const = 0;

        // Actual date based retrieval functions. These are
        // defined in terms of the previous functions, with
        // conversion done in tDateBase. They need not be
        // overridden.

        // Rather than in terms of FromDate and ToDate, we
        // speak in terms of Start (date) and End (date), 
        // to be compatible with classes like tBondMath.
        // But it means the same.

    virtual double forward( tDate Date ) const;
    virtual double forward( tDate Start, tDate End ) const;

    virtual double implied( tDate Maturity ) const;

        // The forward rate can be used to compute discount and
        // accrual factors under continuous compounding.

    virtual double presentValueTU( int nFromUnit, int nToUnit ) const;
    virtual double futureValueTU( int nFromUnit, int nToUnit ) const;
    virtual double presentValueSI( int nFromSub, int nToSub ) const;
    virtual double futureValueSI( int nFromSub, int nToSub ) const;

        // The default is to retrieve by subinterval:

    virtual double presentValue( int nFromSub, int nToSub ) const {
        return presentValueSI( nFromSub, nToSub );
    }

    virtual double futureValue( int nFromSub, int nToSub ) const {
        return futureValueSI( nFromSub, nToSub );
    }

    virtual double presentValueTU( double gFromUnit, double gToUnit ) const;
    virtual double futureValueTU( double gFromUnit, double gToUnit ) const;
    virtual double presentValueSI( double gFromSub, double gToSub ) const;
    virtual double futureValueSI( double gFromSub, double gToSub ) const;

        // The default is to retrieve by subinterval:

    virtual double presentValue( double gFromSub, double gToSub ) const {
        return presentValueSI( gFromSub, gToSub );
    }

    virtual double futureValue( double gFromSub, double gToSub ) const {
        return futureValueSI( gFromSub, gToSub );
    }                         

        // The following two functions are consistently based
        // on the day count convention and work under all scales.
        // It is assumed the investment is made at the beginning
        // of date Start, and paid back at the beginning of
        // date End.

    virtual double presentValue( tDate Start, tDate End ) const;
    virtual double futureValue( tDate Start, tDate End ) const;

        // To compute the discounted value of a payment structure
        // is also possible (see tInterestSpline):

    virtual double presentValue( tDate Start,
        const tHeap<tPayment>& Payment ) const;
    virtual double presentValue( tDate Start, tDate End,
        double gRedemption, double gCoupon,
        const tCompounder& Compounder ) const;
    virtual double presentValue( tDate Start, double gRedemption,
        double gCoupon, const tCouponCompounder& Compounder ) const;

        // The swap rate is the coupon value of a hypothetical
        // bond valued at par.

    virtual double swapRate( tDate Start, tDate End, tBondMath& Math ) const;
    virtual double swapRate( tDate Start, tDate End, int nPeriods ) const;

    static tRetCode parse( tParser& Parser, tSystem& System, tObject*& pObj );

#if defined(_MTG_WITH_TCL)

        // Used from tShell to create drift commands:

    static bool tclTest( const tObject* pObj );
    static int tclCommand( ClientData clientData, Tcl_Interp *pInterp,
        int objc, Tcl_Obj *CONST objv[] );
    static void tclDelete( ClientData clientData );

#endif
};

MTG_END_NAMESPACE

#endif
