// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#if ! defined(_MTG_DT_GENERATOR_)
#define _MTG_DT_GENERATOR_

MTG_BEGIN_NAMESPACE

// During the preparation of the probability measure (mtgmeas.h),
// an upper bound for the timestep between slices is computed
// to keep the explicit numerical methods stable. The spacing
// between slices cannot be larger than this maximum timestep.
// The amount of space actually put between slices is determined
// by target timesteps, specified by the user. As long as they
// don't exceed the stable timestep, they provide the reference
// for the spacing between adjacent tree slices. Since it is
// desirable to have small timesteps now and larger timesteps
// in the distant future, the target timesteps may depend
// linearly or exponentially on time. tMtgDtGenerator objects
// provide the interface to compute the desired timestep for a
// particular day.
//
// How is the desired timestep introduced into the
// discretization? At all times, the desired timestep is
// capped by the numerically stable timestep. Furthermore,
// if two events are closer together than required by the
// desired timestep, the so implied smaller value is used
// instead (in other words, all events are matched). And,
// if the distance between two adjacent events is larger
// than the interpolated target timestep, ministeps are
// introduced as uniformly as possible for that particular
// interval.
//
// Note that all timesteps are measured in days (but can
// be less than a day, of course!

    
//
//   t D t G e n e r a t o r
//
    
class tDtGenerator {

protected:

    static void init( double gDt0, double gDtN, int nDayN,
        double& gDtIPolA, double& gDtIPolB );

public:

    virtual tDtGenerator* clone() const = 0;

        // The core of a tMtgDtGenerator object is simply a
        // function that takes a day (counted from today = 0)
        // and returns the desired timestep for a slice that
        // starts on that day. Note that the result is
        // interpreted in days also!

    virtual double nextDt( int nDay ) const = 0;
    virtual double maxDt( int nMaturity ) const;
};


//
//   t L i n D t G e n e r a t o r
//

class tLinDtGenerator : public tDtGenerator {

        // The target timestep at day t is defined
        // as m_gDtIpolA * t + m_gDtIpolB.

    double m_gDtIpolA;
    double m_gDtIpolB;

public:

        // For time 0, gDt0 is the desired timestep, in days.
        // nDayN indicates at which day the timestep should be gDtN.
        // After nDayN, interpolation naturally turns into extrapolation.

    tLinDtGenerator( double gDt0, double gDtN, int nDayN );

    tDtGenerator* clone() const;

    double nextDt( int nDay ) const;
};


//
//   t E x p D t G e n e r a t o r
//

class tExpDtGenerator : public tDtGenerator {

        // tMtgExpDtGenerator works like tMtgLinDtGenerator, but
        // interpolates exponentially to the base m_gBase;

    double m_gBase;

        // The target timestep at day t is defined as
        // m_gBase^(m_gDtIpolA * t + m_gDtIpolB).

    double m_gDtIpolA;
    double m_gDtIpolB;

public:

        // Today, the desired timestep is gDt0. At day nDayN,
        // the desired timestep is gDtN.
        // The first constructor uses e as base.

    tExpDtGenerator( double gDt0, double gDtN, int nDayN );
    tExpDtGenerator( double gDt0, double gDtN, int nDayN, double gBase );

    tDtGenerator* clone() const;

    double nextDt( int nDay ) const;
};

MTG_END_NAMESPACE

#endif
