// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#include "MtgIncl.h"
#include "MtgClaim.h"
#include "MtgDateRange.h"
#include "MtgEngine.h"
#include "MtgOptimizer.h"
#include "MtgPortfolio.h"

MTG_BEGIN_NAMESPACE


//
//   d i r t y 2 C l e a n
//

void tEngine::dirty2Clean( double& gTotal, tHeap<double>& Gradient )

{
    for( int k = 0; k < portfolio().numOfClaims(); ++k ) {
        const tClaim& Claim = portfolio().claim( k );
        double gShift = Claim.unitPriceShift();

        gTotal -= gShift * multiplier( Claim.index() );
        Gradient[Claim.index()] -= gShift;
    }
}


//
//   d i r t y 2 C l e a n
//

void tEngine::dirty2Clean( double& gTotal )

{
    for( int k = 0; k < portfolio().numOfClaims(); ++k ) {
        const tClaim& Claim = portfolio().claim( k );
        gTotal -= Claim.unitPriceShift() * multiplier( Claim.index() );
    }
}


//
//   t E n g i n e
//

tEngine::tEngine()

{
    m_pOptimizer = 0;
    m_pPortfolio = 0;
    m_pModel = 0;
}


//
//   ~ t E n g i n e
//

tEngine::~tEngine()

{
}


//
//   b e f o r e R u n
//

tRetCode tEngine::beforeRun()

{
    MTG_ASSERT( m_pPortfolio != 0 );
    MTG_ASSERT( m_pModel != 0 );

    tRetCode nRet;

    m_Settlement = m_pPortfolio->settlement();

    if( ( nRet = m_pPortfolio->matchFactors( m_pModel,
            m_FactorXlat ) ) != OK ) {
        return nRet;
    }

        // Initialize multiplier index with original multiplier
        // in portfolio.

    m_pPortfolio->getMultipliers( m_Multiplier );

        // If there's an optimizer, we might want to override 
        // some of the multipliers (note that the optimizer 
        // doesn't store the multipliers; we store them):

    if( hasOptimizer() ) {
        m_CurveContainer.merge( optimizer().curveContainer() );
        if( ( nRet = optimizer().loadMultipliers(
                        *m_pPortfolio, m_Multiplier ) ) != OK ) {
            return nRet;
        }
    }

    if( ( nRet = m_ImageContainer.prepare( m_CurveContainer ) ) != OK )
        return nRet;

    if( ( nRet = curveContainer().prepare( *this ) ) != OK )
        return nRet;

    return OK;
}


//
//   a f t e r R u n
//

tRetCode tEngine::afterRun()

{
    tRetCode nRet;

        // If there's an optimizer, we might want to save
        // the new multipliers:

    if( ( nRet = m_CurveContainer.propagate() ) != OK ||
        ( nRet = m_CurveContainer.save() ) != OK ) {
        return nRet;
    }

        // Now do all the images.

    if( ( nRet = m_ImageContainer.save() ) != OK ) 
        return nRet;

    if( hasOptimizer() ) {
        if( ( nRet = optimizer().saveMultipliers(
                        *m_pPortfolio, m_Multiplier ) ) != OK ) {
            return nRet;
        }
    }

    return OK;
}


//
//   s e t O p t i m i z e r
//

void tEngine::setOptimizer( tOptimizer* pOpt )

{
    m_pOptimizer = pOpt;
}


//
//   s e t P o r t f o l i o
//

void tEngine::setPortfolio( tPortfolio* pPf )

{
    m_pPortfolio = pPf;
}


//
//   s e t M o d e l
//

void tEngine::setModel( tModel* pModel )

{
    m_pModel = pModel;
}


//
//   s e t C u r v e C o n t a i n e r
//

void tEngine::setCurveContainer( const tCurveContainer* pContainer )

{
    if( pContainer != 0 )
        m_CurveContainer = *pContainer;
}


//
//   s e t I m a g e C o n t a i n e r
//

void tEngine::setImageContainer( const tImageContainer* pContainer )

{
    if( pContainer != 0 )
        m_ImageContainer = *pContainer;
}


//
//   d a t e B a s e
//

const tDateBase& tEngine::dateBase() const

{
    throw tException( NOT_AVAILABLE );

    static tDateBase DB;
    return DB;
}


//
//   d a t e R a n g e
//

const tDateRange& tEngine::dateRange() const

{
    throw tException( NOT_AVAILABLE );

    static tDateRange Range;
    return Range;
}


//
//   i s C o v e r e d
//

bool tEngine::isCovered( int nDay ) const

{
    const tDateRange& Range = dateRange();

    if( Range.scale() == xDay )
        return Range.isCoveredTU( nDay );
    return Range.isCoveredTU( nDay * Range.dtDay() );
}


//
//   i s C o v e r e d
//

bool tEngine::isCovered( double gDay ) const

{
    const tDateRange& Range = dateRange();

    if( Range.scale() == xDay )
        return Range.isCoveredTU( gDay );
    return Range.isCoveredTU( gDay * Range.dtDay() );
}


//
//   i s C o v e r e d
//

bool tEngine::isCovered( tDate Date ) const

{
    return dateRange().isCovered( Date );
}


//
//   f o r w a r d
//

double tEngine::forward( double gFromDay, double gToDay ) const

{
    throw tException( NOT_AVAILABLE );
    return 0;
}


//
//   f o r w a r d
//

double tEngine::forward( tDate Start, tDate End ) const

{
    throw tException( NOT_AVAILABLE );
    return 0;
}


//
//   p r e s e n t V a l u e
//

double tEngine::presentValue( double gFromDay, double gToDay ) const

{
    throw tException( NOT_AVAILABLE );
    return 0;
}


//
//   f u t u r e V a l u e
//

double tEngine::futureValue( double gFromDay, double gToDay ) const

{
    throw tException( NOT_AVAILABLE );
    return 0;
}


//
//   p r e s e n t V a l u e
//

double tEngine::presentValue( tDate Start, tDate End ) const

{
    throw tException( NOT_AVAILABLE );
    return 0;
}


//
//   f u t u r e V a l u e
//

double tEngine::futureValue( tDate Start, tDate End ) const

{
    throw tException( NOT_AVAILABLE );
    return 0;
}


//
//   f o r w a r d F o r w a r d
//

double tEngine::forwardForward( double gRefDay,
    double gFromDay, double gToDay ) const

{
    throw tException( NOT_AVAILABLE );
    return 0;
}


//
//   f o r w a r d F o r w a r d
//

double tEngine::forwardForward( tDate Ref,
    tDate Start, tDate End ) const

{
    throw tException( NOT_AVAILABLE );
    return 0;
}


//
//   d i s c o u n t F a c t o r
//

double tEngine::discountFactor( double gRefDay,
    double gFromDay, double gToDay ) const

{
    throw tException( NOT_AVAILABLE );
    return 0;
}


//
//   a c c r u a l F a c t o r
//

double tEngine::accrualFactor( double gRefDay,
    double gFromDay, double gToDay ) const

{
    throw tException( NOT_AVAILABLE );
    return 0;
}


//
//   d i s c o u n t F a c t o r
//

double tEngine::discountFactor( tDate Ref,
    tDate Start, tDate End ) const

{
    throw tException( NOT_AVAILABLE );
    return 0;
}


//
//   a c c r u a l F a c t o r
//

double tEngine::accrualFactor( tDate Ref,
    tDate Start, tDate End ) const

{
    throw tException( NOT_AVAILABLE );
    return 0;
}


//
//   i s C o v e r e d S I
//

bool tEngine::isCoveredSI( int nSub ) const

{
    return dateRange().isCoveredSI( nSub );
}


//
//   i s C o v e r e d S I
//

bool tEngine::isCoveredSI( double gSub ) const

{
    return dateRange().isCoveredSI( gSub );
}


//
//   f o r w a r d S I
//

double tEngine::forwardSI( double gFromSub, double gToSub ) const

{
    throw tException( NOT_AVAILABLE );
    return 0;
}


//
//   p r e s e n t V a l u e S I
//

double tEngine::presentValueSI( double gFromSub, double gToSub ) const

{
    throw tException( NOT_AVAILABLE );
    return 0;
}


//
//   f u t u r e V a l u e S I
//

double tEngine::futureValueSI( double gFromSub, double gToSub ) const

{
    throw tException( NOT_AVAILABLE );
    return 0;
}


//
//   f o r w a r d F o r w a r d S I
//

double tEngine::forwardForwardSI( double gRefSub,
    double gFromSub, double gToSub ) const

{
    throw tException( NOT_AVAILABLE );
    return 0;
}


//
//   d i s c o u n t F a c t o r S I
//

double tEngine::discountFactorSI( double gRefSub,
    double gFromSub, double gToSub ) const

{
    throw tException( NOT_AVAILABLE );
    return 0;
}


//
//   a c c r u a l F a c t o r S I
//

double tEngine::accrualFactorSI( double gRefSub,
    double gFromSub, double gToSub ) const

{
    throw tException( NOT_AVAILABLE );
    return 0;
}

MTG_END_NAMESPACE