// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#if ! defined(_MTG_ENGINE_)
#define _MTG_ENGINE_

#include "MtgCurveContainer.h"
#include "MtgImageContainer.h"

MTG_BEGIN_NAMESPACE

class tDateBase;
class tDateRange;
class tOptimizer;
class tPortfolio;
class tModel;


//
//   t E n g i n e
//
    
class tEngine {

        // The following pointers are merely references;
        // the objects are not owned!

    tOptimizer* m_pOptimizer;
    tPortfolio* m_pPortfolio;
    tModel* m_pModel;

        // The date on which pricing is based. Initialized
        // with the settlement date of the portfolio (path
        // space or lattice can sometimes start earlier).

    tDate m_Settlement;

        // We might want to generate a few curves. The curve
        // container contains references to curves we wish
        // to generate.

    tCurveContainer m_CurveContainer;

        // We also might want to generate some images,
        // from curves.

    tImageContainer m_ImageContainer;

        // Translate the factor id's from the portfolio scope to
        // the model scope (initialized in beforeRun()):

    tHeap<int> m_FactorXlat;

        // Keep the multipliers of claims in a separate array,
        // as they might change during computation:

    tHeap<double> m_Multiplier;

        // Engines cannot be copied; they exist only
        // for a unique evaluation.

    tEngine( const tEngine& Engine ) {
        throw tException( INTERNAL_ERROR );
    }

protected:

    virtual tRetCode beforeRun();
    virtual tRetCode afterRun();

    int portfolio2ModelFactor( int nFactor ) const {
        return m_FactorXlat[nFactor];
    }

        // This array is writeable:
        
    tHeap<double>& multiplier() {
        return m_Multiplier;
    }

        // Pricing normally yields a "dirty" value. Some instruments
        // mau adjust:

    void dirty2Clean( double& gTotal, tHeap<double>& Gradient );
    void dirty2Clean( double& gTotal );

public:

    tEngine();

    virtual ~tEngine();

        // Initialize the engine stub:

    void setOptimizer( tOptimizer& Opt ) {
        setOptimizer( &Opt );
    }

    void setOptimizer( tOptimizer* pOpt );

    void setPortfolio( tPortfolio& Pf ) {
        setPortfolio( &Pf );
    }

    void setPortfolio( tPortfolio* pPf );

    void setModel( tModel& Model ) {
        setModel( &Model );
    }

    void setModel( tModel* pModel );

    void setCurveContainer( const tCurveContainer& Container ) {
        setCurveContainer( &Container );
    }

    void setCurveContainer( const tCurveContainer* pContainer );

    void setImageContainer( const tImageContainer& Container ) {
        setImageContainer( &Container );
    }

    void setImageContainer( const tImageContainer* pContainer );

        // Retrieval:

    bool hasOptimizer() const {
        return m_pOptimizer != 0;
    }

    tOptimizer& optimizer() const {
        MTG_ASSERT( m_pOptimizer != 0 );
        return *m_pOptimizer;
    }

    tPortfolio& portfolio() const {
        MTG_ASSERT( m_pPortfolio != 0 );
        return *m_pPortfolio;
    }

    tModel& model() const {
        MTG_ASSERT( m_pModel != 0 );
        return *m_pModel;
    }

    tCurveContainer& curveContainer() {
        return m_CurveContainer;
    }

    tDate settlement() const {
        return m_Settlement;
    }

        // Now come the functions used by the instruments to
        // compute payoffs, cashflows etc.
        
        // Single claims are accessed with their index in the
        // portfolio array m_aNonAux, plus a tag.

    virtual void getClaim( int nIndex, int nTag, double& gUnitValue ) = 0;

        // The following functions offer interest rate calculations,
        // based on days or dates. THEY NEED NOT BE IMPLEMENTED.
        // If they are not implemented, the NOT_AVAILABLE exception
        // should be thrown.

        // Possible implementations use a tDrift or tHJMTermStruct
        // or tShortRateTermStruct object.

    virtual const tDateBase& dateBase() const;
    virtual const tDateRange& dateRange() const;

    virtual bool isCovered( int nDay ) const;
    virtual bool isCovered( double gDay ) const;
    virtual bool isCovered( tDate Date ) const;

    virtual double forward( double gFromDay, double gToDay ) const;
    virtual double forward( tDate Start, tDate End ) const;

    virtual double presentValue( double gFromDay, double gToDay ) const;        
    virtual double futureValue( double gFromDay, double gToDay ) const;        

    virtual double presentValue( tDate Start, tDate End ) const;
    virtual double futureValue( tDate Start, tDate End ) const;

    virtual double forwardForward( double gRefDay,
        double gFromDay, double gToDay ) const;
    virtual double forwardForward( tDate Ref,
        tDate Start, tDate End ) const;

    virtual double discountFactor( double gRefDay,
        double gFromDay, double gToDay ) const;
    virtual double accrualFactor( double gRefDay,
        double gFromDay, double gToDay ) const;

    virtual double discountFactor( tDate Ref,
        tDate Start, tDate End ) const;
    virtual double accrualFactor( tDate Ref,
        tDate Start, tDate End ) const;

        // The interest rate functions are also available in "SI" form,
        // in which the arguments are interpreted as sub-intervals.
        // (NOTE that also the previous functions without the "SI" prefix
        // are redirected to SI versions in some classes, here their
        // arguments are interpreted as days.) In this case, the relevant
        // sub-interval belongs to the underlying path space or lattice.

    virtual bool isCoveredSI( int nSub ) const;
    virtual bool isCoveredSI( double gSub ) const;

    virtual double forwardSI( double gFromSub, double gToSub ) const;

    virtual double presentValueSI( double gFromSub, double gToSub ) const;        
    virtual double futureValueSI( double gFromSub, double gToSub ) const;        

    virtual double forwardForwardSI( double gRefSub,
        double gFromSub, double gToSub ) const;

    virtual double discountFactorSI( double gRefSub,
        double gFromSub, double gToSub ) const;
    virtual double accrualFactorSI( double gRefSub,
        double gFromSub, double gToSub ) const;

        // After the run, the multiplier (lambda) might have changed:

    double multiplier( int nIndex ) const {
        return m_Multiplier[nIndex];
    }
};

MTG_END_NAMESPACE

#endif
