// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#include "MtgIncl.h"
#include "MtgEntropyOpt.h"

MTG_BEGIN_NAMESPACE


//
//   t M C E n t r o p y O p t I n s t a n c e
//

tEntropyOpt::tMCEntropyOptInstance::tMCEntropyOptInstance(
    const tEntropyOpt& Optimizer, tMCEngine& Engine )
    : super( Optimizer, Engine )

{
}


//
//   ~ t M C E n t r o p y O p t I n s t a n c e
//

tEntropyOpt::tMCEntropyOptInstance::~tMCEntropyOptInstance()

{
}


//
//   e v a l
//

void tEntropyOpt::tMCEntropyOptInstance::eval(
    const tHeap<double>& Lambda, double& gTotal, tHeap<double>& Gradient )

{
    const tHeap2<double>& Value = value();

    int n = Value.numOfCols() - 1;  // number of priced instruments
    int m = m_Weight.numOfElems();  // number of paths

        // This is the partition function for minimum relative
        // entropy, with a uniform prior.

        // We have to compute the sum over all exp(sum lambda * value),
        // which can be huge. Therefore, we divide by the maximum
        // exponent p1:
        //
        //   log( exp(p1) + exp(p2) + exp(p3) + exp(p4) )
        // = p1 + log( 1 + exp(p2-p1) + exp(p3-p1) + exp(p4-p1) )
        //
        // where all the exponents are now negative.

    gTotal = 0;

        // First, find the max positive exponents and store it
        // in gTotal.

    for( int k = 0; k < m; ++k ) {
        double gSum = Value[k][n];  // non-priced instruments
                                    // with fixed lambda
        for( int i = 0; i < n; ++i )
            gSum += Lambda[i] * Value[k][i];

        if( k == 0 || gSum > gTotal )
            gTotal = gSum;

        m_Weight[k] = gSum;         // for later use
    }

        // Adjust exponents and do the summation over the
        // exponents, which are all negative now.

    double gRest = 0;

    for( MTG_FOR_INIT( int ) k = 0; k < m; ++k )
        gRest += exp( m_Weight[k] - gTotal );

        // At this point, it is safe to compute gTotal which
        // is now the log of the sum over all exponentials.
        // Thus, exp(gTotal) becomes mu in the theory.

    gTotal += log( gRest );

        // Normalize weights and compute the gradient and entropy.

    Gradient.fill( 0 );

    for( MTG_FOR_INIT( int ) k = 0; k < m; ++k ) {
        m_Weight[k] = exp( m_Weight[k] - gTotal );

        for( int i = 0; i < n; ++i )
            Gradient[i] += m_Weight[k] * Value[k][i];
    }

        // The weights are now probabilities.

    for( int i = 0; i < n; ++i ) {
        Gradient[i] -= m_Price[i];
        gTotal -= Lambda[i] * m_Price[i];
    }

        // gTotal now contains the entropy.

    if( m_Optimizer.isVerbose() ) {
            // Do some reporting:

        double gNorm = 0;

        for( int i = 0; i < n; ++i ) {
            if( fabs( Gradient[i] ) > gNorm )
                gNorm = fabs( Gradient[i] );
        }

        m_Optimizer.verbose()
            << "\r                                                     \r"
            << "... entropy " << gTotal
            << ", max gradient " << gNorm
            << flush;
    }
}


//
//   m i n i m i z e
//

tRetCode tEntropyOpt::tMCEntropyOptInstance::minimize( double& gTotal )

{
    const tEntropyOpt& Opt = static_cast<const tEntropyOpt&>( m_Optimizer );

    if( m_Optimizer.isVerbose() )
        m_Optimizer.verbose() << "Minimizing relative entropy" << endl;

    tMinimizer::minimize( Opt.m_gLoBound, Opt.m_gHiBound,
        m_Lambda, m_Gradient );
    gTotal = result();

    if( m_Optimizer.isVerbose() )
        m_Optimizer.verbose() << "\r   " << endl;

    return OK;
}


//
//   c o p y F r o m
//

void tEntropyOpt::copyFrom( const tEntropyOpt& Opt )

{
    if( &Opt == this )
        return;

    m_gLoBound = Opt.m_gLoBound;
    m_gHiBound = Opt.m_gHiBound;

    super::copyFrom( Opt );
}


//
//   p a r s e P a r a m
//

tRetCode tEntropyOpt::parseParam( tParser& Parser, tParseInfoStub& Info )

{
    tRetCode nRet;

    tParseInfo& I = static_cast<tParseInfo&>( Info );

    switch( Parser.curToken() ) {
        case xTokLow : 
            if( I.m_bParam1 ) {
                return Parser.setError(
                    "lower bound defined more than once" );
            }
            if( ( nRet = Parser.readToken() ) != OK ||
                ( nRet = Parser.scanDouble( m_gLoBound ) ) != OK ) {
                return nRet;
            }
            I.m_bParam1 = true;
            break;

        case xTokHigh :
            if( I.m_bParam2 ) {
                return Parser.setError(
                    "upper bound defined more than once" );
            }
            if( ( nRet = Parser.readToken() ) != OK ||
                ( nRet = Parser.scanDouble( m_gHiBound ) ) != OK ) {
                return nRet;
            }
            I.m_bParam2 = true;
            break;

        default :
            return super::parseParam( Parser, Info );
    }

    return OK;
}


//
//   p a r s e P o s t f i x
//

tRetCode tEntropyOpt::parsePostfix( tParser& Parser, tParseInfoStub& Info )

{
    tParseInfo& I = static_cast<tParseInfo&>( Info );

    if( ! I.m_bParam1 ) {
        if( I.m_bParam2 ) {
            if( m_gHiBound > 0 )
                m_gLoBound = -m_gHiBound;
            else
                m_gLoBound = 2 * m_gHiBound;
        }
        else {
            m_gLoBound = -10;
        }
    }

    if( ! I.m_bParam2 ) {
        if( I.m_bParam1 ) {
            if( m_gLoBound < 0 )
                m_gHiBound = -m_gLoBound;
            else
                m_gHiBound = 2 * m_gLoBound;
        }
        else {
            m_gHiBound = 10;
        }
    }

    if( m_gLoBound >= m_gHiBound )
        return Parser.setError( "inconsistent boundaries" );

    return super::parsePostfix( Parser, Info );
}


//
//   t E n t r o p y O p t
//

tEntropyOpt::tEntropyOpt()

{
}


//
//   t E n t r o p y O p t
//

tEntropyOpt::tEntropyOpt( const tEntropyOpt& Opt )

{
}


//
//   ~ t E n t r o p y O p t
//

tEntropyOpt::~tEntropyOpt()

{
}


//
//   o p e r a t o r =
//

tEntropyOpt& tEntropyOpt::operator=( const tEntropyOpt& Opt )

{
    if( &Opt != this )
        copyFrom( Opt );
    return *this;
}


//
//   c l o n e
//

tObject* tEntropyOpt::clone() const

{
    return new tEntropyOpt( *this );
}


//
//   f i n a l i z e
//

tRetCode tEntropyOpt::finalize()

{
    return super::finalize();
}


//
//   c r e a t e I n s t a n c e
//

tRetCode tEntropyOpt::createInstance( tMCEngine* pEngine,
    tMCOptInstance*& pOpt ) const

{
    pOpt = new tMCEntropyOptInstance( *this, *pEngine );
    return OK;
}

MTG_END_NAMESPACE

