// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#if ! defined(_MTG_EV_GENERATOR_)
#define _MTG_EV_GENERATOR_

#include "MtgHeap.h"

MTG_BEGIN_NAMESPACE

// Event generators are used when the grid structure is calculated,
// since all explicitely known events should be matched by the grid.
// Events include maturity dates of instruments, events triggered
// on certain days, etc - everything that can be precomputed and
// causes some cashflow. Also, one might wish to include certain
// dates for which events are not known with respect to the current
// portfolio, but very likely for future ones. The calibrated tree
// is then matched to these dates, making it more efficient in
// subsequent linear pricing rounds.
//
// The basic interface is defined in tMtgEventGenerator. These
// generators are expected to generated a monotone sequence
// of days, after rewind() is called.

class tPortfolio;    


//
//   t E v G e n e r a t o r
//
    
class tEvGenerator {

public:

    virtual ~tEvGenerator();

        // rewind()/nextEvent() iterate the sequence of events.
        // Day 0 should not be returned as an event.
        // lastEvent() should return the last element of the sequence.

        // A return value of -1 indicates that the sequence has been
        // exhausted. If the sequence is empty, lastEvent() must
        // return 0.

    virtual void rewind() = 0;
    virtual int nextEvent() = 0;
    virtual int lastEvent() = 0;
};


//
//   t S e q E v G e n e r a t o r
//

class tSeqEvGenerator : public tEvGenerator {

        // A tSeqEvGenerator object implements the event generator
        // interface above. It acts as a collector that can be filled
        // with dates (days) of events, sorts them and presents the
        // sorted sequence according to the interface.

    tHeap<int> m_aSeq;

    int m_nNextEvent;

    void init();
    void insert( int nDay );

public:

    tSeqEvGenerator();
    tSeqEvGenerator( const tSeqEvGenerator& EvGen );

    ~tSeqEvGenerator();

    void reset();

    tSeqEvGenerator& operator=( const tSeqEvGenerator& EvGen );

    void copyFrom( const tSeqEvGenerator& EvGen );
 
        // Add one event or a whole bunch of events (don't have to be
        // sorted; multiple dates are filtered out as well).

    void addEvent( int nDay );
    void addEvent( int nNumOfDays, int aDay[] );
    void addEvent( const tHeap<int>& aDay );
    void addEvent( tEvGenerator& EvGen );

        // Merge with the event trace of a portfolio.

    void addEvent( tPortfolio& Pf );

    tSeqEvGenerator& operator+=( int nDay ) {
        addEvent( nDay );
        return *this;
    }

    tSeqEvGenerator& operator+=( const tHeap<int>& aDay ) {
        addEvent( aDay );
        return *this;
    }

    tSeqEvGenerator& operator+=( tEvGenerator& EvGen ) {
        addEvent( EvGen );
        return *this;
    }

    tSeqEvGenerator& operator+=( tPortfolio& Pf ) {
        addEvent( Pf );
        return *this;
    }

    void cap( int nMaxDay );

    void rewind();
    int nextEvent();
    int lastEvent();
};


//
//   t D a i l y E v G e n e r a t o r
//

class tDailyEvGenerator : public tEvGenerator {

        // This event generator supports the special case that
        // each day should be matched by the grid (the classical
        // vop case).

        // Generate events up to and including the date of maturity.

    int m_nMaturity;
    int m_nNextEvent;

public:

    tDailyEvGenerator( int nMaturity );

    void rewind();
    int nextEvent();
    int lastEvent();
};

MTG_END_NAMESPACE

#endif
