// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#include "MtgIncl.h"
#include "MtgFDEngine.h"
#include "MtgClaim.h"
#include "MtgModel.h"
#include "MtgPortfolio.h"
#include "MtgProfile.h"
#include "MtgScenario.h"

MTG_BEGIN_NAMESPACE


//
//   t y p e s
//

struct tFDEngine::tTask {
    tMode m_nMode;              // Basic mode, may be overridden.
    int m_nCurRound;
    int m_nLevel;               // Current level.
    int m_nTag;                 // Tags are used for the scenario.
    int m_nSize;                // Number of instruments in task.
    int m_nDoCnt;               // How often (re)started?
    tTask* m_pNext;
    tTask* m_pPrev;
    tSlotLayer* m_pLayer;

    tMode m_nBetwMode;          // Store for mode and other flags
    bool m_bBetwPreprocess;     // between the two halves of one
    bool m_bBetwPostprocess;    // timestep.

    tSpecific* m_pSpecific;     // Engine-specific information.

    ~tTask() {
        if( m_pLayer != 0 )
            delete m_pLayer;
        if( m_pSpecific != 0 )
            delete m_pSpecific;
    }
};


//
//   i n i t
//

void tFDEngine::init()

{
    m_pRunLattice = 0;
    m_pFirstTask = 0;
    m_pLastTask = 0;
    m_pLattice = 0;
    m_pScenario = 0;
    m_pResultLayer = 0;
    m_pAuxTask = 0;
    m_pCurTask = 0;
    m_nAccuracy = xExact;
}


//
//   c l e a n u p
//

void tFDEngine::cleanup()

{
    m_SigAssoc.reset();
    m_ClaimAssoc.reset();
    m_Level.reset();

    if( m_pRunLattice ) {
        delete m_pRunLattice;
        m_pRunLattice = 0;
    }

    tTask* p = m_pFirstTask;

    m_pFirstTask = 0;
    m_pLastTask = 0;

    while( p != 0 ) {
        tTask* q = p->m_pNext;
        delete p;
        p = q;
    }

    m_UnderControl.reset();
    m_Pos.reset();

    m_pResultLayer = 0;

    m_pAuxTask = 0;
    m_pCurTask = 0;
}


//
//   a p p e n d L e v e l
//

void tFDEngine::appendLevel()

{
    MTG_ASSERT( m_nCurLevel >= m_Level.numOfElems() - 2 );

        // The new level is created above the current level.

    if( m_nCurLevel == m_Level.numOfElems() - 1 ) {
        ++m_Level;

        tLevel& V = m_Level.last();

        V.m_pFirstTask = 0;
        V.m_pLastTask = 0;
        V.m_pCurTask = 0;
        V.m_nCurRound = 0;
        V.m_nToRound = m_nCurRound + 1;

        if( m_nCurLevel >= 0 ) {
            MTG_TRACE( "Engine: need to redo %d round%s\n",
                V.m_nToRound, V.m_nToRound == 1 ? "" : "s" );
        }
    }
    else {
        MTG_ASSERT( m_Level.last().m_nToRound == m_nCurRound + 1 );
    }
}


//
//   d e c r e a s e L e v e l
//

void tFDEngine::decreaseLevel()

{
    MTG_ASSERT( m_nCurLevel == m_Level.numOfElems() - 1 );

    if( m_nCurLevel > 0 ) {
            // The 0 level list is never deleted.
        if( m_Level.last().m_pFirstTask != 0 )
            moveTask( m_Level.last().m_pFirstTask, m_nCurLevel - 1 );
    }
    else {
        m_pFirstTask = m_Level[0].m_pFirstTask;
        m_pLastTask = m_Level[0].m_pLastTask;
    }

    m_Level.numOfElems( m_nCurLevel );
}


//
//   m o v e T a s k
//

void tFDEngine::moveTask( tTask* pTask, int nLevel )

{
    tLevel& L1 = m_Level[pTask->m_nLevel];
    tLevel& L2 = m_Level[nLevel];

    if( nLevel > pTask->m_nLevel ) {
            // Remove task and re-append at new level.

        MTG_ASSERT( nLevel > m_nCurLevel );

        if( pTask->m_pNext == 0 )
            L1.m_pLastTask = pTask->m_pPrev;
        else
            pTask->m_pNext->m_pPrev = pTask->m_pPrev;
        if( pTask->m_pPrev == 0 )
            L1.m_pFirstTask = pTask->m_pNext;
        else
            pTask->m_pPrev->m_pNext = pTask->m_pNext;

        pTask->m_pNext = 0;
        pTask->m_pPrev = 0;
        pTask->m_nLevel = nLevel;

        ++pTask->m_nDoCnt;
        pTask->m_nCurRound = 0;     // Need to recompute everything.
        pTask->m_pLayer->prepareRollback();

        MTG_ASSERT( L2.m_pLastTask == 0 ||
                    L2.m_pLastTask->m_nCurRound == 0 );
    }
    else {
            // Append entire list.
        MTG_ASSERT( pTask->m_pPrev == 0 );
        MTG_ASSERT( L2.m_pFirstTask == 0 ||
                    L2.m_pLastTask->m_nCurRound == pTask->m_nCurRound ||
                    L2.m_pLastTask->m_nCurRound == pTask->m_nCurRound - 1 );

            // Don't assume pTask is in any list. Find end.
            // Adjust parameters (don't touch m_nCurRound!).

        tTask* p = pTask;
        while( p != 0 ) {
            p->m_nLevel = nLevel;
            p = p->m_pNext;
        }
    }

        // Merge and sort by tag.

    tTask* p = L2.m_pFirstTask;
    tTask** q = &L2.m_pFirstTask;
    L2.m_pLastTask = 0;

    while( p != 0 && pTask != 0 ) {
            // first criterion: by pairs of tags, decreasing:
        int c = p->m_nTag / 2 - pTask->m_nTag / 2;

            // second criterion: by size, increasing:
        if( c == 0 )
            c = pTask->m_nSize - p->m_nSize;

        if( c >= 0 ) {
            p->m_pPrev = L2.m_pLastTask;
            *q = L2.m_pLastTask = p;
            p = p->m_pNext;
        }
        else {
            pTask->m_pPrev = L2.m_pLastTask;
            *q = L2.m_pLastTask = pTask;
            pTask = pTask->m_pNext;
        }
        q = &((*q)->m_pNext);
    }

    if( p != 0 ) {
        p->m_pPrev = L2.m_pLastTask;
        *q = L2.m_pLastTask = p;
    }
    else 
    if( pTask != 0 ) {
        pTask->m_pPrev = L2.m_pLastTask;
        *q = L2.m_pLastTask = pTask;
    }
    else {
        *q = 0;
    }

    while( L2.m_pLastTask->m_pNext != 0 )
        L2.m_pLastTask = L2.m_pLastTask->m_pNext;

    if( nLevel > m_nCurLevel )
        L2.m_pCurTask = L2.m_pFirstTask;
}       


//
//   a p p e n d T a s k
//

tRetCode tFDEngine::appendTask( int nLevel, tSlotLayer* pLayer )   

{
    tRetCode nRet;

    tTask* pTask = new tTask;

    pTask->m_nLevel = nLevel;
    pTask->m_nTag = pLayer->tag();
    pTask->m_nSize = pLayer->signature()->flagsOn();
    pTask->m_nDoCnt = 1;
    pTask->m_pNext = 0;
    pTask->m_pPrev = 0;
    pTask->m_pLayer = pLayer;
    pTask->m_pSpecific = 0;

    pTask->m_nCurRound = 0;
    pTask->m_pLayer->prepareRollback();

    const tSignature* pSig = pLayer->signature();
    MTG_ASSERT( pSig != 0 );

    if( pTask->m_nSize == 1 )
        pTask->m_nMode = xSingleton;
    else
        pTask->m_nMode = xNonLinear;

    modifyMode( pLayer->tag(), pTask->m_nMode );
    MTG_ASSERT( pTask->m_nTag % 2 == 0 || pTask->m_nMode != xNonLinear );

    if( ( nRet = m_SigAssoc.insert( *pSig, pTask ) ) != OK ) {
        delete pTask;
        return nRet;
    }

        // Now insert into the fast lookup table for singletons.

    if( pTask->m_nSize == 1 && pTask->m_nTag >= 0 ) {
        int nIndex = 0;
        while( ! (*pSig)[nIndex] )
            ++nIndex;

        if( m_ClaimAssoc.numOfRows() <= pTask->m_nTag ||
            m_ClaimAssoc[pTask->m_nTag][nIndex] == 0 ) {
            if( m_ClaimAssoc.numOfRows() <= pTask->m_nTag ) {
                int k = m_ClaimAssoc.numOfRows();

                m_ClaimAssoc.numOfRows( pTask->m_nTag + 1 );
                while( k <= pTask->m_nTag ) {
                    for( int i = 0; i < portfolio().m_Sorted.numOfElems(); ++i )
                        m_ClaimAssoc[k][i] = 0;
                    ++k;
                }
            }
        }
        m_ClaimAssoc[pTask->m_nTag][nIndex] = pTask;
    }

#if defined(_DEBUG)
    MTG_TRACE( "Engine: new task for" );
    for( int i = 0; i < portfolio().m_Sorted.numOfElems(); ++i ) {
        if( (*pSig)[i] )
            MTG_TRACE( " %d", i );
    }
    MTG_TRACE( ", tag %d\n", pTask->m_nTag );
#endif

    moveTask( pTask, nLevel );
    return OK;
}


//
//   a p p e n d T a s k
//

tRetCode tFDEngine::appendTask( const tSignature& Sig )

{
    tRetCode nRet;
    tSlotLayer* pLayer;
    
    if( ( nRet = m_pRunLattice->createNonAuxLayer( &portfolio(),
                    &Sig, pLayer ) ) != OK ) {
        return nRet;
    }
       
    modifyNonAuxLayer( *pLayer );
    appendLevel();  // Make sure level exists.

    if( ( nRet = appendTask( m_Level.numOfElems() - 1, pLayer ) ) != 0 ) {
        delete pLayer;
        return nRet;
    }

    return OK;
}


//
//   d o T a s k
//

tRetCode tFDEngine::doTask( tTask* pTask )

{
    m_nSlice = m_pRunLattice->numOfSlices() - m_nCurRound / 2 - 1;
    m_bIsFinal = ( m_nCurRound <= 1 );

    m_nDay = m_pRunLattice->day( m_nSlice );
    m_gFractionOfDay = m_pRunLattice->fractionOfDay( m_nSlice );
    m_gAbsDuration = m_pRunLattice->absDuration( m_nSlice );
    m_gRelDuration = m_pRunLattice->relDuration( m_nSlice );

    if( hasProfile() ) {
        try {
            profile().putTask( *pTask->m_pLayer, m_nSlice,
                m_nDay, m_gFractionOfDay );
        }
        catch( tException E ) {
            return E.ret();
        }
    }

    if( m_bIsFinal ) {
        m_nToDay = m_nDay;
    }
    else {
        m_nToDay = m_pRunLattice->day( m_nSlice + 1 );
        if( m_nCurRound % 2 == 0 )
            pTask->m_pLayer->rotate();
        if( m_nToDay > m_nDay ) {
                // afterRollback must be called for m_nDay + 1,
                // to catch all the instruments which are "born"
                // after m_nDay
            if( m_nCurRound % 2 == 0 )
                pTask->m_pLayer->afterRollback( m_nDay + 1 );
            --m_nToDay;
        }
    }

    m_pCurTask = pTask;

    if( m_nCurRound % 2 == 0 ) {
        if( m_bIsFinal || atDawn() )
            pTask->m_pLayer->beforeRollback( m_nDay );
        m_nMode = pTask->m_nMode;
        m_bPreprocess = false;
        m_bPostprocess = false;
    }
    else {
        m_nMode = pTask->m_nBetwMode;
        m_bPreprocess = pTask->m_bBetwPreprocess;
        m_bPostprocess = pTask->m_bBetwPostprocess;
    }

    try{
        if( m_nCurRound % 2 == 0 )
            doNonAuxTask1( *pTask->m_pLayer, pTask->m_pSpecific );
        else
            doNonAuxTask2( *pTask->m_pLayer, pTask->m_pSpecific );
    }
    catch( tException E ) {
        if( E.ret() == REPEAT ) {
            if( m_nCurRound % 2 == 0 )
                pTask->m_pLayer->rotate();  // rotate back

            if( hasProfile() ) {
                try {
                    profile().putRepeat();
                }
                catch( tException F ) {
                    return F.ret();
                }
            }
        }
        return E.ret();
    }

    if( m_nCurRound % 2 == 0 ) {
        pTask->m_nBetwMode = m_nMode;
        pTask->m_bBetwPreprocess = m_bPreprocess;
        pTask->m_bBetwPostprocess = m_bPostprocess;
    }

    return OK;
}


//
//   b e f o r e R o u n d
//

void tFDEngine::beforeRound()

{
    m_Level.numOfElems( 1 );

    m_Level[0].m_nCurRound = m_nCurRound;
    m_Level[0].m_nToRound = m_nCurRound + 1;
    m_Level[0].m_pFirstTask = m_pFirstTask;
    m_Level[0].m_pLastTask = m_pLastTask;
    m_Level[0].m_pCurTask = m_pFirstTask;
}


//
//   d o R o u n d
//

tRetCode tFDEngine::doRound()

{
    tRetCode nRet;

#if defined(_DEBUG)
    int nCurRound = m_nCurRound;
#endif

    beforeRound();

    while( m_Level.numOfElems() > 0 ) {
        m_nCurLevel = m_Level.numOfElems() - 1;
        tLevel* L = &m_Level[m_nCurLevel];

        while( L->m_nCurRound < L->m_nToRound ) {
            m_nCurRound = L->m_nCurRound;

            while( m_nCurLevel == m_Level.numOfElems() - 1 &&
                   L->m_pCurTask != 0 ) {
        
                if( L->m_pCurTask->m_nCurRound == L->m_nCurRound ) {
                    nRet = doTask( L->m_pCurTask );
                        // need to verify L, since the array can grow in doTask
                    L = &m_Level[m_nCurLevel];

                    if( nRet != OK ) {
                        if( nRet != REPEAT ) 
                            return nRet;
                    }
                    else {
                        ++L->m_pCurTask->m_nCurRound;
                        L->m_pCurTask = L->m_pCurTask->m_pNext;
                    }
                }
                else {
                    L->m_pCurTask = L->m_pCurTask->m_pNext;
                }
            } 

            if( L->m_pCurTask == 0 ) {
                    // Round has been successfully completed.
                ++L->m_nCurRound;
                L->m_pCurTask = L->m_pFirstTask;
            }
            else {
                    // A new level has been created; interrupt.
                MTG_ASSERT( m_nCurLevel < m_Level.numOfElems() - 1 );
                break;
            }
        }

        if( L->m_nCurRound == L->m_nToRound )
            decreaseLevel();
    }

    MTG_ASSERT( nCurRound == m_nCurRound );
    afterRound();
    return OK;
}


//
//   a f t e r R o u n d
//

void tFDEngine::afterRound()

{
    ++m_nCurRound;

#if defined(_DEBUG)
    tTask* p = m_pFirstTask;

    while( p != 0 ) {
        MTG_ASSERT( p->m_nCurRound == m_nCurRound );
        p = p->m_pNext;
    }
#endif
}


//
//   c r e a t e N o n A u x L a y e r
//

tRetCode tFDEngine::createNonAuxLayer( int nTag, tSlotLayer*& pLayer )

{
    tRetCode nRet;

    if( portfolio().m_Sorted.numOfElems() > 0 ) {
        if( ( nRet = m_pRunLattice->createNonAuxLayer( &portfolio(),
                        nTag, pLayer ) ) != OK ) {
            return nRet;
        }
        modifyNonAuxLayer( *pLayer );
        if( ( nRet = appendTask( 0, pLayer ) ) != 0 ) {
            delete pLayer;
            return nRet;
        }
    }
    return OK;
}


//
//   g e t N o n A u x L a y e r
//

void tFDEngine::getNonAuxLayer( const tSignature& Sig, int nTag,
    tSlotLayer*& pLayer, tMode& nMode )

{
    tRetCode nRet;
    tTask* pTask;

    if( ( nRet = m_SigAssoc.retrieve( Sig, nTag, pTask ) ) != OK ) {
        if( nRet != NOT_FOUND )
            throw tException( nRet );
        if( nTag != Sig.tag() ) {
            tSignature S( Sig ); 
            S.setTag( nTag );
            if( ( nRet = appendTask( S ) ) != OK )
                throw tException( nRet );
        }
        else {
            if( ( nRet = appendTask( Sig ) ) != OK )
                throw tException( nRet );
        }
        throw tException( REPEAT );
    }

    MTG_ASSERT( pTask->m_nCurRound > m_nCurRound );

    if( pTask->m_nCurRound > m_nCurRound + 1 ) {
        appendLevel();
        moveTask( pTask, m_nCurLevel + 1 );
        throw tException( REPEAT );
    }

    pLayer = pTask->m_pLayer;
    nMode = pTask->m_nMode;
}


//
//   g e t A u x C l a i m
//

void tFDEngine::getAuxClaim( int nId, double& gUnitValue )

{
    tPortfolio::tWrapper *pWrapper;

    if( ( pWrapper = portfolio().find( nId ) ) == 0 )
        throw tException( NOT_FOUND );

    if( ! pWrapper->m_pClaim->isAuxiliary() )
        throw tException( NOT_AUXILIARY );

    MTG_ASSERT( m_pAuxTask != 0 );

        // Retrieve information.

    tSlotLayer* pLayer = m_pAuxTask->m_pLayer;
    int nIndex = pWrapper->m_nIndex;

        // Now come the checks.

    MTG_ASSERT( m_pAuxTask->m_nCurRound >= m_nCurRound );

    if( m_pAuxTask->m_nLevel != m_nCurLevel ||
            m_pAuxTask->m_nCurRound > m_nCurRound + 1 ) {
        appendLevel();
        moveTask( m_pAuxTask, m_nCurLevel + 1 );
        throw tException( REPEAT );
    }

        // Check whether this slot still holds this claim.

    if( ! m_pAuxTask->m_pLayer->wrapperMatches( pWrapper ) )
        throw tException( INVALID_ID );

    gUnitValue = pLayer->curValue( m_Pos, nIndex );
}


//
//   g e t C l a i m
//

void tFDEngine::getClaim( int nIndex, int nTag, double& gUnitValue )

{
    MTG_ASSERT( nTag >= 0 && ! isLinear() );

    if( m_ClaimAssoc.numOfRows() <= nTag || m_ClaimAssoc[nTag][nIndex] == 0 ) {
        tSignature S( m_SigAssoc.numOfFlags() );
        tRetCode nRet;

        S.setTag( nTag );
        S |= nIndex;

        if( ( nRet = appendTask( S ) ) != OK )
            throw tException( nRet );
        throw tException( REPEAT );
    }

    tTask* pTask = m_ClaimAssoc[nTag][nIndex];

    MTG_ASSERT( pTask->m_nCurRound > m_nCurRound );

    if( pTask->m_nCurRound > m_nCurRound + 1 ) {
        appendLevel();
        moveTask( pTask, m_nCurLevel + 1 );
        throw tException( REPEAT );
    }
    
        // Claim is in first position. The value of the claim
        // is returned as if it was not exercised now. That's
        // why we required nonlinearity above.

    gUnitValue = pTask->m_pLayer->prepRow( m_Pos )[0];
}


//
//   b e f o r e R u n
//

tRetCode tFDEngine::beforeRun()

{                                    
    tRetCode nRet;
    tSlotLayer* pLayer;

    if( ( nRet = super::beforeRun() ) != OK )
        return nRet;

    if( ( nRet = m_pRunLattice->beforeRun() ) != OK )
        return nRet;

        // Initialize the lookup table per instrument.

    m_ClaimAssoc.reset( portfolio().m_Sorted.numOfElems() );

        // Initialize NonAux instruments.

    m_UnderControl.numOfElems( portfolio().m_Sorted.numOfElems() );

    for( int i = 0; i < portfolio().m_Sorted.numOfElems(); ++i )  {
        int nIndex = portfolio().m_Sorted[i]->m_nIndex;

        m_UnderControl[nIndex] =
            scenario().underControl( multiplier( nIndex ) );
    }

        // Note that we have two rounds per timestep.

    m_nCurLevel = -1;
    m_nCurRound = 0;

    m_Level.numOfElems( 1 );
    m_Level[0].m_pFirstTask = 0;
    m_Level[0].m_pLastTask = 0;

    if( portfolio().m_aAuxByBirth.numOfElems() > 0 ) {
        if( ( nRet = m_pRunLattice->createAuxLayer( &portfolio(),
                        pLayer ) ) != OK ) {
            return nRet;
        }
        if( ( nRet = appendTask( 0, pLayer ) ) != 0 ) {
            delete pLayer;
            return nRet;
        }
    }

    if( ( nRet = createNonAuxLayer( 0, m_pResultLayer ) ) != OK )
        return nRet;

    m_Pos.numOfElems( m_pRunLattice->dimension() );

    return OK;
}


//
//   a f t e r R u n
//

tRetCode tFDEngine::afterRun()

{
    for( int i = 0; i < m_Pos.numOfElems(); ++i )
        m_Pos[i] = 0;

#if defined(_DEBUG)
    tTask* p = m_pFirstTask;

    while( p != 0 ) {
        MTG_TRACE( "Engine: task for" );
        for( int j = 0; j < portfolio().m_Sorted.numOfElems(); ++j ) {
            if( (*p->m_pLayer->signature())[j] ) {
                MTG_TRACE( " %d:%s", j,
                    portfolio().m_Sorted[j]->m_pClaim->comment() );
            }
        }
        MTG_TRACE( ", tag %d, was (re)started %d time%s\n",
            p->m_nTag, p->m_nDoCnt, p->m_nDoCnt == 1 ? "" : "s" );
        p = p->m_pNext;
    }
#endif

    return super::afterRun();
}


//
//   t F D E n g i n e
//

tFDEngine::tFDEngine()

{
    init();
}


//
//   ~ t F D E n g i n e
//

tFDEngine::~tFDEngine()

{
    cleanup();
}


//
//   s e t L a t t i c e
//

void tFDEngine::setLattice( tLattice* pLattice )

{
    MTG_ASSERT( pLattice != 0 );

    m_pLattice = pLattice;
    setModel( m_pLattice->model() );
}


//
//   s e t S c e n a r i o
//

void tFDEngine::setScenario( tScenario* pScenario )

{
    m_pScenario = pScenario;    // may be null
}


//
//   s e t A c c u r a c y
//

void tFDEngine::setAccuracy( tAccuracy nAccuracy )

{
    m_nAccuracy = nAccuracy;
}


//
//   r u n
//

tRetCode tFDEngine::run()

{
    MTG_ASSERT( m_pLattice != 0 );
    MTG_ASSERT( m_pScenario != 0 );

    tRetCode nRet;

    cleanup();

    if( portfolio().m_Sorted.numOfElems() == 0 ) {
            // m_pRunLattice == 0 can be used as check
            // whether no calculation was performed
        return OK;
    }

    m_pRunLattice = static_cast<tLattice*>( m_pLattice->clone() );
    if( ( nRet = m_pRunLattice->finalize( m_pScenario ) ) != OK )
        return nRet;

    if( hasProfile() ) {
        tWithProfile::copyTo( m_pRunLattice );
        m_pScenario->setExhaustive( profile().isExhaustive() );

        try { 
            profile().putEngine();
        }
        catch( tException e ) {
            return e.ret();
        }
    }

    if( ( nRet = beforeRun() ) != OK )
        return nRet;

    m_pFirstTask = m_Level[0].m_pFirstTask;
    m_pLastTask = m_Level[0].m_pLastTask;

        // Two rounds per slice: one for barriers, rollback
        // and monitoring, the second for payoff and cashflow.

    int nNumOfRounds = 2 * m_pRunLattice->numOfSlices();

    for( int k = 0; k < nNumOfRounds; ++k ) {
        if( ( nRet = doRound() ) != OK ) {
            cleanup();
            return nRet;
        }
    }

    return afterRun();
}


//
//   t o t a l
//

double tFDEngine::total() const

{
    if( m_pRunLattice == 0 )
        return 0;

    for( int i = 0; i < m_Pos.numOfElems(); ++i )
        m_Pos[i] = 0;
    return m_pResultLayer->curTotal( m_Pos );
}


//
//   d e l t a
//

double tFDEngine::delta() const

{
    if( m_pRunLattice == 0 )
        return 0;

    return m_pRunLattice->calcDelta( 0, 0,
            m_pResultLayer->curTotal( 1 ),
            m_pResultLayer->curTotal( 0 ),
            m_pResultLayer->curTotal( -1 ) );
}


//
//   g a m m a
//

double tFDEngine::gamma() const

{
    if( m_pRunLattice == 0 )
        return 0;

    return m_pRunLattice->calcGamma( 0, 0,
            m_pResultLayer->curTotal( 1 ),
            m_pResultLayer->curTotal( 0 ),
            m_pResultLayer->curTotal( -1 ) );
}


//
//   g r a d i e n t
//

void tFDEngine::gradient( tHeap<double>& Gradient ) const

{
    if( m_pRunLattice == 0 ) {
        Gradient.reset();
        return;
    }

        // Gradient values are returned in the order in which
        // claims were added to the portfolio, without the
        // auxiliaries, of course.

    Gradient.numOfElems( portfolio().m_Sorted.numOfElems() );
    Gradient.fill( 0 );

    for( int i = 0; i < m_Pos.numOfElems(); ++i )
        m_Pos[i] = 0;

    for( tSlotLayer::tIter I( *m_pResultLayer ); I; ++I )
        Gradient[I.index()] = m_pResultLayer->curValue( m_Pos, I.index() );
}


//
//   f r o n t
//

void tFDEngine::front( tHeap2<double>& Front ) const

{
    Front.reset( 2 );

    if( m_pRunLattice == 0 )
        return;

    int nDown = m_pRunLattice->numOfDownLevels( 0, 0 );
    int nUp = m_pRunLattice->numOfUpLevels( 0, 0 );

    Front.numOfRows( nDown + nUp + 1 );

    for( int k = -nDown; k <= nUp; ++k ) {
        Front[k + nDown][0] = m_pRunLattice->factor( 0, k );
        Front[k + nDown][1] = m_pResultLayer->curTotal( k );
    }
}


//
//   n u m O f T a s k s
//

int tFDEngine::numOfTasks() const

{
    int n = 0;
    tTask* p = m_pFirstTask;

    while( p != 0 ) {
        ++n;
        p = p->m_pNext;
    }

    return n;
}

MTG_END_NAMESPACE