// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#if ! defined(_MTG_FDENGINE_)
#define _MTG_FDENGINE_

#include "MtgHeap2.h"
#include "MtgEngine.h"
#include "MtgLattice.h"
#include "MtgProfile.h"
#include "MtgSigAssoc.h"
#include "MtgSlotLayer.h"

MTG_BEGIN_NAMESPACE

class tModel;
class tPortfolio;
class tScenario;


//
//   t F D E n g i n e
//

class tFDEngine : public tEngine, public tWithProfile {

    typedef tEngine super;

public:

        // Used to store engine specific data between task invocations:

    struct tSpecific {
        tSpecific( tFDEngine& Engine ) {}
        virtual ~tSpecific() {}
    };

protected:

    enum tMode {
        xPrior,
        xSingleton,
        xLinear,
        xNonLinear
    };

private:

    struct tTask;

    struct tLevel {
        int m_nCurRound;
        int m_nToRound;
        tTask* m_pFirstTask;
        tTask* m_pLastTask;
        tTask* m_pCurTask;
    };

        // m_nDay+m_gFractionOfDay indicate the position
        // of the current node in time. The time unit is [day]
        // starting at time 0.

    int m_nDay;
    int m_nToDay;
    double m_gFractionOfDay;
    double m_gAbsDuration;
    double m_gRelDuration;

        // Is the model linear from now on? I.e., is there only
        // one instrument in the current layer? Subclasses can
        // switch from nonlinear to linear, but not vice versa.
        // (xSingleton is the same as linear, but in addition 
        // indicates that there is only one instrument.)

    tMode m_nMode;

    tAccuracy m_nAccuracy;

        // These elements set the goal. They are all referenced.

    tLattice* m_pLattice;
    tScenario* m_pScenario;

        // This is the lattice copied from m_pLattice, used 
        // during the run (slightly modified).

    tLattice* m_pRunLattice;
    
        // In order to find the tasks, a signature-to-task 
        // associative array is supplied.

    tSigAssoc<tTask*> m_SigAssoc;

        // For single instruments, this can be done faster
        // by using the following array:

    tHeap2<tTask*> m_ClaimAssoc;

    tHeap<tLevel> m_Level;

        // For all instruments, we know whether we
        // our the opponent controls the exercise policy.

    tHeap<bool> m_UnderControl;

    tSlotLayer* m_pResultLayer;

        // Running variables.

    tTask* m_pAuxTask;
    tTask* m_pCurTask;

    tTask* m_pFirstTask;
    tTask* m_pLastTask;

    int m_nCurRound;
    int m_nCurLevel;

    int m_nSlice;
    bool m_bIsFinal;

    bool m_bPreprocess;
    bool m_bPostprocess;

        // The current position in the lattice. The last dimension
        // may or may not reflect the (translated, absolute) position
        // of the slot under consideration. Use for access with row()
        // function!

    tHeap<int> m_Pos;

    void init();

    void appendLevel();
    void decreaseLevel();

    void moveTask( tTask* pTask, int nLevel );

    tRetCode appendTask( int nLevel, tSlotLayer* pLayer );        
    tRetCode appendTask( const tSignature& Sig );

    tRetCode doTask( tTask* pTask );

    void beforeRound();
    tRetCode doRound();
    void afterRound();

protected:

    tLattice& lattice() const     { return *m_pRunLattice; }
    tScenario& scenario() const   { return *m_pScenario; }

    tHeap<bool>& underControl()   { return m_UnderControl; }
    tHeap<int>& pos()             { return m_Pos; }

    int slice() const             { return m_nSlice; }
    bool isFinal() const          { return m_bIsFinal; }
    bool isInitial() const        { return m_nSlice == 0; }

    tMode mode() const            { return m_nMode; }

    void setLinear()              { m_nMode = xLinear; }

    bool underControl( int nIndex ) {
        return m_UnderControl[nIndex];
    }

        // absDuration() returns the absolut duration in days.
        // The rollback calculation, however, can be based
        // on days or years. The derived engine class is responsible
        // for getting the scaling right; here we simply assume
        // everything is day-based.

    virtual double scaledAbsDuration() const {
        return absDuration(); 
    }

    void cleanup();

    bool atDawn() const {
        return m_nSlice == 0 || m_pRunLattice->day( m_nSlice - 1 ) < m_nDay;
    }

    bool atDusk() const {
        return ! isFinal() && m_pRunLattice->day( m_nSlice + 1 ) > m_nDay;
    }

    bool preprocess() const {
        return m_bPreprocess;
    }

    bool postprocess() const {
        return m_bPostprocess;
    }

    void setPreprocess() {
        m_bPreprocess = true;
    }

    void setPostprocess() {
        m_bPostprocess = true;
    }

    tRetCode createNonAuxLayer( int nTag, tSlotLayer*& pLayer );

    void getNonAuxLayer( const tSignature& Sig, int nTag,
        tSlotLayer*& pLayer, tMode& nMode );

    void getNonAuxLayer( const tSignature& Sig, tSlotLayer*& pLayer,
        tMode& nMode ) {
        getNonAuxLayer( Sig, Sig.tag(), pLayer, nMode );
    }

    void getNonAuxLayer( const tSignature& Sig, int nTag,
        tSlotLayer*& pLayer ) {
        tMode nMode;
        getNonAuxLayer( Sig, nTag, pLayer, nMode );
    }

    void getNonAuxLayer( const tSignature& Sig, tSlotLayer*& pLayer ) {
        tMode nMode;
        getNonAuxLayer( Sig, Sig.tag(), pLayer, nMode );
    }

        // The following function should set the mode to xPrior
        // for the right tag. It's called for each task at
        // allocation.

    virtual void modifyMode( int nTag, tMode& nMode ) const {}

        // The following functions should modify the newly
        // allocated slot layer; for instance, the prep and
        // temp arrays can be turned on.

    virtual void modifyAuxLayer( tSlotLayer& Layer ) const {}
    virtual void modifyNonAuxLayer( tSlotLayer& Layer ) const {}

    virtual void nonAuxPreprocess() {}
    virtual void nonAuxPostprocess() {}

        // The following functions do the real work. The tSpecific
        // pointer may or may not be used to allocate auxiliary
        // space for tasks.

    virtual void doAuxTask1( tSlotLayer& Layer, tSpecific*& pSpecific ) = 0;
    virtual void doAuxTask2( tSlotLayer& Layer, tSpecific*& pSpecific ) = 0;

    virtual void doNonAuxTask1( tSlotLayer& Layer, tSpecific*& pSpecific ) = 0;
    virtual void doNonAuxTask2( tSlotLayer& Layer, tSpecific*& pSpecific ) = 0;

        // If these functions are overridden, kepp in mind that
        // beforeRun() has to be called FIRST in the overridden
        // version, and afterRun() LAST.

    tRetCode beforeRun();
    tRetCode afterRun();
    
public:

    tFDEngine();

    virtual ~tFDEngine();

        // Model comes from lattice. Note that this is only
        // the lattice template, the actual lattice will be
        // constructed later.

    void setLattice( tLattice& Lattice ) {
        setLattice( &Lattice );
    }

    void setLattice( tLattice* pLattice );

    void setScenario( tScenario& Scenario ) {
        setScenario( &Scenario );
    }

    void setScenario( tScenario* pScenario );

    void setAccuracy( tAccuracy nAccuracy );

    tRetCode run();

        // While running, these variables are accessible:

    int day() const              { return m_nDay; }
    int toDay() const            { return m_nToDay; }
    double fractionOfDay() const { return m_gFractionOfDay; }
    double absDuration() const   { return m_gAbsDuration; }
    double relDuration() const   { return m_gRelDuration; }

    bool pricedAtPrior() const   { return m_nMode == xPrior; }
    bool isSingleton() const     { return m_nMode == xSingleton; }
    bool isLinear() const        { return m_nMode != xNonLinear; }

    tAccuracy accuracy() const   { return m_nAccuracy; }

        // And these services:

        // The following function does the translation from
        // portfolio factor space to lattice factor space!

    double factor( int nFactor = 0 ) const {
        nFactor = portfolio2ModelFactor( nFactor );
        return m_pRunLattice->factor( nFactor, m_Pos[nFactor] );
    }

        // Sometimes the first factor in lattic space (!) is
        // needed, bypassing the translation mechanism.
    
    double dominant() const {
        return m_pRunLattice->factor( 0, m_Pos[0] );
    }

    void getAuxClaim( int nId, double& gUnitValue );

        // If a single claim is to be retrieved, it can be done
        // by providing its index (!) in the wrapper array. The index
        // is faster than the id; the index is known since this kind
        // of information is mostly needed for alternative versions
        // of the instruments currently under examination.

    void getClaim( int nIndex, int nTag, double& gUnitValue );

        // If run was sucessful, access results. Note that
        // delta() and gamma() return the directional 
        // derivatives of the first factor only.

    double total() const;
    double delta() const;
    double gamma() const;

    void gradient( tHeap<double>& Gradient ) const;
    void front( tHeap2<double>& Front ) const;

        // The number of subtasks, i.e. layers:
        
    int numOfTasks() const;
};

MTG_END_NAMESPACE

#endif
