// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#include "MtgIncl.h"
#include "MtgForwardCurve.h"
#include "MtgCurveInstance.h"
#include "MtgEngine.h"
#include "MtgParser.h"
#include "MtgYieldCurve.h"

#include <float.h>

MTG_BEGIN_NAMESPACE


//
//   t F o r w a r d C u r v e I n s t a n c e
//

tForwardCurve::tForwardCurveInstance::tForwardCurveInstance(
    const tForwardCurve& Curve, tEngine& Engine )
    : tCurveInstance( Curve, Engine )

{
}


//
//   ~ t F o r w a r d C u r v e I n s t a n c e
//

tForwardCurve::tForwardCurveInstance::~tForwardCurveInstance()

{
}


//
//   e v a l u a t e
//

tRetCode tForwardCurve::tForwardCurveInstance::evaluate( double gWeight )

{
        // compute r_t = - (log P(0,t+dt) - log P(0,t)) / dt

    try {
        for( int k = firstSample(); k <= lastSample(); ++k ) {
            double gSub = xlat( k );
            double gPV = m_Engine.presentValueSI( 0, gSub );

            m_Sample[k] += gWeight * gPV;
        }
    } 
    catch( tException( e ) ) {
        return e.ret();
    }
    return OK;
}


//
//   p o s t P r o c e s s
//

tRetCode tForwardCurve::tForwardCurveInstance::postProcess( double gWeight )

{
    double dt = m_Curve.dtSI();

    int a = firstSample();
    int b = lastSample();

    if( a >= b )
        return RUNTIME_ERROR;

    for( int k = a; k <= b; ++k ) {
        if( m_Sample[k] == 0 )
            m_Sample[k] = DBL_MIN;
        else
            m_Sample[k] = log( m_Sample[k] );
    }

        // use forward and backward difference for ends

    double f1 = ( m_Sample[a] - m_Sample[a + 1] ) / dt;
    double f2 = ( m_Sample[b - 1] - m_Sample[b] ) / dt;

        // use centered difference equation for others

    for( MTG_FOR_INIT( int ) k = a; k < b - 1; ++k ) {
        m_Sample[k] =
            ( m_Sample[k] - m_Sample[k + 2] ) / ( 2 * dt );
    }

    for( MTG_FOR_INIT( int ) k = b - 1; k > a; --k ) {
        m_Sample[k] = m_Sample[k - 1];
    }

    m_Sample[a] = f1;
    m_Sample[b] = f2;

    return super::postProcess( gWeight );
}


//
//   c o n t a i n s E v a l u a t i o n O f
//

bool tForwardCurve::tForwardCurveInstance::containsEvaluationOf(
    const tCurveInstance& Instance ) const

{
    if( &Instance == this )
        return false;

        // various tests... (forward and yield curves have the same
        // evaluation step)

    if( dynamic_cast<const tForwardCurveInstance*>( &Instance ) == 0 &&
        dynamic_cast<const tYieldCurve::tYieldCurveInstance*>( &Instance ) == 0 ) {
        return false;
    }

    if( ! Instance.curve().sameDateBaseAs( m_Curve ) )
        return false;

    if( Instance.curve().numOfSamples() > m_Curve.numOfSamples() )
        return false;

    return true;
}


//
//   c o p y F r o m
//

void tForwardCurve::copyFrom( const tForwardCurve& Curve )

{
    if( &Curve == this )
        return;
    super::copyFrom( Curve );
}


//
//   p a r s e P a r a m
//

tRetCode tForwardCurve::parseParam( tParser& Parser, tParseInfoStub& Info )

{
    return super::parseParam( Parser, Info );
}


//
//   t F o r w a r d C u r v e
//

tForwardCurve::tForwardCurve()

{
}


//
//   t F o r w a r d C u r v e
//

tForwardCurve::tForwardCurve( tScale nScale )
    : tCurve( nScale )

{
}


//
//   t F o r w a r d C u r v e
//

tForwardCurve::tForwardCurve( tScale nScale, int nNumOfPeriods )
    : tCurve( nScale, nNumOfPeriods )

{
}


//
//   t F o r w a r d C u r v e
//

tForwardCurve::tForwardCurve( tScale nScale, const tDayCount& DayCount )
    : tCurve( nScale, DayCount )

{
}


//
//   t F o r w a r d C u r v e
//

tForwardCurve::tForwardCurve( tScale nScale, int nNumOfPeriods,
    const tDayCount& DayCount )
    : tCurve( nScale, nNumOfPeriods, DayCount )

{
}


//
//   t F o r w a r d C u r v e
//

tForwardCurve::tForwardCurve( tScale nScale, const tDayCount& DayCount,
    tDate Base )
    : tCurve( nScale, DayCount, Base )

{
}


//
//   t F o r w a r d C u r v e
//

tForwardCurve::tForwardCurve( tScale nScale, int nNumOfPeriods,
    const tDayCount& DayCount, tDate Base )
    : tCurve( nScale, nNumOfPeriods, DayCount, Base )

{
}


//
//   t F o r w a r d C u r v e
//

tForwardCurve::tForwardCurve( const tForwardCurve& Curve )

{
}


//
//   ~ t F o r w a r d C u r v e
//

tForwardCurve::~tForwardCurve()

{
}


//
//   o p e r a t o r =
//

tForwardCurve& tForwardCurve::operator=( const tForwardCurve& Curve )

{
    if( this != &Curve )
        copyFrom( Curve );
    return *this;
}


//
//   c l o n e
//

tObject* tForwardCurve::clone() const

{
    return new tForwardCurve( *this );
}


//
//   f i n a l i z e
//

tRetCode tForwardCurve::finalize()

{
    return super::finalize();
}


//
//   c r e a t e I n s t a n c e
//

tRetCode tForwardCurve::createInstance( tEngine& Engine,
    tCurveInstance*& pCurve ) const

{
    pCurve = new tForwardCurveInstance( *this, Engine );
    return OK;
}

MTG_END_NAMESPACE

