// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#include "MtgIncl.h"
#include "MtgGeoImplicit.h"
#include "MtgGeoSpaceAxis.h"

MTG_BEGIN_NAMESPACE


//
//   i n i t
//

void tGeoImplicit::init()

{
}


//
//   c a l c W e i g h t s
//

void tGeoImplicit::calcWeights( int nFromLevel, int nToLevel,
    const tProcessParamsStub& Params )

{
    const tProcessParams& P = static_cast<const tProcessParams&>( Params );

        // Here we compute the transition weights for a geometric Brownian
        // motion process on a (possibly) non-uniform lattice,
        // where the up- resp. down-move in log space is determined
        // by \sigma_U \sqrt{dt} and \sigma_D \sqrt{dt}.

        // The tGeoSpaceAxis::tRegion object offers a few relevant
        // helper quantities:
        //
        //    m_gU2            \sigma_U^2
        //    m_gD2            \sigma_D^2
        //    m_gDU            \sigma_D + \sigma_U
        //    m_gD2U2          \sigma_D^2 - \sigma_U^2    
        //    m_gUD2DU2        \sigma_U \sigma_D^2 + \sigma_D \sigma_U^2
        //
        // and, of course, m_gUpSigmaBar and m_gDownSigmaBar themselves.

    double gVol2 = P.m_gVol * P.m_gVol;
    double dt2 = sqrtAbsDuration();
    double g2rDt1 = 2 * ( P.m_gDiscount * absDuration() + 1 );
    double gDt2_2 = dt2 / 2;
    double gMuDt2 = P.m_gDrift * dt2;

    int nLevel = nFromLevel; 

        // The economic parameters in the Params argument hold for
        // all levels from nFromLevel to nToLevel. Since the lattice
        // might be non-uniform, however, we might have to compute
        // more than one set of weights. Within "regions", the lattice
        // is uniform.

    while( nLevel <= nToLevel ) {
        tGeoSpaceAxis::tRegion& R = axis().region( nLevel );

            // We have to compute the weights for the equation
            //
            //   A W[i-1] + B W[i] + C W[i+1] = 
            //       D V[i-1] + E V[i] + F V[i+1]
            //
            // which arises in the implicit scheme. Here, W represents
            // the new values, V the old values. (In an explicit scheme,
            // we would have A == C == 0 and B == 1.)

        double A = gVol2 * ( R.m_gUpSigmaBar + gDt2_2 * R.m_gU2 ) -
                   gMuDt2 * R.m_gU2;
        double B = gVol2 * ( R.m_gDU - gDt2_2 * R.m_gD2U2 ) + 
                   gMuDt2 * R.m_gD2U2;
        double C = gVol2 * ( R.m_gDownSigmaBar - gDt2_2 * R.m_gD2 ) +
                   gMuDt2 * R.m_gD2;
        double E = 2 * R.m_gUD2DU2 - B;

        B += g2rDt1 * R.m_gUD2DU2;

        int nTo = R.m_nToLevel;
        if( R.m_nFromLevel > nTo )
            nTo = R.m_nFromLevel;

        do {
            tRBWeights& W = RBWeights( nLevel );

            W.m_gCurD = -A;
            W.m_gCurM = B;
            W.m_gCurU = -C;
            W.m_gLastD = A;     // D == A
            W.m_gLastM = E;
            W.m_gLastU = C;     // F == C

            ++nLevel;
        } while( nLevel <= nToLevel && nLevel <= nTo );
    }
}


//
//   a d j u s t Z e r o G a m m a U p B o u n d a r y
//

void tGeoImplicit::adjustZeroGammaUpBoundary( int nLevel,
    tRBWeights& RBWeights, tZGWeights& ZGWeights )

{
        // First, adjust the weights at the
        // boundary level. Derive the "C" parameter in
        //
        //   A W[i-1] + B W[i] + C W[i+1] = 
        //       D V[i-1] + E V[i] + F V[i+1]
        //
        // so that gamma becomes zero and modify the
        // right side parameters accordingly.

    tGeoSpaceAxis::tRegion& R = axis().region( nLevel );
    double g = R.m_gUpSigmaBar / R.m_gDownSigmaBar;

    RBWeights.m_gCurM += RBWeights.m_gCurU * ( 1 + g );
    RBWeights.m_gCurD -= RBWeights.m_gCurU * g;
    RBWeights.m_gCurU = 0;

        // Then, store the extrapolation factors for the
        // first level beyond the boundary.

    ZGWeights.m_gBoundary = 1 + g;
    ZGWeights.m_gInner = -g;
}


//
//   a d j u s t Z e r o G a m m a D o w n B o u n d a r y
//

void tGeoImplicit::adjustZeroGammaDownBoundary( int nLevel,
    tRBWeights& RBWeights, tZGWeights& ZGWeights )

{
        // First, adjust the weights at the
        // boundary level. Derive the "A" parameter in
        //
        //   A W[i-1] + B W[i] + C W[i+1] = 
        //       D V[i-1] + E V[i] + F V[i+1]
        //
        // so that gamma becomes zero and modify the
        // right side parameters accordingly.

    tGeoSpaceAxis::tRegion& R = axis().region( nLevel );
    double g = R.m_gDownSigmaBar / R.m_gUpSigmaBar;

    RBWeights.m_gCurM += RBWeights.m_gCurD * ( 1 + g );
    RBWeights.m_gCurU -= RBWeights.m_gCurD * g;
    RBWeights.m_gCurD = 0;

        // Then, store the extrapolation factors for the
        // first level beyond the boundary.

    ZGWeights.m_gBoundary = 1 + g;
    ZGWeights.m_gInner = -g;
}


//
//   t G e o I m p l i c i t
//

tGeoImplicit::tGeoImplicit( tGeoSpaceAxis* pAxis )
    : super( pAxis->m_nRootLevel, pAxis->m_nNumOfUpLevels,
             pAxis->m_nNumOfDownLevels ),
      tGeoSolver( pAxis )

{
    init();
}


//
//   t G e o I m p l i c i t
//

tGeoImplicit::tGeoImplicit( const tGeoImplicit& Implicit )

{
    init();
    copyFrom( Implicit );
}


//
//   ~ t G e o I m p l i c i t
//

tGeoImplicit::~tGeoImplicit()

{
}


//
//   o p e r a t o r =
//

tGeoImplicit& tGeoImplicit::operator=( const tGeoImplicit& Implicit )

{
    if( this != &Implicit )
        copyFrom( Implicit );
    return *this;
}


//
//   c o p y F r o m
//

void tGeoImplicit::copyFrom( const tGeoImplicit& Implicit )

{
    if( this == &Implicit )
        return;

    super::copyFrom( Implicit );
    tGeoSolver::copyFrom( Implicit );
}

MTG_END_NAMESPACE

