// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#include "MtgIncl.h"
#include "MtgHJMEngine.h"
#include "MtgHJMGaussianModel.h"
#include "MtgPathSpace.h"

MTG_BEGIN_NAMESPACE


//
//   i n i t
//

void tHJMEngine::init()

{
}


//
//   b e f o r e R u n
//

tRetCode tHJMEngine::beforeRun()

{
    tRetCode nRet;
    
    if( ( nRet = super::beforeRun() ) != OK )
        return nRet;

    const tModel& Model = model();
    const tPathSpace& Space = pathSpace();

        // know only about gaussian HJM models at this time

    const tHJMGaussianModel* pGModel =
        dynamic_cast<const tHJMGaussianModel*>( &Model );

    MTG_ASSERT( pGModel != 0 );

        // initialize initial forward rate curve

    static_cast<tDateBase&>( m_TermStruct ) = Space;
    if( ( nRet = pGModel->loadInitial( m_TermStruct,
                    Space.numOfSamples() ) ) != OK ) {
        return nRet;
    }

    return OK;
}


//
//   a f t e r R u n
//

tRetCode tHJMEngine::afterRun()

{
    return super::afterRun();
}


//
//   p r e p a r e P a t h
//

tRetCode tHJMEngine::preparePath()

{
    tRetCode nRet;

        // assume that the whatever model has created a
        // evolution structure that contains a evolution
        // substructure based on tHJMTermStruct::tEvolutionStub,
        // and simulate a cross-cast

    tHJMGaussianModel::tEvolution* pEvolution =
        dynamic_cast<tHJMGaussianModel::tEvolution*>( &evolution() );
    MTG_ASSERT( pEvolution != 0 );

    if( ( nRet = m_TermStruct.finalize( pEvolution->proxy() ) ) != OK )
        return nRet;

    return OK;
}


//
//   t H J M E n g i n e
//

tHJMEngine::tHJMEngine()

{
    init();
}


//
//   ~ t H J M E n g i n e
//

tHJMEngine::~tHJMEngine()

{
}


//
//   d a t e B a s e
//

const tDateBase& tHJMEngine::dateBase() const

{
    return pathSpace();
}


//
//   d a t e R a n g e
//

const tDateRange& tHJMEngine::dateRange() const

{
    return pathSpace();
}


//   f o r w a r d
//

double tHJMEngine::forward( double gFromDay, double gToDay ) const

{
    if( m_TermStruct.scale() != xDay )
        throw tException( NOT_AVAILABLE );

    if( ! m_TermStruct.isCoveredTU( gFromDay ) ||
        ! m_TermStruct.isCoveredTU( gToDay ) ) {
        throw tException( OUT_OF_RANGE );
    }

    return m_TermStruct.forwardTU( gFromDay, gToDay );
}


//
//   f o r w a r d
//

double tHJMEngine::forward( tDate Start, tDate End ) const

{
    if( ! m_TermStruct.isCovered( Start ) ||
        ! m_TermStruct.isCovered( End ) ) {
        throw tException( OUT_OF_RANGE );
    }

    return m_TermStruct.forward( Start, End );
}


//
//   p r e s e n t V a l u e
//

double tHJMEngine::presentValue( double gFromDay, double gToDay ) const

{
    if( m_TermStruct.scale() != xDay )
        throw tException( NOT_AVAILABLE );

    if( ! m_TermStruct.isCoveredTU( gFromDay ) ||
        ! m_TermStruct.isCoveredTU( gToDay ) ) {
        throw tException( OUT_OF_RANGE );
    }

    return m_TermStruct.presentValueTU( gFromDay, gToDay );
}


//
//   f u t u r e V a l u e
//

double tHJMEngine::futureValue( double gFromDay, double gToDay ) const

{
    if( m_TermStruct.scale() != xDay )
        throw tException( NOT_AVAILABLE );

    if( ! m_TermStruct.isCoveredTU( gFromDay ) ||
        ! m_TermStruct.isCoveredTU( gToDay ) ) {
        throw tException( OUT_OF_RANGE );
    }

    return m_TermStruct.futureValueTU( gFromDay, gToDay );
}


//
//   p r e s e n t V a l u e
//

double tHJMEngine::presentValue( tDate Start, tDate End ) const

{
    if( ! m_TermStruct.isCovered( Start ) ||
        ! m_TermStruct.isCovered( End ) ) {
        throw tException( OUT_OF_RANGE );
    }

    return m_TermStruct.presentValue( Start, End );
}


//
//   f u t u r e V a l u e
//

double tHJMEngine::futureValue( tDate Start, tDate End ) const

{
    if( ! m_TermStruct.isCovered( Start ) ||
        ! m_TermStruct.isCovered( End ) ) {
        throw tException( OUT_OF_RANGE );
    }

    return m_TermStruct.futureValue( Start, End );
}


//
//   f o r w a r d F o r w a r d
//

double tHJMEngine::forwardForward( double gRefDay,
    double gFromDay, double gToDay ) const

{
    if( m_TermStruct.scale() != xDay )
        throw tException( NOT_AVAILABLE );

    if( ! m_TermStruct.isCoveredTU( gRefDay ) ||
        ! m_TermStruct.isCoveredTU( gFromDay ) ||
        ! m_TermStruct.isCoveredTU( gToDay ) ) {
        throw tException( OUT_OF_RANGE );
    }

    return m_TermStruct.forwardForwardTU( gRefDay, gFromDay, gToDay );
}


//
//   f o r w a r d F o r w a r d
//

double tHJMEngine::forwardForward( tDate Ref, tDate Start, tDate End ) const

{
    if( ! m_TermStruct.isCovered( Ref ) ||
        ! m_TermStruct.isCovered( Start ) ||
        ! m_TermStruct.isCovered( End ) ) {
        throw tException( OUT_OF_RANGE );
    }

    return m_TermStruct.forwardForward( Ref, Start, End );
}


//
//   d i s c o u n t F a c t o r
//

double tHJMEngine::discountFactor( double gRefDay,
    double gFromDay, double gToDay ) const

{
    if( m_TermStruct.scale() != xDay )
        throw tException( NOT_AVAILABLE );

    if( ! m_TermStruct.isCoveredTU( gRefDay ) ||
        ! m_TermStruct.isCoveredTU( gFromDay ) ||
        ! m_TermStruct.isCoveredTU( gToDay ) ) {
        throw tException( OUT_OF_RANGE );
    }

    return m_TermStruct.discountFactorTU( gRefDay, gFromDay, gToDay );
}


//
//   a c c r u a l F a c t o r
//

double tHJMEngine::accrualFactor( double gRefDay,
    double gFromDay, double gToDay ) const

{
    if( m_TermStruct.scale() != xDay )
        throw tException( NOT_AVAILABLE );

    if( ! m_TermStruct.isCoveredTU( gRefDay ) ||
        ! m_TermStruct.isCoveredTU( gFromDay ) ||
        ! m_TermStruct.isCoveredTU( gToDay ) ) {
        throw tException( OUT_OF_RANGE );
    }

    return m_TermStruct.accrualFactorTU( gRefDay, gFromDay, gToDay );
}


//
//   d i s c o u n t F a c t o r
//

double tHJMEngine::discountFactor( tDate Ref, tDate Start, tDate End ) const

{
    if( ! m_TermStruct.isCovered( Ref ) ||
        ! m_TermStruct.isCovered( Start ) ||
        ! m_TermStruct.isCovered( End ) ) {
        throw tException( OUT_OF_RANGE );
    }

    return m_TermStruct.discountFactor( Ref, Start, End );
}


//
//   a c c r u a l F a c t o r
//

double tHJMEngine::accrualFactor( tDate Ref, tDate Start, tDate End ) const

{
    if( ! m_TermStruct.isCovered( Ref ) ||
        ! m_TermStruct.isCovered( Start ) ||
        ! m_TermStruct.isCovered( End ) ) {
        throw tException( OUT_OF_RANGE );
    }

    return m_TermStruct.accrualFactor( Ref, Start, End );
}


//
//   f o r w a r d S I
//

double tHJMEngine::forwardSI( double gFromSub, double gToSub ) const

{
    if( ! m_TermStruct.isCoveredSI( gFromSub ) ||
        ! m_TermStruct.isCoveredSI( gToSub ) ) {
        throw tException( OUT_OF_RANGE );
    }

    return m_TermStruct.forwardSI( gFromSub, gToSub );
}


//
//   p r e s e n t V a l u e S I
//

double tHJMEngine::presentValueSI( double gFromSub, double gToSub ) const

{
    if( ! m_TermStruct.isCoveredSI( gFromSub ) ||
        ! m_TermStruct.isCoveredSI( gToSub ) ) {
        throw tException( OUT_OF_RANGE );
    }

    return m_TermStruct.presentValueSI( gFromSub, gToSub );
}


//
//   f u t u r e V a l u e S I
//

double tHJMEngine::futureValueSI( double gFromSub, double gToSub ) const

{
    if( ! m_TermStruct.isCoveredSI( gFromSub ) ||
        ! m_TermStruct.isCoveredSI( gToSub ) ) {
        throw tException( OUT_OF_RANGE );
    }

    return m_TermStruct.futureValueSI( gFromSub, gToSub );
}


//
//   f o r w a r d F o r w a r d S I
//

double tHJMEngine::forwardForwardSI( double gRefSub,
    double gFromSub, double gToSub ) const

{
    if( ! m_TermStruct.isCoveredSI( gRefSub ) ||
        ! m_TermStruct.isCoveredSI( gFromSub ) ||
        ! m_TermStruct.isCoveredSI( gToSub ) ) {
        throw tException( OUT_OF_RANGE );
    }

    return m_TermStruct.forwardForwardSI( gRefSub, gFromSub, gToSub );
}


//
//   d i s c o u n t F a c t o r S I
//

double tHJMEngine::discountFactorSI( double gRefSub,
    double gFromSub, double gToSub ) const

{
    if( ! m_TermStruct.isCoveredSI( gRefSub ) ||
        ! m_TermStruct.isCoveredSI( gFromSub ) ||
        ! m_TermStruct.isCoveredSI( gToSub ) ) {
        throw tException( OUT_OF_RANGE );
    }

    return m_TermStruct.discountFactorSI( gRefSub, gFromSub, gToSub );
}


//
//   a c c r u a l F a c t o r S I
//

double tHJMEngine::accrualFactorSI( double gRefSub,
    double gFromSub, double gToSub ) const

{
    if( ! m_TermStruct.isCoveredSI( gRefSub ) ||
        ! m_TermStruct.isCoveredSI( gFromSub ) ||
        ! m_TermStruct.isCoveredSI( gToSub ) ) {
        throw tException( OUT_OF_RANGE );
    }

    return m_TermStruct.accrualFactorSI( gRefSub, gFromSub, gToSub );
}

MTG_END_NAMESPACE
