// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#include "MtgIncl.h"
#include "MtgHJMGaussianModel.h"
#include "MtgHJMEngine.h"
#include "MtgBootstrap.h"
#include "MtgDrift.h"
#include "MtgFactor.h"
#include "MtgSamplePath.h"

MTG_BEGIN_NAMESPACE


//
//   t E v o l u t i o n
//

tHJMGaussianModel::tEvolution::tEvolution(
    const tHJMGaussianModel& Model, const tPathSpace& PathSpace )
    : m_Model( Model ), m_PathSpace( PathSpace )

{
    m_bInitialized = false;
    m_Proxy.m_pParent = this;
}


//
//   p r e p a r e
//

tRetCode tHJMGaussianModel::tEvolution::prepare()

{
        // defer rest to init() function
    setInstantiation( 0 );
    return OK;
}


//
//   s e t I n s t a n t i a t i o n
//

void tHJMGaussianModel::tEvolution::setInstantiation( int nInstantiation )

{
    m_Model.setInstantiation( m_FactorSelect, nInstantiation );
    m_bInitialized = false;
}


//
//   i n i t
//

void tHJMGaussianModel::tEvolution::init( const tHJMTermStruct& HJM,
    int nNumOfSamples, const tHeap<tHJMTermStruct::tSample>& Initial )

{
    int nNumOfFactors = m_Model.numOfFactors();

    if( m_bInitialized ) {
        MTG_ASSERT( m_M.numOfCols() == nNumOfSamples );
        MTG_ASSERT( m_M.numOfRows() == nNumOfFactors );
        MTG_ASSERT( m_S.numOfCols() == nNumOfSamples );
        MTG_ASSERT( m_S.numOfRows() == nNumOfFactors );

            // here we assume a different evolution object
            // is created for each simulation!

        return;
    }

    double dt = HJM.dtSI();
    double dt2 = sqrt( dt );

    if( m_M.numOfCols() != nNumOfSamples )
        m_M.reset( nNumOfSamples );
    if( m_M.numOfRows() != nNumOfFactors )
        m_M.numOfRows( nNumOfFactors );

    if( m_S.numOfCols() != nNumOfSamples )
        m_S.reset( nNumOfSamples );
    if( m_S.numOfRows() != nNumOfFactors )
        m_S.numOfRows( nNumOfFactors );

    for( int l = 0; l < nNumOfFactors; ++l ) {
        const tFactorParam& F = *m_Model.m_FactorParam[l];

            // select right sigma and kappa

        double gSigma = F.m_Sigma[m_FactorSelect[l + l]];
        double gKappa = F.m_Kappa[m_FactorSelect[l + l + 1]];

        if( gKappa == 0 ) {
            double q = gSigma * dt2;
            double s = dt * gSigma;
            int t = 3;

            s = s * s / 2;
            for( int d = 0; d < nNumOfSamples; ++d, t += 2 ) {
                m_M[l][d] = s * t;
                m_S[l][d] = q;
            }
        }
        else {
            double h = exp( -dt * gKappa );
            double s = gSigma / gKappa;
            double q = ( h - 1 ) * s / dt2;
            s *= s / 2;
            double t = h;

            for( int d = 0; d < nNumOfSamples; ++d ) {
                m_S[l][d] = q * t;
                double u = 1 - t;
                t *= h;
                double v = 1 - t;
                m_M[l][d] = s * ( v * v - u * u );
            }
        }

#if defined(_DEBUG)
        //for( int d = 0; d < nNumOfSamples; ++d ) {
        //    MTG_TRACE( "m[%d][%d] = %lf, s[%d][%d] = %lf\n",
        //        l, d, m_M[l][d], l, d, m_S[l][d] );
        //}
#endif
    }

    m_bInitialized = true;
}


//
//   p r o g r e s s
//

tRetCode tHJMGaussianModel::tEvolution::progress( int nStep,
    int nNumOfSamples, const tHeap<tHJMTermStruct::tSample>& Initial,
    const tHeap<tHJMTermStruct::tSample>& Last,
    tHeap<tHJMTermStruct::tSample>& Future )

{
    int nNumOfFactors = m_Model.numOfFactors();

    for( int j = nStep; j < nNumOfSamples; ++j ) {
        const double* p = path()[j];
        double f = Last[j];
        int k = j - nStep;

        for( int l = 0; l < nNumOfFactors; ++l )
            f += m_M[l][k] + m_S[l][k] * p[l];
 
        Future[j] = (tHJMTermStruct::tSample) f;
    }
    return OK;
}


//
//   i n i t
//

void tHJMGaussianModel::init()

{
    m_pInitialBoot = 0;
    m_pInitialDrift = 0;
    m_gConstInitial = 0;
}


//
//   c l e a n u p
//

void tHJMGaussianModel::cleanup()

{
    for( int i = 0; i < m_FactorParam.numOfElems(); ++i )
        delete m_FactorParam[i];
    m_FactorParam.reset();
}


//
//   c o p y F r o m
//

void tHJMGaussianModel::copyFrom( const tHJMGaussianModel& Model )

{
    if( &Model == this )
        return;

    cleanup();

    for( int i = 0; i < Model.m_FactorParam.numOfElems(); ++i ) {
        m_FactorParam.append( new tFactorParam );
        *m_FactorParam.last() = *Model.m_FactorParam[i];
    }

    m_gConstInitial = Model.m_gConstInitial;
    setObjRefToZeroOrSo( m_pInitialBoot, Model.m_pInitialBoot );
    setObjRefToZeroOrSo( m_pInitialDrift, Model.m_pInitialDrift );

    super::copyFrom( Model );
}


//
//   p a r s e F a c t o r
//

tRetCode tHJMGaussianModel::parseFactor( tParser& Parser,
    tParseInfoStub& Info )

{
    MTG_ASSERT( Parser.curToken() == xTokId );

    tRetCode nRet;
    tObject* pObj;
    tFactor* pFactor;

    if( ( pObj = Parser.findObject() ) == 0 )
        return Parser.setError( NOT_FOUND );

    if( ( pFactor = dynamic_cast<tFactor*>( pObj ) ) == 0 ) 
        return Parser.setError( OBJECT_MISMATCH );

    if( ( nRet = Parser.readToken() ) != OK )
        return nRet;

    if( ( nRet = appendFactor( pFactor ) ) != OK )
        return nRet;

    if( Parser.curToken() == xTokSigma ) {
        if( ( nRet = Parser.readToken() ) != OK )
            return nRet;
    }

    tFactorParam* pParam = new tFactorParam;

    if( ( nRet = Parser.scanPosPercentageRange( pParam->m_Sigma ) ) != OK ) {
        delete pParam;
        return nRet;
    }

    if( pParam->m_Sigma.numOfElems() == 0 ) {
        delete pParam;
        return Parser.setError( MISSING_RANGE );
    }

    bool bNeedsKappa = false;

    if( Parser.curToken() == xTokKappa ) {
        if( ( nRet = Parser.readToken() ) != OK )
            return nRet;
        bNeedsKappa = true;
    }

    if( Parser.beginOfNumber() || Parser.beginOfObj() ) {
        if( ( nRet = Parser.scanNonNegPercentageRange( 
                pParam->m_Kappa ) ) != OK ) {
            delete pParam;
            return nRet;
        }
    }
    else {
        if( bNeedsKappa ) {
            delete pParam;
            return Parser.setError( MISSING_RANGE );
        }
        pParam->m_Kappa.append( 0 );
    }

    m_FactorParam.append( pParam );
    return OK;
}


//
//   p a r s e P a r a m
//

tRetCode tHJMGaussianModel::parseParam( tParser& Parser,
    tParseInfoStub& Info )

{
    tRetCode nRet;
    tObject* pObj;
    tBootstrap* pBoot;
    tDrift* pDrift;

    tParseInfo& I = static_cast<tParseInfo&>( Info );

    switch( Parser.curToken() ) {
        case xTokInitial :
            if( m_pInitialBoot != 0 || m_pInitialDrift != 0 || I.m_bParam1 )
                return Parser.setError( ATTR_REDEFINITION );
            if( ( nRet = Parser.readToken() ) != OK )
                return nRet;
            if( Parser.beginOfNumber() ) {
                if( ( nRet =
                        Parser.scanPosPercentage( m_gConstInitial ) ) != OK ) {
                    return nRet;
                }
                I.m_bParam1 = true;
            }
            else {
                if( Parser.curToken() != xTokId )
                    return Parser.setError( INVALID_KEYWORD );
                if( ( pObj = Parser.findObject() ) == 0 )
                    return Parser.setError( NOT_FOUND );
                if( ( pBoot = dynamic_cast<tBootstrap*>( pObj ) ) != 0 )
                    setObjRefToZeroOrSo( m_pInitialBoot, pBoot );
                else
                if( ( pDrift = dynamic_cast<tDrift*>( pObj ) ) != 0 )
                    setObjRefToZeroOrSo( m_pInitialDrift, pDrift );
                else
                    return Parser.setError( "bootstrap/drift expected" );   
                if( ( nRet = Parser.readToken() ) != OK )
                    return nRet;
            }
            break;

        case xTokFactor :
            if( ( nRet = Parser.readToken() ) != OK )
                return nRet;
            if( Parser.curToken() != xTokId ) 
                return Parser.setError( INVALID_KEYWORD );

                // fall through

        case xTokId :
            if( ( nRet = parseFactor( Parser, Info ) ) != OK )
                return nRet;
            break;

        default :
            return super::parseParam( Parser, Info );
    }

    return OK;
}


//
//   p a r s e P o s t f i x
//

tRetCode tHJMGaussianModel::parsePostfix( tParser& Parser,
    tParseInfoStub& Info )

{
    tParseInfo& I = static_cast<tParseInfo&>( Info );

    if( m_pInitialBoot == 0 && m_pInitialDrift == 0 && ! I.m_bParam1 )
        return Parser.setError( MISSING_DRIFT );
    if( numOfFactors() == 0 )
        return Parser.setError( MISSING_FACTOR );
    return super::parsePostfix( Parser, Info );
}


//
//   t H J M G a u s s i a n M o d e l
//

tHJMGaussianModel::tHJMGaussianModel()

{
    init();
}


//
//   t H J M G a u s s i a n M o d e l
//

tHJMGaussianModel::tHJMGaussianModel( const tHJMGaussianModel& Model )

{
    init();
    copyFrom( Model );
}


//
//   ~ t H J M G a u s s i a n M o d e l
//

tHJMGaussianModel::~tHJMGaussianModel()

{
    setObjRefToZeroOrSo( m_pInitialBoot );
    setObjRefToZeroOrSo( m_pInitialDrift );
    cleanup();
}


//
//   o p e r a t o r =
//

tHJMGaussianModel& tHJMGaussianModel::operator=(
    const tHJMGaussianModel& Model )

{
    if( &Model != this )
        copyFrom( Model );
    return *this;
}


//
//   c l o n e
//

tObject* tHJMGaussianModel::clone() const

{
    return new tHJMGaussianModel( *this );
}


//
//   c r e a t e E n g i n e
//

tRetCode tHJMGaussianModel::createEngine( tMCEngine*& pEngine )

{
    MTG_ASSERT( isFinalized() );

    pEngine = new tHJMEngine;
    pEngine->setModel( this );
    return OK;
}


//
//   f i n a l i z e
//

tRetCode tHJMGaussianModel::finalize()

{
    return super::finalize();
}


//
//   l o a d I n i t i a l
//

tRetCode tHJMGaussianModel::loadInitial( tHJMTermStruct& TermStruct,
    int nNumOfSamples ) const

{
    MTG_ASSERT( isFinalized() );

    tRetCode nRet;

        // scale, day count convention and base date have
        // already been set for the HJM term structure!

    if( m_pInitialDrift != 0 ) {
        if( ( nRet = TermStruct.loadInitial( *m_pInitialDrift,
                nNumOfSamples ) ) != OK ) {
            return nRet;
        }
    }
    else
    if( m_pInitialBoot != 0 ) {
        if( ( nRet = TermStruct.loadInitial( m_pInitialBoot->spline(),
                nNumOfSamples ) ) != OK ) {
            return nRet;
        }
    }
    else {
        if( ( nRet = TermStruct.loadInitial( m_gConstInitial,
                nNumOfSamples ) ) != OK ) {
            return nRet;
        }
    }

    return OK;
}


//
//   n u m O f I n s t a n t i a t i o n s
//

int tHJMGaussianModel::numOfInstantiations() const

{
    MTG_ASSERT( m_FactorParam.numOfElems() > 0 );

    int n = 1;

    for( int i = 0; i < m_FactorParam.numOfElems(); ++i ) {
        int s = m_FactorParam[i]->m_Sigma.numOfElems();
        int k = m_FactorParam[i]->m_Kappa.numOfElems();

        if( n >= ( INT_MAX / s ) / k )
            return -1;
        n *= s * k;
    }

    return n;
}


//
//   s e t I n s t a n t i a t i o n
//

void tHJMGaussianModel::setInstantiation( tHeap<int>& Select,
    int nInstantiation ) const

{
    MTG_ASSERT( nInstantiation >= 0 &&
                nInstantiation < numOfInstantiations() );

    if( Select.numOfElems() == 0 )
        Select.numOfElems( 2 * m_FactorParam.numOfElems() );

    int k = Select.numOfElems();

    for( int i = numOfFactors() - 1; i >= 0; --i ) {
        const tFactorParam& P = *m_FactorParam[i];

        Select[--k] = nInstantiation % P.m_Kappa.numOfElems();
        nInstantiation /= P.m_Kappa.numOfElems();
        Select[--k] = nInstantiation % P.m_Sigma.numOfElems();
        nInstantiation /= P.m_Sigma.numOfElems();
    }

    MTG_ASSERT( k == 0 && nInstantiation == 0 );
}


//
//   c r e a t e E v o l u t i o n
//

tRetCode tHJMGaussianModel::createEvolution( const tPathSpace& PathSpace,
    tMCEngine::tEvolutionStub*& pEvolution ) const

{
    MTG_ASSERT( isFinalized() );

    tRetCode nRet;
    tEvolution* p = new tEvolution( *this, PathSpace );

    if( ( nRet = p->prepare() ) != OK ) {
        delete p;
        return nRet;
    }

    pEvolution = p;
    return OK;
}


//
//   s a v e W e i g h t s H e a d e r
//

tRetCode tHJMGaussianModel::saveWeightsHeader( ostream& Out ) const

{
    Out << numOfFactors() << " // number of factors" << endl;
    for( int i = 0; i < numOfFactors(); ++i ) {
        Out << "\"" << factor( i )->name() << "\""
            << " 2 // factor name, number of parameters\n"
            << "\"sigma\" \"kappa\" // parameter names"
            << endl;
    }

    return Out.good() ? OK : WRITE_ERROR;
}


//
//   s a v e W e i g h t s D a t a
//

tRetCode tHJMGaussianModel::saveWeightsData( ostream& Out,
    tHeap<int> Select ) const

{
    MTG_ASSERT( Select.numOfElems() == 2 * numOfFactors() );

    for( int i = 0; i < numOfFactors(); ++i ) {
        const tFactorParam& F = *m_FactorParam[i];
        
        double gSigma = F.m_Sigma[Select[2 * i]];
        double gKappa = F.m_Kappa[Select[2 * i + 1]];

        if( i > 0 )
            Out << " ";
        Out << gSigma << " " << gKappa;
    }

    return Out.good() ? OK : WRITE_ERROR;
}

MTG_END_NAMESPACE