// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#include "MtgIncl.h"

#define _MTG_HEAP_BODY_
#include "MtgHeap.h"
#undef _MTG_HEAP_BODY_ 

MTG_BEGIN_NAMESPACE


//
//   v a r i a b l e s
//

template<class T>
const int tHeap<T>::m_nIncrFrac = 5;

template<class T>
const int tHeap<T>::m_nMinIncr  = 10;


//
//   i n i t
//

template<class T>
void tHeap<T>::init()

{
    m_nNumOfElems = 0;
    m_nElemHiwater = 0;
    m_bIsSharing = false;
    m_Data = 0;
}


//
//   r e s i z e
//

template<class T>
void tHeap<T>::resize( int nTargetSize )

{
    MTG_ASSERT( m_nNumOfElems >= 0 );

    if( nTargetSize > m_nElemHiwater ) {
        T *aNew;
        int nNewHiwater;

        nNewHiwater = nTargetSize + nTargetSize / m_nIncrFrac;

        if( nNewHiwater < m_nElemHiwater + m_nMinIncr )
            nNewHiwater = m_nElemHiwater + m_nMinIncr;

        aNew = new T[nNewHiwater];
        if( m_nElemHiwater > 0 ) {
            memcpy( aNew, m_Data, m_nElemHiwater * sizeof(T) );
            if( ! m_bIsSharing )
                delete[] m_Data;
        }

        m_Data = aNew;
        m_nElemHiwater = nNewHiwater;
        m_bIsSharing = false;
    }
}


//
//   t H e a p
//

template<class T>
tHeap<T>::tHeap()

{
    init();
}


//
//   t H e a p
//

template<class T>
tHeap<T>::tHeap( int nNumOfElems, T* Data )

{
    init();

    m_nNumOfElems = nNumOfElems;
    m_nElemHiwater = nNumOfElems;
    m_Data = Data;
    m_bIsSharing = true;
}


//
//   t H e a p
//

template<class T>
tHeap<T>::tHeap( const tHeap<T>& Heap )

{
    init();
    copyFrom( Heap );
}


//
//   ~ t H e a p
//

template<class T>
tHeap<T>::~tHeap()

{
    clear();
}


//
//   c l e a r
//

template<class T>
void tHeap<T>::clear()

{
    if( m_nElemHiwater > 0 ) {
        if( ! m_bIsSharing )
            delete[] m_Data;
        m_nElemHiwater = 0;
    }
    reset();
}


//
//   o p e r a t o r =
//

template<class T>
tHeap<T>& tHeap<T>::operator=( const tHeap<T>& Heap )

{
    if( &Heap != this )
        copyFrom( Heap );
    return *this;
}


//
//   c o p y F r o m
//

template<class T>
void tHeap<T>::copyFrom( const tHeap<T>& Heap )

{
    if( &Heap == this )
        return;

    reset();
    resize( Heap.m_nNumOfElems );

    m_nNumOfElems = Heap.m_nNumOfElems;
    memcpy( m_Data, Heap.m_Data, m_nNumOfElems * sizeof(T) );
}


//
//   m i g r a t e T o
//

template<class T>
void tHeap<T>::migrateTo( tHeap<T>& Heap )

{
    if( &Heap == this )
        return;

    if( m_bIsSharing ) {
        Heap.copyFrom( *this );
        m_bIsSharing = false;
    }
    else {
        Heap.clear();
        Heap.m_nNumOfElems = m_nNumOfElems;
        Heap.m_nElemHiwater = m_nElemHiwater;
        Heap.m_Data = m_Data;
    }

    m_nNumOfElems = 0;
    m_nElemHiwater = 0;
    m_Data = 0;
}


//
//   o p e r a t o r = =
//

template<class T>
bool tHeap<T>::operator==( const tHeap<T>& Heap ) const

{
    if( Heap.m_nNumOfElems != m_nNumOfElems )
        return false;
    for( int i = 0; i < m_nNumOfElems; ++i ) {
        if( m_Data[i] != Heap.m_Data[i] )
            return false;
    }
    return true;
}


//
//   s o r t I n A s c
//

template<class T>
bool tHeap<T>::sortInAsc( const T& Data, bool bUnique )

{
    int k = m_nNumOfElems;
    while( k > 0 && m_Data[k - 1] > Data )
        --k;

    if( ! bUnique || k == 0 || m_Data[k - 1] != Data ) {
        ensureCapacity( 1 );
        for( int i = m_nNumOfElems++; i > k; --i )
             m_Data[i] = m_Data[i - 1];
        m_Data[k] = Data;
        return true;
    }

    return false;
}


//
//   s o r t I n D e s c
//

template<class T>
bool tHeap<T>::sortInDesc( const T& Data, bool bUnique )

{    
    int k = m_nNumOfElems;
    while( k > 0 && m_Data[k - 1] < Data )
        --k;

    if( ! bUnique || k == 0 || m_Data[k - 1] != Data ) {
        ensureCapacity( 1 );
        for( int i = m_nNumOfElems++; i > k; --i )
             m_Data[i] = m_Data[i - 1];
        m_Data[k] = Data;
        return true;
    }

    return false;
}


//
//   f i l l
//

template<class T>
void tHeap<T>::fill( const T& Data )

{
    for( int k = 0; k < m_nNumOfElems; ++k )
        m_Data[k] = Data;
}


//
//   s h a r e
//

template<class T>
void tHeap<T>::share( int nNumOfElems, T* Data )

{
    clear();

    m_nNumOfElems = nNumOfElems;
    m_nElemHiwater = nNumOfElems;
    m_Data = Data;
    m_bIsSharing = true;
}


//
//   l i s t i n g
//

template<class T>
tRetCode tHeap<T>::save( ostream& Out, int nElemsPerCol,
    const char* sPrefix ) const
    
{
    if( sPrefix == 0 )
        sPrefix = "";

    for( int k = 0; k < numOfElems(); ++k ) {
        if( nElemsPerCol > 0 && k % nElemsPerCol == 0 ) {
            if( k > 0 )
                Out << "\n";
            Out << sPrefix;
        }
        else {
            Out << " ";
        }
        Out << m_Data[k];
    }

    if( numOfElems() > 0 )
        Out << "\n";

    return Out.good() ? OK : WRITE_ERROR;
}

#if defined(_MTG_WITH_MATHLINK)

//
//   t o M a t h L i n k
//

template<class T>
tRetCode tHeap<T>::toMathLink( MLINK MathLink ) const

{
    int nRet;
    long nDim;

    nDim = m_nNumOfElems;

        // Speculate about the data type (only supported types are
        // int and double).

    if( sizeof(T) == sizeof(int) ) {
        nRet =
            MLPutIntegerArray( MathLink, (int*) &m_Data[0], &nDim, 0, 1 );
    }
    else {
        nRet =
            MLPutRealArray( MathLink, (double*) &m_Data[0], &nDim, 0, 1 );
    }

    if( nRet == 0 )
        return MATHLINK_ERROR;
    return OK;
}

#endif

MTG_END_NAMESPACE