// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#if ! defined(_MTG_HEAP_)
#define _MTG_HEAP_

MTG_BEGIN_NAMESPACE

// A "heap" is an array of C type objects that grows dynamically.
// Two things should be kept in mind:
//
// A) Because EFFICIENCY was more important than everything
//    else, the objects in the heap are assumed to be C types;
//    that means, copying is performed with memcpy() rather
//    than with the assignment operator! That limits its
//    usability for a lot of C++ purposes.
//    
// B) Because objects in the array can MOVE in memory, no
//    pointers or references to objects in the array must
//    be kept if new elements are appended to the array.
//
// In general, this template is meant as an internal helper template
// for the Mtg library, and not as a component of the library
// exported to the end user.


//
//   t H e a p
//

template<class T>
class tHeap {

    int m_nNumOfElems;
    int m_nElemHiwater;

        // If a heap is sharing, it doesn't own the data.
        // Resizing is not allowed in this case.

    bool m_bIsSharing;

    T *m_Data;

    static const int m_nIncrFrac;
    static const int m_nMinIncr;

    void init();
    void resize( int nTargetSize );

public:

    tHeap();
        
        // This constructor creates a sharing heap:

    tHeap( int nNumOfElems, T* Data );

        // The copy constructor:
        
    tHeap( const tHeap& Heap );

    ~tHeap();

    void reset() {
        m_nNumOfElems = 0;
    }

    void clear();

    tHeap& operator=( const tHeap& Heap );

    void copyFrom( const tHeap& Heap );
    void migrateTo( tHeap& Heap );

    void ensureCapacity( int nIncr = 1 ) {
        resize( m_nNumOfElems + nIncr );
    }

    int numOfElems() const {
        return m_nNumOfElems;
    }

    int numOfElems( int nNumOfElems ) {
        resize( nNumOfElems );
        return m_nNumOfElems = nNumOfElems;
    }

        // Make space at the end:

    tHeap& operator++() {
        ensureCapacity( 1 );
        ++m_nNumOfElems;
        return *this;
    }

    tHeap& operator+=( int nIncrNumOfElems ) {
        MTG_ASSERT( nIncrNumOfElems >= 0 );
        ensureCapacity( nIncrNumOfElems );
        m_nNumOfElems += nIncrNumOfElems;
        return *this;
    }

    tHeap& operator--() {
        if( --m_nNumOfElems < 0 )
            m_nNumOfElems = 0;
        return *this;
    }

    tHeap& operator-=( int nDecrNumOfElems ) {
        m_nNumOfElems -= nDecrNumOfElems;
        if( m_nNumOfElems < 0 )
            m_nNumOfElems = 0;
        return *this;
    }

    T& operator[]( int nPos ) const {
        MTG_ASSERT( nPos >= 0 && nPos < m_nNumOfElems );
        return m_Data[nPos];
    }

    T& last() const {
        MTG_ASSERT( m_nNumOfElems > 0 );
        return m_Data[m_nNumOfElems - 1];
    }

    void append( const T& Data ) {
        operator++();
        last() = Data;
    }

        // Stack semantics:

    void push( const T& Data ) {
        append( Data );
    }

    void pop( T& Data ) {
        Data = last();
        operator--();
    }

        // Uses memcmp():

    bool operator==( const tHeap& Heap ) const;

        // Use memcmp() for comparison of elements:

    bool sortInAsc( const T& Data, bool bUnique = true );
    bool sortInDesc( const T& Data, bool bUnique = true );

    void fill( const T& Data );

        // This function creates a sharing heap. It's mainly
        // used by tHeap2<T> to allow tHeap<T>-access to a
        // single row.

    void share( int nNumOfElems, T* Data );

        // Print the contents of the entire heap:

    tRetCode save( ostream& Out, int nElemsPerCol = -1,
        const char* sPrefix = "" ) const;

#if defined(_MTG_WITH_MATHLINK)
    tRetCode toMathLink( MLINK MathLink ) const;
#endif
};

MTG_END_NAMESPACE

#if ! defined(_MTG_HEAP_BODY_)
    #define _MTG_HEAP_HEADER_
    #if defined(_WIN32)
        #include "MtgHeap.cpp"
    #else   
        #include "MtgHeap.C"
    #endif
    #undef _MTG_HEAP_HEADER_
#endif

#endif
