// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#include "MtgIncl.h"

#define _MTG_HEAP2_BODY_
#include "MtgHeap2.h"
#undef _MTG_HEAP2_BODY_

MTG_BEGIN_NAMESPACE


//
//   v a r i a b l e s
//

template<class T>
const int tHeap2<T>::m_nIncrFrac = 5;

template<class T>
const int tHeap2<T>::m_nMinIncr  = 10;


//
//   i n i t
//

template<class T>
void tHeap2<T>::init()

{
    m_nNumOfCols = 0;
    m_nNumOfRows = 0;
    m_nRowHiwater = 0;
    m_Data = 0;
}


//
//   r e s i z e
//

template<class T>
void tHeap2<T>::resize( int nTargetRows )    

{
    MTG_ASSERT( m_nNumOfRows >= 0 );

    if( nTargetRows > m_nRowHiwater ) {
        T *aNew;
        int nNewHiwater;

        nNewHiwater = nTargetRows + nTargetRows / m_nIncrFrac;

        if( nNewHiwater < m_nRowHiwater + m_nMinIncr )
            nNewHiwater = m_nRowHiwater + m_nMinIncr;

        aNew = new T[nNewHiwater * m_nNumOfCols];
        if( m_nRowHiwater > 0 ) {
            memcpy( aNew, m_Data, m_nRowHiwater * m_nNumOfCols * sizeof(T) );
            delete[] m_Data;
        }

        m_Data = aNew;
        m_nRowHiwater = nNewHiwater;
    }
}


//
//   t H e a p 2
//

template<class T>
tHeap2<T>::tHeap2( int nNumOfCols )

{
    init();
    if( nNumOfCols > 0 ) 
        reset( nNumOfCols );
}


//
//   t H e a p 2
//

template<class T>
tHeap2<T>::tHeap2( const tHeap2<T>& Heap )

{
    init();
    copyFrom( Heap );
}


//
//   ~ t H e a p 2
//

template<class T>
tHeap2<T>::~tHeap2()

{
    clear();
}


//
//   r e s e t
//

template<class T>
void tHeap2<T>::reset()

{
    m_nNumOfRows = 0;
}


//
//   r e s e t
//

template<class T>
void tHeap2<T>::reset( int nNumOfCols )

{
    clear();
    m_nNumOfCols = nNumOfCols;
}


//
//   c l e a r
//

template<class T>
void tHeap2<T>::clear()

{
    if( m_nRowHiwater > 0 ) {
        delete[] m_Data;
        m_nRowHiwater = 0;
    }
    reset();
}


//
//   o p e r a t o r =
//

template<class T>
tHeap2<T>& tHeap2<T>::operator=( const tHeap2<T>& Heap )

{
    if( &Heap != this )
        copyFrom( Heap );
    return *this;
}


//
//   c o p y F r o m
//

template<class T>
void tHeap2<T>::copyFrom( const tHeap2<T>& Heap )

{
    if( &Heap == this )
        return;

    if( m_nNumOfCols == Heap.m_nNumOfCols )
        reset();
    else
        reset( Heap.m_nNumOfCols );

    resize( Heap.m_nNumOfRows );
    m_nNumOfRows = Heap.m_nNumOfRows;

    memcpy( m_Data, Heap.m_Data, m_nNumOfRows * m_nNumOfCols * sizeof(T) );
}


//
//   m i g r a t e T o
//

template<class T>
void tHeap2<T>::migrateTo( tHeap2<T>& Heap )

{
    if( &Heap == this )
        return;

    Heap.clear();

    Heap.m_nNumOfCols = m_nNumOfCols;
    Heap.m_nNumOfRows = m_nNumOfRows;
    Heap.m_nRowHiwater = m_nRowHiwater;
    Heap.m_Data = m_Data;

    m_nNumOfRows = 0;
    m_nRowHiwater = 0;
    m_Data = 0;
}

#if defined(_MTG_WITH_MATHLINK)

//
//   t o M a t h L i n k
//

template<class T>
tRetCode tHeap2<T>::toMathLink( MLINK MathLink ) const

{
    int nRet;
    long Dim[2];

    Dim[0] = m_nNumOfRows;
    Dim[1] = m_nNumOfCols;

        // Speculate about the data type (only supported types are
        // int and double).

    if( sizeof(T) == sizeof(int) )
        nRet = MLPutIntegerArray( MathLink, (int*) &m_Data[0], Dim, 0, 2 );
    else
        nRet = MLPutRealArray( MathLink, (double*) &m_Data[0], Dim, 0, 2 );

    if( nRet == 0 )
        return MATHLINK_ERROR;
    return OK;
}


//
//   t o M a t h L i n k
//

template<class T>
tRetCode tHeap2<T>::toMathLink( MLINK MathLink, int nRow ) const

{
    int nRet;
    long nDim;

    nDim = m_nNumOfCols;

        // Speculate about the data type (only supported types are
        // int and double).

    if( sizeof(T) == sizeof(int) )
        nRet = MLPutIntegerArray( MathLink, (int*) row( nRow ), &nDim, 0, 1 );
    else
        nRet = MLPutRealArray( MathLink, (double*) row( nRow ), &nDim, 0, 1 );

    if( nRet == 0 )
        return MATHLINK_ERROR;
    return OK;
}

#endif

MTG_END_NAMESPACE