// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#if ! defined(_MTG_HEAP2_)
#define _MTG_HEAP2_

#include "MtgHeap.h"

MTG_BEGIN_NAMESPACE

// This is the 2-dimensional version of tHeap, again, for
// C-type objects that survive memcpy() and yield consistent
// results under memcmp(). This template is mainly for
// internal purposes as well.


//
//   t H e a p 2
//

template<class T>
class tHeap2 {

    int m_nNumOfCols;
    int m_nNumOfRows;
    int m_nRowHiwater;

    T *m_Data;

    static const int m_nIncrFrac;
    static const int m_nMinIncr;

    void init();
    void resize( int nTargetRows );

public:

    tHeap2( int nNumOfCols = 1 );
    tHeap2( const tHeap2& Heap );

    ~tHeap2();

    void reset();
    void reset( int nNumOfCols );

    void clear();

    tHeap2& operator=( const tHeap2& Heap );

    void copyFrom( const tHeap2& Heap );
    void migrateTo( tHeap2& Heap );

    void ensureCapacity( int nIncr = 1 ) {
        resize( m_nNumOfRows + nIncr );
    }

    int numOfCols() const {
        return m_nNumOfCols;
    }

    int numOfRows() const {
        return m_nNumOfRows;
    }

    int numOfRows( int nNumOfRows ) {
        MTG_ASSERT( nNumOfRows >= 0 );
        resize( nNumOfRows );
        return m_nNumOfRows = nNumOfRows;
    }

    tHeap2& operator++() {
        ensureCapacity( 1 );
        ++m_nNumOfRows;
        return *this;
    }

    tHeap2& operator+=( int nIncrNumOfRows ) {
        MTG_ASSERT( nIncrNumOfRows >= 0 );
        ensureCapacity( nIncrNumOfRows );
        m_nNumOfRows += nIncrNumOfRows;
        return *this;
    }

        // The [] operator returns a C-style array.

    T* operator[]( int nRow ) const {
        MTG_ASSERT( nRow >= 0 && nRow < m_nNumOfRows );
        return &m_Data[nRow * m_nNumOfCols];
    }

        // For tHeap<T>-access to a single row, a sharing
        // heap is constructed with the following function:

    void getRow( int nRow, tHeap<T>& H ) const {
        H.share( numOfCols(), operator[]( nRow ) );
    }
    
#if defined(_MTG_WITH_MATHLINK)
    tRetCode toMathLink( MLINK MathLink ) const;
    tRetCode toMathLink( MLINK MathLink, int nRow ) const;
#endif
};

MTG_END_NAMESPACE

#if ! defined(_MTG_HEAP2_BODY_)
    #define _MTG_HEAP2_HEADER_
    #if defined(_WIN32)
        #include "MtgHeap2.cpp"
    #else   
        #include "MtgHeap2.C"
    #endif
    #undef _MTG_HEAP2_HEADER_
#endif

#endif
