// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#include "MtgIncl.h"
#include "MtgHttpDetour.h"

MTG_BEGIN_NAMESPACE


//
//   v a r i a b l e s
//

const int tHttpDetour::m_nTimeout = 600;


//
//   g e t L i n e
//

bool tHttpDetour::getLine( char* sCurrent, char*& sNext )

{
    if( ! *sCurrent )
        return false;

    sNext = sCurrent;
    char* sBlank = 0;

    while( *sNext && *sNext != '\n' ) {
        if( sBlank == 0 ) {
            if( isspace( *sNext ) )
                sBlank = sNext;
        }
        else {
            if( ! isspace( *sNext ) )
                sBlank = 0;
        }
        ++sNext;
    }

    if( sBlank != 0 )
        *sBlank = 0;

    if( *sNext == '\n' ) {
        if( sBlank == 0 )
            *sNext = 0;
        ++sNext;
    }

    return true;
}


//
//   g e t T o k e n
//

bool tHttpDetour::getToken( const char*& sLine )

{
    while( isspace( *sLine ) )
        ++sLine;

    if( ! *sLine )
        return false;

    if( *sLine == ':' || *sLine == '=' || *sLine == '.' || *sLine == '/' ) {
        m_sToken[0] = *sLine++;
        m_sToken[1] = 0;
    }
    else {
        size_t k = 0;

        while( isalnum( *sLine ) || *sLine == '_' || *sLine == '-' ) {
            if( k < sizeof(m_sToken) - 1 ) {
                if( isalpha( *sLine ) )
                    m_sToken[k++] = tolower( *sLine );
                else
                    m_sToken[k++] = *sLine;
            }
            ++sLine;
        }

        if( k == 0 )
            return false;

        m_sToken[k] = 0;
    }

    return true;
}


//
//   g e t K e y w o r d
//

bool tHttpDetour::getKeyword( const char*& sLine, const char* sExpect )

{
    if( ! getToken( sLine ) )
        return false;
    if( strcmp( m_sToken, sExpect ) != 0 )
        return false;
    return true;
}


//
//   g e t N u m b e r
//

bool tHttpDetour::getNumber( const char*& sLine, int& nNumber )

{
    if( ! getToken( sLine ) )
        return false;

    size_t k = 0;
    nNumber = 0;

    while( m_sToken[k] ) {
        if( ! isdigit( m_sToken[k] ) )
            return false;
        int c = m_sToken[k] - '0';
        if( nNumber > ( INT_MAX - c ) / 10 )
            return false;
        nNumber = nNumber * 10 + c;
        ++k;
    }

    return true;
}


//
//   p r o c e s s H t t p H e a d e r
//

tRetCode tHttpDetour::processHttpHeader( char* sHeader )

{
    int nTag, nSeq;
    char* sLine = sHeader;

    if( ! getLine( sLine, sHeader ) )
        return DETOUR_BAD_HEADER;

    if( ! getKeyword( sLine, "post" ) || ! getKeyword( sLine, "/" ) ||
        ! getNumber( sLine, nTag ) || ! getKeyword( sLine, "/" ) ||
        ! getNumber( sLine, nSeq ) || ! getKeyword( sLine, "http" ) ||
        ! getKeyword( sLine, "/" ) || ! getKeyword( sLine, "1" ) ||
        ! getKeyword( sLine, "." ) || ! getNumber( sLine, m_nProt ) ) {
        return DETOUR_BAD_HEADER;
    }

    if( m_nTag < 0 )
        m_nTag = nTag;
    else
        if( m_nTag != nTag )
            return DETOUR_BAD_TAG;

    if( nSeq != m_nSeq )
        return DETOUR_BAD_SEQ;
    ++m_nSeq;

    m_nContentLength = -1;

    while( getLine( sLine = sHeader, sHeader ) ) {
        if( getToken( sLine ) ) {
            if( strcmp( m_sToken, "content-length" ) == 0 ) {
                if( m_nContentLength >= 0 )
                    return DETOUR_BAD_HEADER;

                if( ! getKeyword( sLine, ":" ) || 
                    ! getNumber( sLine, m_nContentLength ) ) {
                    return DETOUR_BAD_HEADER;
                }
            }
        }
    }

    if( m_nContentLength < 0 ) 
        return DETOUR_BAD_HEADER;

    return OK;
}


//
//   g e t H t t p H e a d e r
//

tRetCode tHttpDetour::getHttpHeader( tSocket& Sock )

{
    tRetCode nRet;
    char sHeader[1024];
    int nFilled = 0;
    bool bCr = false;
    int nLf = 0;

    m_nProt = 0;

    while( nLf < 2 ) {
        if( nFilled >= (int) sizeof(sHeader) )
            return DETOUR_BAD_HEADER;

        if( ( nRet = Sock.readChar( sHeader[nFilled] ) ) != OK )
            return nRet;

        switch( sHeader[nFilled++] ) {
            case '\r' :
                if( bCr )
                    nLf = 0;
                else
                    bCr = true;
                break;

            case '\n' :
                bCr = false;
                ++nLf;
                break;

            default :
                bCr = false;
                nLf = 0;
                break;
        }
    }

    if( --nFilled > 0 && sHeader[nFilled - 1] == '\r' )
        --nFilled;
    sHeader[nFilled] = 0;

    return processHttpHeader( sHeader );
}


//
//   g e t H t t p B o d y
//

tRetCode tHttpDetour::getHttpBody( tSocket& Sock )

{
    tRetCode nRet;

    if( m_nContentHiwater < m_nContentLength ) {
        char* s = new char[m_nContentLength];
        if( m_sContent != 0 )
            delete m_sContent;
        m_sContent = s;
        m_nContentHiwater = m_nContentLength;
    }

    for( int k = 0; k < m_nContentLength; ++k ) {
        if( ( nRet = Sock.readChar( m_sContent[k] ) ) != OK )
             return nRet;
    }

    return OK;
}


//
//   g e t H t t p R e q u e s t
//

tRetCode tHttpDetour::getHttpRequest( tSocket& Sock )

{
    tRetCode nRet;

    if( ( nRet = getHttpHeader( Sock ) ) != OK ||
        ( nRet = getHttpBody( Sock ) ) != OK ) {
        return nRet;
    }

    return OK;
}


//
//   r e c e i v e
//

tRetCode tHttpDetour::receive( tSocket& NativeSock, bool& bStop )

{
    tRetCode nRet;

    do{
        if( m_nResponseLength + 256 > m_nResponseHiwater ) {
            int nNewHiwater = m_nResponseLength + 256;
            char* s = new char[nNewHiwater];

            if( m_sResponse != 0 ) {
                if( m_nResponseLength > 0 )
                    memcpy( s, m_sResponse, m_nResponseLength );
                delete m_sResponse;
            }
            m_sResponse = s;
            m_nResponseHiwater = nNewHiwater;
        }

        size_t n = (size_t) ( m_nResponseHiwater - m_nResponseLength );
        if( ( nRet = NativeSock.read(
                        m_sResponse + m_nResponseLength, n ) ) != OK ) {
            return nRet;
        }
        m_nResponseLength += (int) n;

    } while( NativeSock.readPending( 0 ) == OK );

    if( ! bStop && strstr( m_sResponse, "\nbye\n" ) != 0 )
        bStop = true;

    return OK;
}


//
//   a c c e p t
//

tRetCode tHttpDetour::accept( tSocket& OutSock, tSocket& NativeSock )

{
    tRetCode nRet;
    char sBuffer[512];

    OutSock.setTimeout( m_nTimeout );

    if( ( nRet = getHttpRequest( OutSock ) ) != OK )
        goto fail;

    if( m_nContentLength > 0 ) {
        if( ( nRet = NativeSock.write( m_sContent, 
                m_nContentLength ) ) != OK ) {
            goto fail;
        }
    }

    return OK;

fail:

        // no matter what caused the error, try to send
        // a reply

    sprintf( sBuffer, 
        "HTTP/1.%d 503 Service not available\n"
        "Content-Type: plain/text\n" 
        "Content-Length: 0\n"
        "Pragma: ret=%d\n\n", m_nProt, nRet );

    OutSock.write( sBuffer );   // ignore errors
    OutSock.close();

    return nRet;
}


//
//   r e s p o n s e
//

tRetCode tHttpDetour::response( tSocket& OutSock, tSocket& HttpSock )

{
    tRetCode nRet;
    char sBuffer[512];

    sprintf( sBuffer, 
        "HTTP/1.%d 200 OK\n"
        "Content-type: plain/text\n" 
        "Content-length: %d\n"
        "Pragma: port=%d\n\n",
        m_nProt, m_nResponseLength, HttpSock.myPort() );

    if( ( nRet = OutSock.write( sBuffer ) ) != OK ) {
        OutSock.close();
        return nRet;
    }

    if( m_nResponseLength > 0 ) {
        if( ( nRet = OutSock.write( m_sResponse,
                        (size_t) m_nResponseLength ) ) != OK ) {
            OutSock.close();
            return nRet;
        }
        m_nResponseLength = 0;
    }

    OutSock.close();    // even if HTTP 1.1
    return OK;
}


//
//   o p e n
//

tRetCode tHttpDetour::open( tSocket& OrgSock, tSocket& HttpSock,
    tSocket& NativeSock, int nNativePort )

{
    tRetCode nRet;
    char sBuffer[512];

    m_nTag = -1;
    m_nSeq = 0;

    if( ( nRet = getHttpRequest( OrgSock ) ) != OK )
        goto fail;

    if( ( nRet = HttpSock.listen( 0 ) ) != OK )
        goto fail;

    if( ( nRet = NativeSock.connect( "127.0.0.1", nNativePort ) ) != OK ) {
        HttpSock.close();
        goto fail;
    }

    if( m_nContentLength > 0 ) {
        if( ( nRet = NativeSock.write( m_sContent, 
                m_nContentLength ) ) != OK ) {
            HttpSock.close();
            NativeSock.close();
            goto fail;
        }
    }

        // OrgSock remains open for reply

    m_nResponseLength = 0;
    return OK;

fail:

    sprintf( sBuffer, 
        "HTTP/1.%d 503 Service not available\n"
        "Content-Type: plain/text\n" 
        "Content-Length: 0\n"
        "Pragma: ret=%d\n\n", m_nProt, nRet );

    OrgSock.write( sBuffer );   // ignore errors
    OrgSock.close();

    return nRet;
}


//
//   t H t t p D e t o u r
//

tHttpDetour::tHttpDetour()

{
    m_sContent = 0;
    m_nContentHiwater = 0;
    m_sResponse = 0;
    m_nResponseHiwater = 0;
}


//
//   ~ t H t t p D e t o u r
//

tHttpDetour::~tHttpDetour()

{
    if( m_sContent != 0 )
        delete m_sContent;
    if( m_sResponse != 0 )
        delete m_sResponse;
}


//
//   r u n
//

tRetCode tHttpDetour::run( tSocket& OrgSock, int nNativePort )

{
    int nWho;
    tRetCode nRet;
    tSocket OutSock, HttpSock, NativeSock;

    if( ( nRet = open( OrgSock, HttpSock, NativeSock, nNativePort ) ) != OK )
        return nRet;

        // OrgSock is still open at this point.

    tSocket* pOutSock = &OrgSock;
    bool bStop = false;

    while( true ) {        
        nRet = tSocket::readPending( NativeSock, HttpSock, nWho, m_nTimeout );

        if( nRet == TIMEOUT ) {
                // send empty response, to avoid timeout at client
            if( pOutSock != 0 ) {
                if( ( nRet = response( *pOutSock, HttpSock ) ) != OK )
                    break;
                pOutSock = 0;
                if( bStop )     // "bye" received?
                    break;
            }
            else {
                    // no request from client.. shut down
                break;
            }
            continue;
        }

        if( nRet != OK )
            break;

        if( nWho & 1 ) {
            if( ( nRet = receive( NativeSock, bStop ) ) != OK )
                break;
            if( pOutSock != 0 ) {
                if( ( nRet = response( *pOutSock, HttpSock ) ) != OK )
                    break;
                pOutSock = 0;
                if( bStop )     // "bye" received?
                    break;
            }
        }

        if( nWho & 2 ) {
            if( pOutSock != 0 ) {
                nRet = DETOUR_PROTOCOL_ERROR;
                break;
            }
            if( ( nRet = HttpSock.singleServerAccept( OutSock ) ) != OK )
                break;
            if( ( nRet = accept( OutSock, NativeSock ) ) != OK )
                break;
            pOutSock = &OutSock;

            if( bStop ) {   // answer immediately?
                nRet = response( *pOutSock, HttpSock );
                pOutSock = 0;
                break;
            }
        }
    }

    if( pOutSock != 0 )
        pOutSock->close();

    NativeSock.close();
    HttpSock.close();

    return nRet;
}

MTG_END_NAMESPACE
