// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#if ! defined(_MTG_INTEREST_)
#define _MTG_INTEREST_

#include "MtgCompounder.h"
#include "MtgDayCount.h"

MTG_BEGIN_NAMESPACE


//
//   t I n t e r e s t
//

class tInterest {

public:

        // Exponential compounding as in
        //
        //      (1 + yield/n)^(na) * (1 + yield/n)^(nb)
        //
        // linear or money market compounding as in
        //
        //      (1 + a yield) * (1 + b yield)
        //
        // for two compounding periods with day count
        // fractions a and b (relative to one year), 
        // and n compounding periods per year.

    enum tType {
        xExponential,        // standard (exponential) compounding
        xLinear              // money market (linear) compounding
    };

    enum tQuote {
        xDiscount,          // annualized discount rate
        xYield              // annualized yield
    };

private:

    tType m_nType;          // Default: standard
    tQuote m_nQuote;        // Default: yield

        // Day count convention. ACT/365 is the default.

    tDayCount m_DayCount;

        // For compounding, an explicit compounder may be provided.
        // The compounder is used to generate compounding periods.
        // The compounder can be passed temporarily for the
        // next interest calculation only, or be registered
        // permanently with the object, by which it is copied
        // and then owned.

        // Note that tInterest objects can handle compounders
        // of type tPeriodCompounder, although these are only
        // partial (they were actually made for this purpose).

    tCompounder* m_pCompounder;

        // Max number of iterations in Newton-Raphson method
        // for value to yield calculations:

    static const int m_nMaxIter;

    void init( tType nType, tQuote nQuote, const tDayCount& DayCount );
    void cleanup();

    void copyFrom( const tInterest& Interest );

        // Find the day count fractions for the period Start to End,
        // under the registered day count policy and with the
        // compounder passed as argument. Return the fractions in
        // Fraction, where nPow0 denotes how often Fraction[0]
        // is repeated, and nPeriods holds the number of compounding
        // periods per year (0 for continuous compounding.

    void findFractions( tDate Start, tDate End, 
        const tCompounder* pCompounder, tHeap<double>& Fraction,
        int& nPow0, int& nPeriods ) const;

        // Do the same thing (as good as possible) if only the
        // day count fraction is known.

    void findFractions( double gOverallFraction,
        const tCompounder* pCompounder, tHeap<double>& Fraction,
        int& nPow0, int& nPeriods ) const;

        // This function is courtesy of tInterestSpline.

    void findFractions( tDate Start, tDate End, 
        tHeap<double>& Fraction, int& nPow0 ) const;

        // Future value helper functions:

    double accrueExponential( tHeap<double>& Fraction, int nPow0,
        double gYield, int nPeriods, double* pGradient = 0 ) const;
    double accrueLinear( tHeap<double>& Fraction, int nPow0,
        double gYield, int nPeriods, double* pGradient = 0 ) const;
    double accrue( tHeap<double>& Fraction, int nPow0, double gYield,
        int nPeriods, double* pGradient = 0 ) const;

        // Newton Raphson for value to yield calculations
        // (note that if Quote is xDiscount, it'll find the
        // discount, not the yield):

    bool inverseExponential( tHeap<double>& Fraction, int nPow0,
        double gValue, int nPeriods, double& gYield ) const;
    bool inverseLinear( tHeap<double>& Fraction, int nPow0,
        double gValue, int nPeriods, double& gYield ) const;
    bool inverse( tHeap<double>& Fraction, int nPow0,
        double gValue, int nPeriods, double& gYield ) const;

        // tInterestSpline is supposed to use only findFraction().

    friend class tInterestSpline;

public:

    tInterest();
    tInterest( tType nType, tQuote nQuote, const tDayCount& DayCount );        
    tInterest( tType nType, tQuote nQuote, const tDayCount& DayCount,
        const tCompounder& Compounder );

    tInterest( const tInterest& Interest );

    ~tInterest();

    tInterest& operator=( const tInterest& Interest );

    void set( tType nType, tQuote nQuote, const tDayCount& DayCount );
    void set( tType nType, tQuote nQuote, const tDayCount& DayCount,
        const tCompounder& Compounder );

        // Change only some components:

    void set( tType nType );
    void set( tQuote nQuote );
    void set( const tDayCount& DayCount );
    void set( const tCompounder& Compounder );

        // Retrieve components:

    const tDayCount& dayCount() const {
        return m_DayCount;
    }

    const tCompounder* compounder() const {
        return m_pCompounder;
    }

    bool hasCompoundingDates() const;
    bool isContinuousCompounding() const;

        // The following functions all have 3 versions:
        //
        //  - One in which no compounder is provided; in this case,
        //    m_pCompounder is substituted, or, if that is 0 as well,
        //    continuous compounding is assumed.
        //  - One in which a compounder is provided (or the 0 pointer
        //    to indicate continuous compounding). In this case,
        //    m_pCompounder will be ignored and replaced with the
        //    temporary compounder for the next calculation only.
        //  - One in which only the number of compounding periods
        //    per year is supplied. If this number is 0, continuous
        //    compounding is assumed. Compounding doesn't necessarily
        //    occur on integer days, but rather at equidistant intervals
        //    with distance 1/<number of periods> per year according
        //    to the day count policy, beginning with the start date.
        //
        // Note that within compounding periods, all calculations
        // are either standard or Moosmueller type, never mixed.

        // -----------------------

        // Compute the present value of $1, invested at start date
        // and redeemed at value date, where the yield is gYield,
        // under the accural policy manifest in this object:

    double presentValue( tDate Start, tDate End, double gYield ) const;
    double presentValue( tDate Start, tDate End, double gYield,
        const tCompounder* pCompounder ) const;
    double presentValue( tDate Start, tDate End, double gYield,
        int nNumOfPeriodsPerYear ) const;

        // If, instead of start and end date, only the day
        // count fraction is known, the following functions
        // may be used.

    double presentValue( double gOverallFraction, double gYield ) const;
    double presentValue( double gOverallFraction, double gYield,
        const tCompounder* pCompounder ) const;
    double presentValue( double gOverallFraction, double gYield,
        int nNumOfPeriodsPerYear ) const;

        // The same function; returns, in addition, the sensitivity
        // to the yield parameter.

    double presentValue( tDate Start, tDate End, double gYield,
        double& gGradient ) const;
    double presentValue( tDate Start, tDate End, double gYield,
        double& gGradient, const tCompounder* pCompounder ) const;
    double presentValue( tDate Start, tDate End, double gYield,
        double& gGradient, int nNumOfPeriodsPerYear ) const;

        // If, instead of start and end date, only the day
        // count fraction is known, the following functions
        // may be used.

    double presentValue( double gOverallFraction, double gYield,
        double& gGradient ) const;
    double presentValue( double gOverallFraction, double gYield,
        double& gGradient, const tCompounder* pCompounder ) const;
    double presentValue( double gOverallFraction, double gYield,
        double& gGradient, int nNumOfPeriodsPerYear ) const;

        // The inverse of the previous function: assuming the present
        // value at the start date of $1 at the value date is gValue,
        // what's the yield or discount rate under the policy of the
        // object?

    double invPresentValue( tDate Start, tDate End, double gValue ) const;
    double invPresentValue( tDate Start, tDate End, double gValue,
        const tCompounder* pCompounder ) const;
    double invPresentValue( tDate Start, tDate End, double gValue,
        int nNumOfPeriodsPerYear ) const;

        // If, instead of start and end date, only the day
        // count fraction is known, the following functions
        // may be used.

    double invPresentValue( double gOverallFraction, double gValue ) const;
    double invPresentValue( double gOverallFraction, double gValue,
        const tCompounder* pCompounder ) const;
    double invPresentValue( double gOverallFraction, double gValue,
        int nNumOfPeriodsPerYear ) const;

        // Compute the future value of $1, invested at start date
        // and redeemed at value date, where the yield is gYield,
        // under the accural policy manifest in this object:

    double futureValue( tDate Start, tDate End, double gYield ) const;
    double futureValue( tDate Start, tDate End, double gYield,
        const tCompounder* pCompounder ) const;
    double futureValue( tDate Start, tDate End, double gYield,
        int nNumOfPeriodsPerYear ) const;

        // If, instead of start and end date, only the day
        // count fraction is known, the following functions
        // may be used.

    double futureValue( double gOverallFraction, double gYield ) const;
    double futureValue( double gOverallFraction, double gYield,
        const tCompounder* pCompounder ) const;
    double futureValue( double gOverallFraction, double gYield,
        int nNumOfPeriodsPerYear ) const;

        // The same function; returns, in addition, the sensitivity
        // to the yield parameter.

    double futureValue( tDate Start, tDate End, double gYield,
        double& gGradient ) const;
    double futureValue( tDate Start, tDate End, double gYield,
        double& gGradient, const tCompounder* pCompounder ) const;
    double futureValue( tDate Start, tDate End, double gYield,
        double& gGradient, int nNumOfPeriodsPerYear ) const;

        // If, instead of start and end date, only the day
        // count fraction is known, the following functions
        // may be used.

    double futureValue( double gOverallFraction, double gYield,
        double& gGradient ) const;
    double futureValue( double gOverallFraction, double gYield,
        double& gGradient, const tCompounder* pCompounder ) const;
    double futureValue( double gOverallFraction, double gYield,
        double& gGradient, int nNumOfPeriodsPerYear ) const;

        // The inverse of the previous function: assuming the future
        // value at the value date of $1 invested at the start date
        // is gValue, what's the yield or discount rate under the
        // policy of the object?

    double invFutureValue( tDate Start, tDate End, double gValue ) const;
    double invFutureValue( tDate Start, tDate End, double gValue,
        const tCompounder* pCompounder ) const;
    double invFutureValue( tDate Start, tDate End, double gValue,
        int nNumOfPeriodsPerYear ) const;

        // If, instead of start and end date, only the day
        // count fraction is known, the following functions
        // may be used.

    double invFutureValue( double gOverallFraction, double gValue ) const;
    double invFutureValue( double gOverallFraction, double gValue,
        const tCompounder* pCompounder ) const;
    double invFutureValue( double gOverallFraction, double gValue,
        int nNumOfPeriodsPerYear ) const;
};

MTG_END_NAMESPACE

#endif
