// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#include "MtgIncl.h"
#include "MtgInterestConvention.h"
#include "MtgParser.h"

MTG_BEGIN_NAMESPACE


//
//   c o p y F r o m
//

void tInterestConvention::copyFrom( const tInterestConvention& Convention )

{
    if( &Convention == this )
        return;

    m_Interest = Convention.m_Interest;
    super::copyFrom( Convention );
}


//
//   p a r s e P a r a m
//

tRetCode tInterestConvention::parseParam( tParser& Parser,
    tParseInfoStub& Info )

{
    tRetCode nRet;
    tDayCount DayCount;

    tParseInfo& I = static_cast<tParseInfo&>( Info );

    nRet = DayCount.parse( Parser );

    if( nRet == OK ) {
        if( I.m_bDayCount ) 
            return Parser.setError( ATTR_REDEFINITION );
        m_Interest.set( DayCount );
        I.m_bDayCount = true;
        return nRet;
    }

    if( nRet != GO )
        return nRet;

    switch( Parser.curToken() ) {
        case xTokQuote :
            if( ( nRet = Parser.readToken() ) != OK )
                return nRet;
            // fall through

        case xTokYield :
        case xTokDiscount :
            if( I.m_bQuote )
                return Parser.setError( ATTR_REDEFINITION );

            switch( Parser.curToken() ) {
                case xTokYield :
                    m_Interest.set( tInterest::xYield );
                    break;

                case xTokDiscount :
                    m_Interest.set( tInterest::xDiscount );
                    break;

                default :
                    return Parser.setError( INVALID_KEYWORD );
            }

            if( ( nRet = Parser.readToken() ) != OK )
                return nRet;
            I.m_bQuote = true;
            break;

        case xTokSimpleCompounding :
            if( I.m_bType )
                return Parser.setError( ATTR_REDEFINITION );
            m_Interest.set( tInterest::xLinear );
            if( ( nRet = Parser.readToken() ) != OK )
                return nRet;
            if( Parser.beginOfNumber() ) {
                int nPeriods;

                if( ( nRet = Parser.scanInteger( nPeriods, 0 ) ) != OK )
                    return nRet;
                m_Interest.set( tPeriodCompounder( nPeriods ) );
            }
            else {
                m_Interest.set( tPeriodCompounder( 1 ) );
            }
            I.m_bType = true;
            break;

        case xTokContinuousCompounding :
            if( I.m_bType )
                return Parser.setError( ATTR_REDEFINITION );
            m_Interest.set( tInterest::xExponential );
            if( ( nRet = Parser.readToken() ) != OK )
                return nRet;
            I.m_bType = true;
            break;

        default :
            if( ( nRet = super::parseParam( Parser, Info ) ) != OK )
                return nRet;
    }

    return OK;
}


//
//   p a r s e P o s t f i x
//

tRetCode tInterestConvention::parsePostfix( tParser& Parser,
    tParseInfoStub& Info )

{
    tParseInfo& I = static_cast<tParseInfo&>( Info );

    if( ! I.m_bDayCount )
        m_Interest.set( system().dayCount() );
    if( ! I.m_bQuote )
        m_Interest.set( tInterest::xYield );
    if( ! I.m_bType )
        m_Interest.set( tInterest::xExponential );

    return super::parsePostfix( Parser, Info );
}


//
//   t I n t e r e s t C o n v e n t i o n
//

tInterestConvention::tInterestConvention()

{
}


//
//   t I n t e r e s t C o n v e n t i o n
//

tInterestConvention::tInterestConvention(
    const tInterestConvention& Convention )

{
    copyFrom( Convention );
}


//
//   ~ t I n t e r e s t C o n v e n t i o n
//

tInterestConvention::~tInterestConvention()

{
}


//
//   o p e r a t o r =
//

tInterestConvention& tInterestConvention::operator=(
    const tInterestConvention& Convention )

{
    if( &Convention != this )
        copyFrom( Convention );
    return *this;
}


//
//   c l o n e
//

tObject* tInterestConvention::clone() const

{
    return new tInterestConvention( *this );
}


//
//   f i n a l i z e
//

tRetCode tInterestConvention::finalize()

{
    return super::finalize();
}


//
//   p a r s e
//

tRetCode tInterestConvention::parse( tParser& Parser, tSystem& System,
    tObject*& pObj )

{
    tRetCode nRet;
    tInterestConvention* pConvention;
    tParseInfo Info;

    Info.m_bDayCount = false;
    Info.m_bQuote = false;
    Info.m_bType = false;

    if( ( nRet = Parser.scanBeginOfObj() ) != OK )
        return nRet;

    pConvention = new tInterestConvention;
    pConvention->setSystem( System );

    if( ( nRet = pConvention->super::parse( Parser, &Info ) ) != OK ) {
        delete pConvention;
        return nRet;
    }

    pObj = pConvention;
    return OK;
}

MTG_END_NAMESPACE