// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#include "MtgIncl.h"
#include "MtgInterestSpline.h"
#include "MtgUSBondMath.h"
#include "MtgHeap2.h"

MTG_BEGIN_NAMESPACE


//
//   v a r i a b l e s
//

const int tInterestSpline::m_nMaxIter = 50;


//
//   i n i t
//

void tInterestSpline::init( tType nType )

{
    m_nType = nType;
    m_bFinalized = false;
}


//
//   c l e a n u p
//

void tInterestSpline::cleanup()

{
    for( int k = 0; k < m_Leg.numOfElems(); ++k ) {
        if( m_Leg[k].m_nType == xPaymentLeg )
            delete m_Leg[k].m_pPayment;
    }
    m_Leg.reset();
    m_bFinalized = false;
}


//
//   c o p y F r o m
//

void tInterestSpline::copyFrom( const tInterestSpline& Spline )

{
    if( &Spline == this )
        return;

    cleanup();

    m_nType = Spline.m_nType;
    m_Interest = Spline.m_Interest;  
    m_RefDate = Spline.m_RefDate;
    m_Index = Spline.m_Index;
    m_bFinalized = Spline.m_bFinalized;

    m_Leg.numOfElems( Spline.m_Leg.numOfElems() );

    for( int k = 0; k < Spline.m_Leg.numOfElems(); ++k ) {
        m_Leg[k] = Spline.m_Leg[k];
        if( m_Leg[k].m_nType == xPaymentLeg ) {
                // duplicate payment stream
            m_Leg[k].m_pPayment =
                new tHeap<tPayment>( *m_Leg[k].m_pPayment );
        }
    }
}


//
//   a d d
//

void tInterestSpline::add( const tLeg& Leg )

{
    int k = m_Leg.numOfElems();

        // find insertion point
    while( k - 1 >= 0 && m_Leg[k - 1].m_nEndDay >= Leg.m_nEndDay )
        --k;

    if( k < m_Leg.numOfElems() ) {
        if( m_Leg[k].m_nEndDay != Leg.m_nEndDay ) {
            ++m_Leg;    // shift and insert
            for( int j = m_Leg.numOfElems() - 1; j > k; --j )
                m_Leg[j] = m_Leg[j - 1];
        }
        else {
                        // replace
            if( m_Leg[k].m_nType == xPaymentLeg )
                delete m_Leg[k].m_pPayment;
        }
        m_Leg[k] = Leg;
    }
    else {
        m_Leg.append( Leg );
    }

        // need to redo everything:
    m_bFinalized = false;
}


//
//   r e v e r s e P a y m e n t
//

double tInterestSpline::reversePayment( const tHeap<tPayment>& Payment,
    double gPV, tDate Start, int nPos ) const 

{
        // This is how it works: every vertical bar represents
        // an element in Payment. All payments beginning at
        // position nPos are rolled back to date Start, and
        // the yield is computed so that the discounted values
        // combined are equal to -gPV.
        //
        //                             nPos       Payment.last()
        //              |   |   |   |   |   |   |   |  
        //              |   |   |   |   |   |   |   |
        //          +---+---+---+---+-+-+---+---+---+
        //          |                 :
        //          |                 :
        //   Payment[0]          gPV at Start
        //                            :<------------>
        //                                 reverse
        //
        // The discounting is done with m_Interest, since
        // there is no break date between Start and the date of
        // the last payment (that's when reversePayment() gets
        // called).

    MTG_ASSERT( Payment.numOfElems() > nPos );

    if( Payment.numOfElems() == nPos - 1 ) {
            // only one payment left, do it directly
        return m_Interest.invFutureValue( Start,
            Payment.last().m_Date, Payment.last().m_gAmount / -gPV );
    }

        // a yield of 0 is the initial guess for Newton-Raphson

    double gYield = 0;
    int nIter = 0;

    while( true ) {
        double gGradient = 0;
        double gValue = 0;

            // discount all future payments

        for( int k = nPos; k < Payment.numOfElems(); ++k ) {
            double g;
            gValue += Payment[k].m_gAmount *
                        m_Interest.presentValue( Start, 
                            Payment[k].m_Date, gYield, g );
            gGradient += Payment[k].m_gAmount * g;
        }

        double y = gYield - ( gValue + gPV ) / gGradient;
        double e;

        if( fabs( gYield ) < 1e-30 )
            e = fabs( y - gYield );
        else
            e = fabs( y / gYield - 1 );

        if( e < 1e-10 ) {
            gYield = y;
            break;
        }
        
        if( ++nIter >= m_nMaxIter )
            throw tException( NO_CONVERGENCE );

        gYield = y;
    }

    return gYield;
}


//
//   d i s c o u n t
//

double tInterestSpline::discount( const tHeap<tPayment>& Payment )

{
    double gValue = 0;

        // Discount all elements in Payment, starting with the
        // element at position 1, back to the start of the
        // payment structure. Do full splining; there might be
        // break dates between the start date and the date of
        // the last payment. We assume the relevant legs have
        // all been initialized.

    for( int k = 1; k < Payment.numOfElems(); ++k ) {
        gValue += Payment[k].m_gAmount *
            presentValue( Payment[0].m_Date, Payment[k].m_Date );
    }

    return gValue;
}


//
//   r e f i n e L e g
//

void tInterestSpline::refineLeg( const tHeap<tPayment>& Payment, int nLeg )

{
        // The situation is similar to the one for reversePayment():
        //
        //                                        Payment.last()
        //              |   |   |   |   |   |   |   |  
        //              |   |   |   |   | t |   |   |
        //          +---+---+---+---+---+-+-+---+---+
        //          |                     :<-- nLeg starts here
        //          |<------------------------------>                 
        //   Payment[0]          reverse
        //          
        // This time, however, the yield for the leg nLeg is
        // computed by looking at the entire payment structure
        // (for reversePayment(), everything before t had been
        // propagated to t). This is necessary if the type of
        // the spline is xAggregate, for then the break date
        // at t must not become a compounding date.

        // In this case, we don't have a gradient, so we need
        // to use bisection.

    tLeg& L = m_Leg[nLeg];

    double gPV = -Payment[0].m_gAmount;
    double gTrial = 0.1;

        // first find initial bracket

    while( true ) {
        L.m_gFwdYield = 0;      // initial guess 1
        double gValue1 = discount( Payment ) - gPV;

        L.m_gFwdYield = gTrial; // initial guess 2
        double gValue2 = discount( Payment ) - gPV;

            // extrapolate:
        double gZero = ( gTrial * gValue1 ) / ( gValue1 - gValue2 );

        if( gTrial > 0 ) {
            if( gZero < 0 ) {
                gTrial = -gTrial;
                if( gTrial > gZero )
                    gTrial = gZero;
            }
            else
            if( gZero > gTrial ) {
                gTrial = gZero;
            }
            else {
                break;  // found it
            }
        }
        else {
                // this is pathetic, but for completeness' sake,
                // we also handle negative yields
            if( gZero > 0 ) {
                gTrial = -gTrial;
                if( gTrial < gZero )
                    gTrial = gZero;
            }
            else
            if( gZero < gTrial ) {
                gTrial = gZero;
            }
            else {
                break;  // found it
            }
        }

        gTrial *= 1.5;
    }

        // initial bracket is now between gTrial and 0

    double gLeft, gRight;   // brackets

    if( gTrial > 0 ) {
        gLeft = 0;
        gRight = gTrial;
    }
    else {
        gLeft = gTrial;
        gRight = 0;
    }

    while( true ) {
            // next probe
        L.m_gFwdYield = ( gLeft + gRight ) / 2;
        double gValue = discount( Payment );

        if( gValue == gPV )
            return;

        if( gValue > gPV ) {
            // need to discount more
            gLeft = L.m_gFwdYield;
        }
        else {
            // need to discount less
            gRight = L.m_gFwdYield;
        }

        double e = gRight - gLeft;

        if( fabs( gLeft ) >= 1e-20 )
            e /= gLeft;

        if( e < 1e-10 )
            break;
    }
}


//
//   f i n a l i z e
//

void tInterestSpline::finalize()

{
    if( m_bFinalized )
        return;

        // there could be recursive calls
    m_bFinalized = true;
    m_Index.reset();

    if( m_Leg.numOfElems() == 0 )
        return;

        // This picture shows reference date and m_Index:
        //
        //         Reference date (date 0)
        //          |
        //    Leg 0 | Leg 1   Leg 2            Leg n
        //  +-------+-------+-------+---  ---+--------+
        //                 :       :        :        :
        //          11111111222222223333  mmmnnnnnnnnn  (m = n-1)
        //
        // Leg 0 and leg n extent logically to -infinity and infinity.
        // m_Index has entries from (including) 0 to the end day
        // of the last leg - 1. The digits show which value is returned
        // by m_Index for which day. m_Index indicates which leg should
        // be used to do interest calculations for the day.

    int nShift = m_Leg[0].m_nEndDay;
    m_RefDate += nShift;

    for( int k = 0; k < m_Leg.numOfElems(); ++k ) {
        tLeg& L = m_Leg[k];

        L.m_nStartDay -= nShift;
        L.m_nEndDay -= nShift;

            // At finalization, the leader of each leg is the
            // leg itself. If merge() is called this may change:
            // if the right neighbor of leg L has the same
            // yield, then the leader of L is set to the leader
            // of that right neighbor.

        L.m_nLeader = k;

        if( k == 0 ) {
            switch( L.m_nType ) {
                case xForwardLeg :
                    L.m_gImpYield = L.m_gFwdYield;
                    break;

                case xImpliedLeg :
                    L.m_gFwdYield = L.m_gImpYield;
                    break;

                case xPaymentLeg :
                    L.m_gFwdYield =
                        reversePayment( *L.m_pPayment,
                            (*L.m_pPayment)[0].m_gAmount, 
                            (*L.m_pPayment)[0].m_Date, 1 );
                    break;

                default :
                    throw tException( INTERNAL_ERROR );
            }
        }
        else {
                // m_Index is used to find the start date of
                // the interest period. On days that join two
                // legs, it therefore needs to lead to the later
                // leg.

            m_Index.append( L.m_nEndDay - 1, k );

            tDate D1 = m_RefDate + L.m_nStartDay;
            tDate D2 = m_RefDate + m_Leg[k - 1].m_nEndDay;
            tDate D3 = m_RefDate + L.m_nEndDay;

                // Always D1 < D3 and D2 < D3,
                // but D2 may be before or after D1.

            switch( L.m_nType ) {
                case xForwardLeg :
                        // nothing to do
                    break;

                case xImpliedLeg :
                    if( D2 <= D1 ) {
                        L.m_gFwdYield = L.m_gImpYield;
                    }
                    else {
                        double gPV =
                            m_Interest.presentValue( D1, D3, L.m_gImpYield );

                        if( m_nType == xExtraCompounding ) {
                            // We know what the implied rate between
                            // D1 and D3 is supposed to be. We can
                            // also compute how much $1 appreciates
                            // between D1 and D2, as the forward rates
                            // have already been fixed for that interval.

                            double f = futureValue( D1, D2 );
                            L.m_gFwdYield = 
                                m_Interest.invPresentValue( D2, D3, f * gPV );
                        }
                        else {
                            MTG_ASSERT( m_nType == xAggregate );

                            tHeap<tPayment> Payment;
                            Payment.append( tPayment( D1, -gPV ) );
                            Payment.append( tPayment( D3, 1 ) );
                            refineLeg( Payment, k );
                        }
                    }
                    break;

                case xPaymentLeg :
                    if( m_nType == xExtraCompounding ) {
                        // Propagate payment to last date for
                        // which forward rate has been computed;
                        // this saves some time.

                        tHeap<tPayment>& Payment = *L.m_pPayment;
                        double f = Payment[0].m_gAmount;
                        int j = 1;

                        while( Payment[j].m_Date <= D2 ) {
                            f += Payment[j].m_gAmount *
                                presentValue( D1, Payment[j].m_Date );
                            ++j;
                        }

                        // From here on, it's new, and we have to
                        // find the new forward yield.

                        f *= futureValue( D1, D2 );
                        L.m_gFwdYield =
                            reversePayment( Payment, f, D2, j );
                    }
                    else {
                        // We can't save time here as easily, since
                        // we have to aggregate across break dates.

                        MTG_ASSERT( m_nType == xAggregate );
                        refineLeg( *L.m_pPayment, k );
                    }
                    break;

                default :
                    throw tException( INTERNAL_ERROR );
            }
        }
    }
}


//
//   a c c r u e E x t r a C o m p o u n d i n g
//

double tInterestSpline::accrueExtraCompounding( tDate Start, tDate End,
    tDate tPeriodStart, tDate PeriodEnd ) const

{
    long nStartDay = Start - m_RefDate;
    long nEndDay = End - m_RefDate;

    int nLeg;

        // extend first and last leg to infinity

    if( nStartDay < 0 )
        nLeg = m_Leg[0].m_nLeader;
    else
    if( nStartDay >= m_Index.numOfElems() )
        nLeg = m_Leg.numOfElems() - 1;
    else
        nLeg = m_Index[(int) nStartDay];

    double gValue = 1;

        // spline piecewise

    while( nStartDay <= nEndDay ) {
        tLeg& L = m_Leg[nLeg];

        tDate D1 = m_RefDate + nStartDay;
        tDate D2;

        if( nEndDay > L.m_nEndDay && nLeg + 1 < m_Leg.numOfElems() ) {
            D2 = m_RefDate + L.m_nEndDay;
            nStartDay = L.m_nEndDay;
            nLeg = m_Leg[nLeg + 1].m_nLeader;
        }
        else {
            D2 = m_RefDate + nEndDay;
            nStartDay = nEndDay + 1;
        }

        gValue *= m_Interest.futureValue( D1, D2, L.m_gFwdYield );
    }

    return gValue;
}


//
//   a c c r u e A g g r e g a t e
//

double tInterestSpline::accrueAggregate( tDate Start, tDate End,
    tDate PeriodStart, tDate PeriodEnd ) const

{
    if( m_Interest.isContinuousCompounding() ) {
            // under continuous compounding, both methods
            // are the same; we arbitrarily use the other method
            // in this case
        return accrueExtraCompounding( Start, End,
            PeriodStart, PeriodEnd );
    }

        // This picture shows what happends if the type of
        // the spline is xAggregate:
        //
        //           
        //    Leg A         Leg B         Leg C  
        //  +----*--------+--------*----+----------*--+
        //       |<===============>|<------------->|
        //   Start              Compounding        End
        //    Date              Date               Date
        //
        // We average the yield of leg A and leg B over the
        // period between the start date and the compounding
        // date, and we average the yield of leg B and leg C
        // over the period between the compounding date and
        // the end date. There is no compounding at the break
        // dates between A/B and B/C!
        //
        // This function is called to do the averaging and
        // the accrual over ONE of the pieces (for instance,
        // the one marked <=====>). The parameters Start and
        // End mark the start and end dates of that piece, and
        // must lie within a single compounding period.
        // (Note, however, that if there are no explicit
        // compounding dates, this one piece may actually span
        // the entire investment period.)

    tHeap2<double> Subperiod( 2 );

        // now walk through all subperiods and average
        // over all forward yields

    long nStartDay = Start - m_RefDate;
    long nEndDay = End - m_RefDate;

    int nLeg;

        // extend first and last leg to infinity

    if( nStartDay < 0 )
        nLeg = m_Leg[0].m_nLeader;
    else
    if( nStartDay >= m_Index.numOfElems() )
        nLeg = m_Leg.numOfElems() - 1;
    else
        nLeg = m_Index[(int) nStartDay];

    bool bHasCompDates = m_Interest.hasCompoundingDates();

    double gYield = 0;
    double gFraction, gFractionFactor;

    if( bHasCompDates ) {
        MTG_ASSERT( Start >= PeriodStart && End <= PeriodEnd );

            // In this case, we're exact and follow the rules,
            // just in case the day count convention is ACT/ACT.

        double gFraction = m_Interest.dayCount().fraction( Start, End,
            PeriodStart, PeriodEnd );

        if( ! m_Interest.compounder()->getYearFactor( m_Interest.dayCount(),
                gFractionFactor ) ) {
            throw tException( INTERNAL_ERROR );
        }

        gFraction *= gFractionFactor;
    }
    else {
        gFraction = m_Interest.dayCount().fraction( Start, End );
        gFractionFactor = 1;
    }

    if( gFraction == 0 )
        return 1;

        // spline piecewise

    while( nStartDay <= nEndDay ) {
        tLeg& L = m_Leg[nLeg];

        tDate D1 = m_RefDate + nStartDay;
        tDate D2;

        if( nEndDay > L.m_nEndDay && nLeg + 1 < m_Leg.numOfElems() ) {
            D2 = m_RefDate + L.m_nEndDay;
            nStartDay = L.m_nEndDay;
            nLeg = m_Leg[nLeg + 1].m_nLeader;
        }
        else {
            D2 = m_RefDate + nEndDay;
            nStartDay = nEndDay + 1;
        }

            // we don't do the accrual here, just get the
            // day count fraction

        double f;
        
        if( bHasCompDates ) {
            f = m_Interest.dayCount().fraction( D1, D2,
                    PeriodStart, PeriodEnd );
            f *= gFractionFactor;
        }
        else {
            f = m_Interest.dayCount().fraction( D1, D2 );

                // if there are no explicit compounding dates,  
                // we need to record all the day count fractions
                // for later use

            int k = Subperiod.numOfRows();

            ++Subperiod;
            Subperiod[k][0] = f;
            Subperiod[k][1] = L.m_gFwdYield;
        }

        gYield += f * L.m_gFwdYield;
    }

    gYield /= gFraction;

    if( bHasCompDates ) {
        // use the average yield, and we are done:
        return m_Interest.futureValue( Start, End, gYield );
    }

        // in this case, we match the fractions provided by
        // the heuristic used by tInterest::findFractions()
        // against the fractions of the subperiods

    tHeap<double> Fraction;
    int nPow0;

    m_Interest.findFractions( Start, End, Fraction, nPow0 );

        // normalize the fractions in Subperiod:
    double gNorm1 = Fraction[0] * nPow0;
    for( int i = 1; i < Fraction.numOfElems(); ++i )
        gNorm1 += Fraction[i];

        // we should have gNorm1 == gFraction, but just to protect
        // against changes in tInterest::findFractions(), we computed
        // gNorm1 explicitely

    double gNorm2 = 0;
    for( MTG_FOR_INIT( int ) i = 0; i < Subperiod.numOfRows(); ++i )
        gNorm2 += Subperiod[i][0];

    gNorm1 /= gNorm2;
    for( MTG_FOR_INIT( int ) i = 0; i < Subperiod.numOfRows(); ++i )
        Subperiod[i][0] *= gNorm1;

        // We now have two sequences of day count fractions of
        // equal length: Fraction for the compounding periods,
        // and Subperiod for the periods with equal yield:
        //
        //   +----------+----------+----------+-----+ Fraction
        //   *======*==========*===*========*=======* Subperiod
        //
        // We loop through the pieces of Fraction and gather
        // the pieces or partial pieces of Subperiod as we go
        // along.

    int n = Fraction.numOfElems() + nPow0 - 1;
    int k = 0;  // current Fraction element
    int l = 0;  // current Subperiod element
        
    double gValue = 1;

    for( MTG_FOR_INIT( int ) i = 0; i < n; ++i ) {
            // this much we have to "spend":
        double f = Fraction[k];
        double y = 0;

        if( f > 0 ) {
                // compute average yield:
            while( f > 0 && l < Subperiod.numOfRows() ) {
                if( Subperiod[l][0] <= f ) {
                    y += Subperiod[l][0] * Subperiod[l][1];
                    f -= Subperiod[l][0];
                    ++l;
                }
                else {
                    y += f * Subperiod[l][1];
                    Subperiod[l][0] -= f;
                    f = 0;
                }
            }

            gValue *=
                m_Interest.futureValue( Fraction[k], y / Fraction[k] );
        }

            // Fraction[0] is used nPow0 times
        if( i + 1 >= nPow0 )
            ++k;
    }

    return gValue;
}


//
//   a c c r u e
//

double tInterestSpline::accrue( tDate Start, tDate End,
    tDate PeriodStart, tDate PeriodEnd ) const

{
    double v;

    switch( m_nType ) {
        case xExtraCompounding :
            v = accrueExtraCompounding( Start, End, PeriodStart, PeriodEnd );
            break;

        case xAggregate :
            v = accrueAggregate( Start, End, PeriodStart, PeriodEnd );
            break;

        default :
            throw tException( INTERNAL_ERROR );
    }

    return v;
}


//
//   t I n t e r e s t S p l i n e
//

tInterestSpline::tInterestSpline()

{
    init( xExtraCompounding );
}


//
//   t I n t e r e s t S p l i n e
//

tInterestSpline::tInterestSpline( tType nType )

{
    init( nType );
}


//
//   t I n t e r e s t S p l i n e
//

tInterestSpline::tInterestSpline( const tInterest& Interest )

{
    init( xExtraCompounding );
    m_Interest = Interest;
}


//
//   t I n t e r e s t S p l i n e
//

tInterestSpline::tInterestSpline( tType nType, const tInterest& Interest )

{
    init( nType );
    m_Interest = Interest;
}


//
//   t I n t e r e s t S p l i n e
//

tInterestSpline::tInterestSpline( const tInterestSpline& Spline )

{
    init( xExtraCompounding );
    copyFrom( Spline );
}


//
//   ~ t I n t e r e s t S p l i n e
//

tInterestSpline::~tInterestSpline()

{
    cleanup();
}


//
//   o p e r a t o r =
//

tInterestSpline& tInterestSpline::operator=( const tInterestSpline& Spline )

{
    if( &Spline != this )
        copyFrom( Spline );
    return *this;
}


//
//   s e t
//

void tInterestSpline::set( tType nType )

{
    m_nType = nType;
    m_bFinalized = false;
}


//
//   s e t
//

void tInterestSpline::set( const tInterest& Interest )

{
    m_Interest = Interest;
    m_bFinalized = false;
}


//
//   a d d I m p l i e d
//

void tInterestSpline::addImplied( tDate Start, tDate End, double gYield )

{
    MTG_ASSERT( Start < End );

    tLeg Leg;

    Leg.m_nStartDay = Start - m_RefDate;
    Leg.m_nEndDay = End - m_RefDate;
    Leg.m_nType = xImpliedLeg;
    Leg.m_gImpYield = gYield;

    add( Leg );
}


//
//   a d d F o r w a r d
//

void tInterestSpline::addForward( tDate Start, tDate End, double gYield )

{
    MTG_ASSERT( Start < End );

    tLeg Leg;

    Leg.m_nStartDay = Start - m_RefDate;
    Leg.m_nEndDay = End - m_RefDate;
    Leg.m_nType = xForwardLeg;
    Leg.m_gFwdYield = gYield;

    add( Leg );
}


//
//   a d d P a y m e n t
//

void tInterestSpline::addPayment( tDate Start, double gPV,
    tDate End, double gFV )

{
    addPayment( Start, gPV, tPayment( End, gFV ) ); 
}


//
//   a d d P a y m e n t
//

void tInterestSpline::addPayment( tDate Start, double gPV, tPayment Payment )

{
    tHeap<tPayment> P;

    P.append( Payment );
    addPayment( Start, gPV, P );
}


//
//   a d d P a y m e n t
//

void tInterestSpline::addPayment( tDate Start, double gPV,
    const tHeap<tPayment>& Payment )    

{
    if( Payment.numOfElems() == 0 )
        return;

    tHeap<tPayment>* p = new tHeap<tPayment>;

        // the present value is stored as negative first payment:

    p->append( tPayment( Start, -gPV ) );

        // create the new payment structure from the old one,
        // basically just sorting and consolidating:

    for( int k = 0; k < Payment.numOfElems(); ++k ) {
        const tPayment& P = Payment[k];

        if( P.m_gAmount != 0 && P.m_Date >= Start ) {
            int j = p->numOfElems();

            while( j > 0 && (*p)[j - 1].m_Date >= P.m_Date )
                --j;

            if( j < p->numOfElems() ) { 
                if( (*p)[j].m_Date != P.m_Date ) {
                    ++(*p);
                    for( int i = p->numOfElems() - 1; i > j; --i )
                        (*p)[i] = (*p)[i - 1];
                    (*p)[j] = P;
                }
                else {
                    (*p)[j].m_gAmount += P.m_gAmount;
                }
            }
            else {
                p->append( P );
            }
        }
    }

    if( p->numOfElems() == 1 ) {
        delete p;
        return;
    }
    
    tLeg Leg;

    Leg.m_nStartDay = Start - m_RefDate;
    Leg.m_nEndDay = p->last().m_Date - m_RefDate;
    Leg.m_nType = xPaymentLeg;
    Leg.m_pPayment = p;

    add( Leg );
}


//
//   a d d P a y m e n t
//

void tInterestSpline::addPayment( tDate Start, double gPV, tDate End,
    double gRedemption, double gCoupon, const tCompounder& Compounder )

{
    tHeap<tPayment> Payment;

    tPayment::create( Start, End, gRedemption, gCoupon, Compounder, Payment );
    addPayment( Start, gPV, Payment );
}


//
//   a d d P a y m e n t
//

void tInterestSpline::addPayment( tDate Start, double gPV, double gRedemption,
    double gCoupon, const tCouponCompounder& Compounder )

{
    addPayment( Start, gPV, Compounder.maturity(), gRedemption,
        gCoupon, Compounder );
}


//
//   m e r g e
//

void tInterestSpline::merge()

{    
    finalize();

        // After finalization, adjacent legs with the
        // same forward yield are merged.
        //
        //         Reference date (date 0)
        //          |
        //    Leg 0 | Leg 1   Leg 2            Leg n
        //  +-------+=======+=======+---  ---+--------+
        //                 :       :        :        :
        //          22222222222222223333  mmmnnnnnnnnn  (m = n-1)
        //
        // In the picture, we assume that leg 1 and leg 2 have
        // the same forward yield. m_Index points to leg 2 for
        // both legs. Leg 2 is the "leader" of leg 1.

    m_Index.reset();

    if( m_Leg.numOfElems() == 0 )
        return;

        // the last leg is never merged:

    m_Leg.last().m_nLeader = m_Leg.numOfElems() - 1;

    for( int k = m_Leg.numOfElems() - 2; k >= 0; --k ) {
        tLeg& L1 = m_Leg[k];
        tLeg& L2 = m_Leg[k + 1];

        if( L1.m_gFwdYield == L2.m_gFwdYield ) {
                // don't anchor this leg in the index structure;
                // merge it to the next leader
            L1.m_nLeader = L2.m_nLeader;
        }
        else {
                // this leg is not merged; it's by itself,
                // or the last in a sequence
            L1.m_nLeader = k;
        }
    }

        // now construct the index

    for( MTG_FOR_INIT( int ) k = 1; k < m_Leg.numOfElems(); ++k ) {
        if( m_Leg[k].m_nLeader == k )
            m_Index.append( m_Leg[k].m_nEndDay - 1, k );
    }
}


//
//   p r e s e n t V a l u e
//

double tInterestSpline::presentValue( tDate Start, tDate End )

{
    return 1 / futureValue( Start, End );
}


//
//   f u t u r e V a l u e
//

double tInterestSpline::futureValue( tDate Start, tDate End )

{
    finalize();

    if( m_Leg.numOfElems() == 0 )
        return 1;

    if( ! m_Interest.hasCompoundingDates() )
        return accrue( Start, End, Start, End );

    double gValue = 1;

    tDate Date = Start;
    tDate PeriodStart, PeriodEnd;

    const tCompounder* pCompounder = m_Interest.compounder();

    pCompounder->getCompoundingPeriod( Date, PeriodStart, PeriodEnd );

        // jump from compounding period to compounding period

    while( true ) {
        tDate Mid = ( PeriodEnd < End ) ? PeriodEnd : End;

        gValue *= accrue( Date, Mid, PeriodStart, PeriodEnd );

        if( Mid >= End )
            break;

        pCompounder->getNextCompoundingPeriod( PeriodStart, PeriodEnd );
        Date = PeriodStart;
    }

    return gValue;
}


//
//   p r e s e n t V a l u e
//

double tInterestSpline::presentValue( tDate Start,
    const tHeap<tPayment>& Payment )

{
    double gPV = 0;

    for( int i = 0; i < Payment.numOfElems(); ++i ) {
        gPV += Payment[i].m_gAmount *
                presentValue( Start, Payment[i].m_Date );
    }
    return gPV;
}


//
//   p r e s e n t V a l u e
//

double tInterestSpline::presentValue( tDate Start, tDate End,
    double gRedemption, double gCoupon, const tCompounder& Compounder )

{
    tHeap<tPayment> Payment;

    tPayment::create( Start, End, gRedemption, gCoupon, Compounder, Payment );
    return presentValue( Start, Payment );
}


//
//   p r e s e n t V a l u e
//

double tInterestSpline::presentValue( tDate Start, double gRedemption,
    double gCoupon, const tCouponCompounder& Compounder )

{
    return presentValue( Start, Compounder.maturity(), gRedemption,
        gCoupon, Compounder );
}


//
//   y i e l d
//

double tInterestSpline::yield( tDate Start, tDate End )

{
    return yield( Start, End, m_Interest );
}


//
//   y i e l d
//

double tInterestSpline::yield( tDate Start, tDate End,
    const tInterest& Interest )

{
    MTG_ASSERT( m_Leg.numOfElems() > 0 );

    if( Start == End )
        ++End;

    double f = futureValue( Start, End );
    return Interest.invFutureValue( Start, End, f );
}


//
//   s w a p R a t e
//

double tInterestSpline::swapRate( tDate Start, tDate End, tBondMath& Math )

{
        // this code is identical to the code in MtgDrift.cpp

    double gRate = 5;
    tIdentityDayShift Id;

    if( Math.bondDirtyPrice( Start, End, 0, *this, Id ) > 100 )
        throw tException( OUT_OF_RANGE );

        // find initial bracket for bisection

    double gLeft = 0;
    double gPV = Math.bondDirtyPrice( Start, End, gRate, *this, Id );

    while( gPV < 100 ) {
        gLeft = gRate;
        gRate *= 2;
        gPV = Math.bondDirtyPrice( Start, End, gRate, *this, Id );
    }

    double gRight = gRate;
    double gLastError = 0;
    int nIter = 0;

    while( true ) {
            // next probe
        double gNewRate = ( gLeft + gRight ) / 2;
        double gNewPV = Math.bondDirtyPrice( Start, End, gNewRate, *this, Id );
        double gError = fabs( gNewPV / 100 - 1 );

        if( gError == 0 ) {
            gRate = gNewRate;
            break;
        }

        ++nIter;

        if( gLastError > 0 &&
            gError >= gLastError &&
            nIter > m_nMaxIter / 2 ) {
                // No improvement; be content with last guess.
                // (This escape only after a considerable number
                // of iterations.)
            break;
        }

        gRate = gNewRate;

        if( gError < 1e-10 ) {
                // Be content with 10 digits accuracy.
            gRate = gNewRate;
            break;
        }

        if( nIter >= m_nMaxIter )
            throw tException( NO_CONVERGENCE );

        gLastError = gError;

        if( gNewPV < 100 )
            gLeft = gNewRate;
        else
            gRight = gNewRate;
    }

    return gRate;
}


//
//   s w a p R a t e
//

double tInterestSpline::swapRate( tDate Start, tDate End, int nPeriods )

{
    tUSBondMath Math( nPeriods );
    return swapRate( Start, End, Math );
}


//
//   n u m O f L e g s
//

int tInterestSpline::numOfLegs()

{
    finalize();

    int n = 0;
    for( int i = 0; i < m_Leg.numOfElems(); ++i ) {
        if( m_Leg[i].m_nLeader == i )
            ++n;
    }

    return n;
}


//
//   g e t F i r s t L e g
//

void tInterestSpline::getFirstLeg( tDate& End, double& gYield )

{
    MTG_ASSERT( m_Leg.numOfElems() > 0 );

    finalize();

    int k = m_Leg[0].m_nLeader;

    End = m_RefDate + m_Leg[k].m_nEndDay;
    gYield = m_Leg[k].m_gFwdYield;
}


//
//   g e t N e x t L e g
//

bool tInterestSpline::getNextLeg( tDate& End, double& gYield )

{
    MTG_ASSERT( m_Leg.numOfElems() > 0 );

    finalize();

    long nEndDay = End - m_RefDate;

    if( nEndDay >= m_Index.numOfElems() )
        return false;
    if( nEndDay < 0 ) {
        getFirstLeg( End, gYield );
        return true;
    }

        // note that even when merged, this leads to
        // the correct leg:

    tLeg& L = m_Leg[m_Index[(int) nEndDay]];

    End = m_RefDate + L.m_nEndDay;
    gYield = L.m_gFwdYield;
    return true;
}


//
//   g e t F i r s t L e g
//

void tInterestSpline::getFirstLeg( tDate& End, double& gYield,
    const tInterest& Interest )

{
    MTG_ASSERT( m_Leg.numOfElems() > 0 );

    finalize();

    int k = m_Leg[0].m_nLeader;

    tDate Start = m_RefDate + m_Leg[k].m_nStartDay;
    End = m_RefDate + m_Leg[k].m_nEndDay;

    double f = futureValue( Start, End );
    gYield = Interest.invFutureValue( Start, End, f );
}


//
//   g e t N e x t L e g
//

bool tInterestSpline::getNextLeg( tDate& End, double& gYield,
    const tInterest& Interest )

{
    MTG_ASSERT( m_Leg.numOfElems() > 0 );

    finalize();

        // that's the last end date:

    long nEndDay = End - m_RefDate;

    if( nEndDay >= m_Index.numOfElems() )
        return false;
    if( nEndDay < 0 ) {
        getFirstLeg( End, gYield, Interest );
        return true;
    }
               
    int nLeg = m_Index[(int) nEndDay];
    MTG_ASSERT( nLeg > 0 );

        // here's the new value date:

    int nPrevLeg = nLeg - 1;
    while( m_Leg[nPrevLeg].m_nLeader == nLeg )
        --nPrevLeg;

    tDate Start = m_RefDate + m_Leg[nPrevLeg].m_nEndDay;
    End = m_RefDate + m_Leg[nLeg].m_nEndDay;

    double f = futureValue( Start, End );
    gYield = Interest.invFutureValue( Start, End, f );

    return true;
}

MTG_END_NAMESPACE


//#define _TEST
#if defined(_TEST)

#if defined(_WIN32)
    #include <conio.h>
#else
    #define getche getchar
#endif

MTG_USING_NAMESPACE

//
//   m a i n
//

void main( int argc, char *argv[] )

{
    printf( "\nTest tInterestSpline\n\n" );

    char sBuf[256];
    int cCmd;
    tDate D1, D2;
    double gYield;

    int nPeriods = 2;
    tDayCount DayCount = DayCountACT_365;
    tInterest::tType nType = tInterest::xExponential;
    tInterest::tQuote nQuote = tInterest::xYield;

    tInterest Interest( nType, nQuote, DayCount,
        tPeriodCompounder( nPeriods ) );
    tInterestSpline InterestSpline( Interest );

        // continuous compounding interest convention:

    tInterest CCInterest( tInterest::xExponential, tInterest::xYield, 
        DayCountACT_365_25, tPeriodCompounder( 0 ) );

    bool bGo = true;

    while( bGo ) { 
        printf( "<D>ayct <T>ype <Q>uote <C>omp <I>mp F<w>d "
                "Pa<y> <P>eriod <L>ist E<x>it: " );
        cCmd = getche();
        printf( "\n" );

        switch( cCmd ) {
            case 'd' :
            case 'D' :
                printf( "1=ACT/ACT     2=ACT/365   3=ACT/365NL\n"
                        "4=ACT/365ISDA 5=ACT/360   6=30/360ISDA\n"
                        "7=30/360PSA   8=30/360SIA 9=30E/360\n" 
                        "Choose: " );
                cCmd = getche();
                printf( "\n" );
                switch( cCmd ) {
                    case '1' : DayCount = DayCountACT_ACT;      break;
                    case '2' : DayCount = DayCountACT_365;      break;
                    case '3' : DayCount = DayCountACT_365_NL;   break;
                    case '4' : DayCount = DayCountACT_365_ISDA; break;
                    case '5' : DayCount = DayCountACT_360;      break;
                    case '6' : DayCount = DayCount30_360_ISDA;  break;
                    case '7' : DayCount = DayCount30_360_PSA;   break;
                    case '8' : DayCount = DayCount30_360_SIA;   break;
                    case '9' : DayCount = DayCount30E_360;      break;
                }
                Interest.set( DayCount );
                InterestSpline.set( Interest );
                break;

            case 't' :
            case 'T' :
                if( nType == tInterest::xExponential ) {
                    nType = tInterest::xLinear;
                    printf( "Type is now linear\n" );
                }
                else {
                    nType = tInterest::xExponential;
                    printf( "Type is now exponential\n" );
                }
                Interest.set( nType );
                InterestSpline.set( Interest );
                break;

            case 'q' :
            case 'Q' :
                if( nQuote == tInterest::xDiscount ) {
                    nQuote = tInterest::xYield;
                    printf( "Quote is now yield\n" );
                }
                else {
                    nQuote = tInterest::xDiscount;
                    printf( "Quote is now discount\n" );
                }
                Interest.set( nQuote );
                InterestSpline.set( Interest );
                break;

            case 'c' :
            case 'C' :
                printf( "Number of periods: " );
                gets( sBuf );
                sscanf( sBuf, "%d", &nPeriods );
                Interest.set( tPeriodCompounder( nPeriods ) );
                InterestSpline.set( Interest );
                break;

            case 'i' :
            case 'I' :
            case 'w' :
            case 'W' :
            case 'y' :
            case 'Y' :
                printf( "Start date: " );
                gets( sBuf );
                if( D1.set( sBuf ) == 0 ) {
                    printf( "Format error\n" );
                    break;
                }
                if( cCmd == 'y' || cCmd == 'Y' )
                    printf( "Maturity: " );
                else
                    printf( "End date: " );
                gets( sBuf );
                if( sBuf[0] == '+' ) {
                    int n;

                    sscanf( sBuf, "%d", &n );
                    D2 = D1 + n;
                }
                else {
                    if( D2.set( sBuf ) == 0 ) {
                        printf( "Format error\n" );
                        break;
                    }
                }

                if( cCmd == 'y' || cCmd == 'Y' ) {
                    double gCoupon, gPrice;
                    tCouponCompounder Compounder( D2, 2 );

                    printf( "Coupon: " );
                    gets( sBuf );
                    sscanf( sBuf, "%lg", &gCoupon );
                    printf( "Price: " );
                    gets( sBuf );
                    sscanf( sBuf, "%lg", &gPrice );
                    InterestSpline.addPayment( D1, gPrice,
                        D2, 100, gCoupon, Compounder );
                }
                else {
                    if( nQuote == tInterest::xDiscount )
                        printf( "Discount: " );
                    else
                        printf( "Yield: " );
                    gets( sBuf );
                    sscanf( sBuf, "%lg", &gYield );

                    if( cCmd == 'w' || cCmd == 'W' )
                        InterestSpline.addForward( D1, D2, gYield );
                    else
                        InterestSpline.addImplied( D1, D2, gYield );
                }
                break;

            case 'p' :
            case 'P' :
                printf( "Start date: " );
                gets( sBuf );
                if( D1.set( sBuf ) == 0 ) {
                    printf( "Format error\n" );
                    break;
                }
                printf( "End date: " );
                gets( sBuf );
                if( sBuf[0] == '+' ) {
                    int n;

                    sscanf( sBuf, "%d", &n );
                    D2 = D1 + n;
                }
                else {
                    if( D2.set( sBuf ) == 0 ) {
                        printf( "Format error\n" );
                        break;
                    }
                }
                printf( "Future value of $1 is %lg\n",
                    InterestSpline.futureValue( D1, D2 ) );
                printf( "Present value of $1 is %lg\n",
                    InterestSpline.presentValue( D1, D2 ) );
                printf( "Yield under current convention is %lg\n",
                    InterestSpline.yield( D1, D2 ) );
                printf( "Yield with continuous compounding is %lg\n",
                    InterestSpline.yield( D1, D2, CCInterest ) );
                break;

            case 'l' :
            case 'L' :
                printf( "Under current convention:\n" );
                InterestSpline.getFirstLeg( D1, gYield );
                printf( "  to %02d/%02d/%02d: %lg\n",
                    D1.month(), D1.day(), D1.year(), gYield );
                while( InterestSpline.getNextLeg( D1, gYield ) ) {
                    printf( "  to %02d/%02d/%02d: %lg\n",
                        D1.month(), D1.day(), D1.year(), gYield );
                }
                printf( "With continuous compounding:\n" );
                InterestSpline.getFirstLeg( D1, gYield, CCInterest );
                printf( "  to %02d/%02d/%02d: %lg\n",
                    D1.month(), D1.day(), D1.year(), gYield );
                while( InterestSpline.getNextLeg( D1, gYield, CCInterest ) ) {
                    printf( "  to %02d/%02d/%02d: %lg\n",
                        D1.month(), D1.day(), D1.year(), gYield );
                }
                break;

            case 'x' :
            case 'X' :
                bGo = false;
                break;
        }
    }

#if ! defined(_WIN32)
    printf( "\n" );
#endif
}

#endif