// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#if ! defined(_MTG_INTEREST_SPLINE_)
#define _MTG_INTEREST_SPLINE_

#include "MtgInterest.h"
#include "MtgMap.h"
#include "MtgPayment.h"

MTG_BEGIN_NAMESPACE

class tBondMath;


//
//   t I n t e r e s t S p l i n e
//

class tInterestSpline {

public:

    enum tType {
        xExtraCompounding,      // Default
        xAggregate
    };

private:

        // The interest spline creates a step function for the
        // forward rate from a list of given rates or payment
        // structures. This list consists of pieces of one of
        // the three following types:

    enum tLegType {
        xForwardLeg,            // a forward rate itself
        xImpliedLeg,            // an implied rate
        xPaymentLeg             // a payment structure like
    };                          // a coupon bearing bond        

    struct tLeg {
        long m_nStartDay;       // counting from m_RefDate
        long m_nEndDay;         // counting from m_RefDate
        tLegType m_nType;
        double m_gFwdYield;     // interpreted under m_Interest
        double m_gImpYield;     // interpreted under m_Interest
        tHeap<tPayment>* m_pPayment;
        int m_nLeader;          // points to the leg to which
    };                          // this leg is merged 

    tType m_nType;

    tInterest m_Interest;       // interest convention used
    tDate m_RefDate;

    tHeap<tLeg> m_Leg;
    tMap<int> m_Index;          // index day -> m_Leg

    bool m_bFinalized;          // is m_Index up-to-date?

    static const int m_nMaxIter;
                                // for Newton-Raphson

    void init( tType nType );
    void cleanup();

    void copyFrom( const tInterestSpline& Spline );

    void add( const tLeg& Leg );

        // Incremental finalization under xExtraCompounding:

    double reversePayment( const tHeap<tPayment>& Payment,
        double gPV, tDate Start, int nPos ) const;

        // Incremental finalization under xAggregate:

    double discount( const tHeap<tPayment>& Payment );
    void refineLeg( const tHeap<tPayment>& Payment, int nLeg );

    void finalize();

    double accrueExtraCompounding( tDate Start, tDate End,
        tDate PeriodStart, tDate PeriodEnd ) const;
    double accrueAggregate( tDate Start, tDate End,
        tDate PeriodStart, tDate PeriodEnd ) const;
    double accrue( tDate Start, tDate End,
        tDate PeriodStart, tDate PeriodEnd ) const;

public:

    tInterestSpline();
    tInterestSpline( tType nType );
    tInterestSpline( const tInterest& Interest );
    tInterestSpline( tType nType, const tInterest& Interest );
    tInterestSpline( const tInterestSpline& Spline );

    ~tInterestSpline();

    tInterestSpline& operator=( const tInterestSpline& Spline );

        // The interest convention registered with the object is
        // used a) to interprete the yield information in forward
        // and implied legs, and b) to compute the output forward
        // rate. For the latter purpose, a temporary interest
        // convention can also be passed to the yield(), getFirstLeg()
        // or getNextLeg() functions explicitely (this will most likely
        // happen if the forward curve is needed in continuously
        // compounded form, because forward and implied yields are
        // usually not continuously compounded and therefore require
        // an interest convention for discrete compounding.)

    void set( tType nType );
    void set( const tInterest& Interest );

    const tInterest& interest() const {
        return m_Interest;
    }

        // Register yield data directly:

    void addImplied( tDate Start, tDate End, double gYield );
    void addForward( tDate Start, tDate End, double gYield );

        // Register payment streams, together with their present
        // value:

    void addPayment( tDate Start, double gPV, tDate End, double gFV );
    void addPayment( tDate Start, double gPV, tPayment Payment );

        // This is the most general function and fits all kinds
        // of payment strucures:

    void addPayment( tDate Start, double gPV,
        const tHeap<tPayment>& Payment );
        
        // For vanilla bonds, things are made a bit easier
        // with the following functions:

    void addPayment( tDate Start, double gPV, tDate End,
        double gRedemption, double gCoupon, const tCompounder& Compounder );
    void addPayment( tDate Start, double gPV, double gRedemption,
        double gCoupon, const tCouponCompounder& Compounder );

        // Merging means to consolidate adjacent legs with the
        // same forward rate. This has an effect if the type
        // is xExtraCompounding.

    void merge();

        // The interest spline represents a step function of forward
        // rates. This step function can, of course, be used to
        // discount and accrue interest. ALL FUNCTIONS BELOW THIS
        // LINE ARE SAFE in the sense that no exception is thrown
        // if the spline is ALREADY finalized when the service 
        // function is called.

        // Use the interest spline to compute the future or
        // present value of $1:

    double presentValue( tDate Start, tDate End );
    double futureValue( tDate Start, tDate End );

        // To compute the discounted value of a payment structure
        // is also possible:

    double presentValue( tDate Start, const tHeap<tPayment>& Payment );
        
        // For vanilla bonds, things are made a bit easier
        // with the following functions:

    double presentValue( tDate Start, tDate End,
        double gRedemption, double gCoupon, const tCompounder& Compounder );
    double presentValue( tDate Start, double gRedemption,
        double gCoupon, const tCouponCompounder& Compounder );

        // Use the interest spline to compute the yield implied
        // by all the pieces under the registered interest convention,
        // or under a custom interest convention:

    double yield( tDate Start, tDate End );
    double yield( tDate Start, tDate End, const tInterest& Interest );

        // The swap rate is the coupon value of a hypothetical
        // bond valued at par.

    double swapRate( tDate Start, tDate End, tBondMath& Math );
    double swapRate( tDate Start, tDate End, int nPeriods );

        // Query the entire structure by iterating through the
        // list of forward yields, again, under the registered
        // interest convention, or under a custom interest
        // convention:

    int numOfLegs();

    void getFirstLeg( tDate& End, double& gYield );
    bool getNextLeg( tDate& End, double& gYield );

    void getFirstLeg( tDate& End, double& gYield,
        const tInterest& Interest );
    bool getNextLeg( tDate& End, double& gYield,
        const tInterest& Interest );
};

MTG_END_NAMESPACE

#endif
