// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#include "MtgIncl.h"
#include "MtgJobPipe.h"
#include "MtgJobPool.h"
#include "MtgLog.h"

#if defined(_WIN32)
    #include <process.h>
#endif

MTG_BEGIN_NAMESPACE

//
//   c o n s t a n t s
//

const char* tJobPipe::m_sStdName = "\\\\.\\pipe\\Mtg";


//
//   g e t L i n e
//

char* tJobPipe::tRequest::getLine( const char*& sRequest ) const

{
    int k = 0;
    char* s = 0;

    while( sRequest[k] && sRequest[k] != '\r' && sRequest[k] != '\n' )
        ++k;

    if( k > 0 ) {
        s = new char[k + 1];
        memcpy( s, sRequest, k );
        s[k] = 0;
    }

    if( sRequest[k] == '\r' )
        ++k;
    if( sRequest[k] == '\n' )
        ++k;

    sRequest += k;
    return s;
}


//
//   t R e q u e s t
//

tJobPipe::tRequest::tRequest( const char* sRequest )

{
    m_sGroup = getLine( sRequest );
    m_sDir = getLine( sRequest );
    m_sProgram = getLine( sRequest );
    m_sCmdLine = getLine( sRequest );
}


//
//   ~ t R e q u e s t
//

tJobPipe::tRequest::~tRequest()

{
    if( m_sGroup != 0 )
        delete m_sGroup;
    if( m_sDir != 0 )
        delete m_sDir;
    if( m_sProgram != 0 )
        delete m_sProgram;
    if( m_sCmdLine != 0 )
        delete m_sCmdLine;
}


//
//   l o c k L o g
//

bool tJobPipe::lockLog()

{
        // we need to put log access into a critical section 
        // because at shutdown, the log object might die
        // faster than the pipes
        
#if defined(_WIN32)
    EnterCriticalSection( &m_LogLock );
    if( m_pLog != 0 )
        return true;
    unlockLog();
    return false;
#else
    return true;
#endif
}


//
//   u n l o c k L o g
//

void tJobPipe::unlockLog()

{
#if defined(_WIN32)
    LeaveCriticalSection( &m_LogLock );
#endif
}


//
//   p r o c e s s
//

tRetCode tJobPipe::process( const char* sRequest )

{
    MTG_ASSERT( m_pJobPool != 0 );

    tRetCode nRet;
    tRequest Req( sRequest );

    nRet = m_pJobPool->createJob(
        Req.m_sGroup, Req.m_sDir, Req.m_sProgram, Req.m_sCmdLine, m_pLog );

    if( lockLog() ) {
        if( nRet == OK ) {
            m_pLog->putNormal( *this, "Started %s in directory %s",
                Req.m_sProgram ? Req.m_sProgram : "?", 
                Req.m_sDir ? Req.m_sDir : "?" );
        }
        else {
            m_pLog->putNormal( *this, "Program %s in directory %s: error %d",
                Req.m_sProgram ? Req.m_sProgram : "?", 
                Req.m_sDir ? Req.m_sDir : "?", (int) nRet );
        }
        unlockLog();
    }

    return nRet;
}


//
//   r u n
//

void tJobPipe::run()

{
    MTG_ASSERT( m_bAlive );

#if defined(_WIN32)

    unsigned long nBytesRead, nBytesWritten;
    char sClient[512];

    while( m_bAlive ) {
        if( ConnectNamedPipe( m_hPipe, NULL ) ||
            GetLastError() == ERROR_PIPE_CONNECTED ) {

            GetNamedPipeHandleState( m_hPipe, NULL, NULL, NULL, NULL,
                sClient, sizeof(sClient) );

            if( lockLog() ) {
                m_pLog->putNormal( *this, "Connection opened to %s",
                    sClient );
                unlockLog();
            }

            if( ReadFile( m_hPipe, m_sRequest, sizeof(m_sRequest),
                    &nBytesRead, NULL ) && nBytesRead > 0 ) {

                m_sRequest[nBytesRead] = 0;

                tRetCode nRet = process( m_sRequest );
                sprintf( m_sResponse, "%d", (int) nRet );

                unsigned long n = strlen( m_sResponse );

                if( WriteFile( m_hPipe, m_sResponse, n,
                        &nBytesWritten, NULL ) || nBytesWritten != n ) {
                    FlushFileBuffers( m_hPipe ); 
                }
                else {
                    if( lockLog() ) {
                        m_pLog->putError(
                            *this, "Error %d during transaction",
                            WRITE_ERROR );
                        unlockLog();
                    }
                }
            }
            else {
                if( lockLog() ) {
                    m_pLog->putError( *this,
                        "Error %d during transaction",
                        READ_ERROR );
                    unlockLog();
                }
            }

            DisconnectNamedPipe( m_hPipe );
        }
    }

#endif

    m_pJobPool->downRef();
    delete this;        // commit suicide

#if defined(_WIN32)
    _endthread();
#endif
}


//
//   r u n
//

void tJobPipe::run( void* p )

{
    MTG_ASSERT( p != 0 );
    static_cast<tJobPipe*>( p )->run();
}


//
//   ~ t J o b P i p e
//

tJobPipe::~tJobPipe()

{
    MTG_ASSERT( ! m_bAlive && m_pLog == 0 );
    if( m_sName != 0 )
        delete m_sName;
}


//
//   t J o b P i p e
//

tJobPipe::tJobPipe( tJobPool* pJobPool )

{
    MTG_ASSERT( pJobPool != 0 );

    m_sName = StrCopy( m_sStdName );
    m_bAlive = false;
    m_pJobPool = pJobPool;
    m_pLog = 0;

#if defined(_WIN32)
    InitializeCriticalSection( &m_LogLock );
#endif
}


//
//   s e t N a m e
//

void tJobPipe::setName( const char* sName )

{
    if( m_sName != 0 )
        delete m_sName;
    m_sName = StrCopy( sName );
}


//
//   s e t L o g
//

void tJobPipe::setLog( tLog* pLog )

{
    m_pLog = pLog;
}


//
//   s t a r t
//

tRetCode tJobPipe::start()

{
    MTG_ASSERT( m_pJobPool != 0 && m_sName != 0 );

#if defined(_WIN32)

    m_hPipe = CreateNamedPipe(
          m_sName,
          PIPE_ACCESS_DUPLEX,
          PIPE_TYPE_MESSAGE |
          PIPE_READMODE_MESSAGE |
          PIPE_WAIT,
          PIPE_UNLIMITED_INSTANCES,
          sizeof(m_sResponse),          // output buffer size 
          sizeof(m_sRequest),           // input buffer size 
          3000,                         // client time-out 
          NULL );

    if( m_hPipe == INVALID_HANDLE_VALUE )
        return PIPE_ERROR;

        // now start the thread
        
    m_bAlive = true;
    m_pJobPool->upRef();

    if( ( m_hThread = _beginthread(
            run, 64 * 1024, this ) ) == (unsigned long) -1 ) {
        m_bAlive = false;
        m_pJobPool->downRef();
        return FORK_ERROR;
    }

        // the thread will clean up

    return OK;

#else
    return NOT_IMPLEMENTED;
#endif
}


//
//   d i e
//

void tJobPipe::die()

{
    if( lockLog() ) {
        m_pLog = 0;
        unlockLog();
    }

    if( m_bAlive )
        m_bAlive = false;   // run() must handle this
    else
        delete this;
}


//
//   s u b m i t
//

tRetCode tJobPipe::submit( const char* sGroup, const char* sDir,
    const char* sProgram, const char* sCmdLine )

{
    MTG_ASSERT( sProgram != 0 || sCmdLine != 0 );

#if defined(_WIN32)

    int l = 5;
    tRetCode nRet;

    if( sGroup != 0 )
        l += strlen( sGroup );
    if( sDir != 0 )
        l += strlen( sDir );
    if( sProgram != 0 )
        l += strlen( sProgram );
    if( sCmdLine != 0 )
        l += strlen( sCmdLine );

    char* sRequest = new char[l];

    sRequest[0] = 0;

    if( sGroup != 0 )
        strcat( sRequest, sGroup );
    strcat( sRequest, "\n" );

    if( sDir != 0 )
        strcat( sRequest, sDir );
    strcat( sRequest, "\n" );

    if( sProgram != 0 )
        strcat( sRequest, sProgram );
    strcat( sRequest, "\n" );

    if( sCmdLine != 0 )
        strcat( sRequest, sCmdLine );
    strcat( sRequest, "\n" );

    char sResponse[128];
    unsigned long nBytesRead;

    if( ! CallNamedPipe( m_sStdName, sRequest, strlen( sRequest ),
            sResponse, sizeof(sResponse), &nBytesRead,
            NMPWAIT_USE_DEFAULT_WAIT ) || nBytesRead == 0 ) {
        DWORD nError = GetLastError();
        return PIPE_ERROR;
    }

    delete sRequest;
    sResponse[nBytesRead] = 0;
    sscanf( sResponse, "%d", (int) &nRet );
    return nRet;

#else
    return NOT_IMPLEMENTED;
#endif
}


//
//   s u b m i t
//

tRetCode tJobPipe::submit( const char* sCmdLine )

{
#if defined(_WIN32)

    tRetCode nRet;
    char sProgram[512];

    if( GetModuleFileName( NULL, sProgram, sizeof(sProgram) ) == 0 )
        return FORK_ERROR;

    char* sDir = GetDir();

    nRet = submit( sDir, sDir, sProgram, sCmdLine );
    delete sDir;

    return nRet;

#else
    return NOT_IMPLEMENTED;
#endif

}

MTG_END_NAMESPACE
